package io.aether.utils;

import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.ASupplier;
import io.leangen.geantyref.GenericTypeReflector;
import org.jetbrains.annotations.NotNull;
import sun.reflect.ReflectionFactory;

import java.lang.annotation.Annotation;
import java.lang.annotation.Inherited;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.lang.reflect.*;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

public abstract class CType<T> implements CTypeI<T> {
    static final Map<Type, CTypeI<Object>> CACHE1;
    private static final CTypeI<Object>[] permittedSubclassesEmpty;
    private final static MethodHandles.Lookup lookup;
    private static final CTypeI<Object>[] EMPTY_AR;
    private static final Annotation[] EMPTY_AR_A;
    int hashCode;
    String declareCode;
    private String rawSimpleNameWithParameters;

    CType(Type ignoredType) {

    }

    @Override
    public void toString(AString sb) {
        getRawSimpleNameWithParameters(sb);
    }

    @Override
    public boolean mutable() {
        return false;
    }

    @Override
    public <FT> FieldAccessor<FT> getFieldAccessor(String field) {
        return getFieldAccessor(getAllFields().getFirstOrNull(f -> f.getName().equals(field)));
    }

    @Override
    public <FT> FieldAccessor<FT> getFieldAccessor(Field field) {
        assert field != null;
        field.setAccessible(true);
        VarHandle vh;
        try {
            if (!Mods.isPublic(field)) {
                vh = MethodHandles.privateLookupIn(field.getDeclaringClass(), lookup).unreflectVarHandle(field);
            } else {
                vh = lookup.unreflectVarHandle(field);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        if (Mods.isFinal(field)) {
            return new FieldAccessor<>() {
                @Override
                public FT get(Object owner) {
                    try {
                        return RU.cast(vh.getVolatile(owner));
                    } catch (Throwable e) {
                        return RU.error(e);
                    }
                }

                @Override
                public void toString(AString sb) {
                    CType.this.getRawSimpleNameWithParameters(sb);
                    sb.add('.').add(field.getName());
                }

                @Override
                public String toString() {
                    return CType.this.getRawSimpleNameWithParameters() + "." + field.getName();
                }

                @Override
                public void set(Object owner, FT value) {
                    try {
                        field.set(owner, value);
                    } catch (IllegalAccessException e) {
                        RU.error(e);
                    }
                }
            };
        } else {
            return new FieldAccessor<>() {
                @Override
                public String toString() {
                    return CType.this.getRawSimpleNameWithParameters() + "." + field.getName();
                }

                @Override
                public void toString(AString sb) {
                    CType.this.getRawSimpleNameWithParameters(sb);
                    sb.add('.').add(field.getName());
                }

                @Override
                public FT get(Object owner) {
                    try {
                        return RU.cast(vh.getVolatile(owner));
                    } catch (Throwable e) {
                        return RU.error(e);
                    }
                }

                @Override
                public void set(Object owner, FT value) {
                    vh.setVolatile(owner, value);
                }
            };
        }
    }

    @Override
    public CTypeI<T> addAnnotations(Annotation... aa) {
        return CTypeI.of(aa, this);
    }

    @Override
    public String getRawSimpleNameWithParameters() {
        var sb = AString.of();
        getRawSimpleNameWithParameters(sb);
        return sb.toString();
    }

    @Override
    public void getRawSimpleNameWithParameters(AString sb2) {
        if (rawSimpleNameWithParameters == null) {
            var sb = AString.of();
            sb.add(getRawSimpleName());
            if (isParameterizedType()) {
                sb.add("<");
                boolean f2 = true;
                for (var tt : getParameters()) {
                    if (tt == null) {
                        continue;
                    }
                    if (f2) {
                        f2 = false;
                    } else {
                        sb.add(", ");
                    }
                    tt.getRawSimpleNameWithParameters(sb);
                }
                sb.add(">");
            }
            rawSimpleNameWithParameters = sb.toString();
        }
        sb2.add(rawSimpleNameWithParameters);
    }

    @Override
    public CharSequence getCanonicalNameWithParameters() {
        AString sb = AString.of();
        getCanonicalNameWithParameters(sb);
        return sb;
    }

    @Override
    public void getCanonicalNameWithParameters(AString sb) {
        sb.add(getCanonicalName());
        if (isParameterizedType()) {
            sb.add("<");
            boolean f2 = true;
            for (var tt : getParameters()) {
                if (tt == null) {
                    continue;
                }
                if (f2) {
                    f2 = false;
                } else {
                    sb.add(", ");
                }
                tt.getRawSimpleNameWithParameters(sb);
            }
            sb.add(">");
        }
    }

    @Override
    public String getRawSimpleName() {
        return getRaw2().getSimpleName();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        return toString().equals(o.toString());
    }

    @Override
    public int hashCode() {
        if (hashCode == 0) {
            hashCode = toString().hashCode();
        }
        return hashCode;
    }

    @Override
    public String toString() {
        return declareCode();
    }

    @Override
    public String declareCode() {
        if (declareCode == null) {
            var sb = AString.of();
            var raw = getRaw2();
            var aad = Flow.flow(raw.getDeclaredAnnotations()).toSet();
            for (var a : Flow.flow(getAnnotations()).filterNot(aad::contains)) {
                if (raw.isAnnotationPresent(a.annotationType())) continue;
                sb.add("@");
                sb.add(a.annotationType().getSimpleName());
                var mm = a.annotationType().getDeclaredMethods();
                if (mm.length > 0) {
                    sb.add("(");
                    boolean first = true;
                    for (var m : mm) {
                        if (first) {
                            first = false;
                        } else {
                            sb.add(", ");
                        }
                        try {
                            var v = m.invoke(a);
                            if (!m.getName().equals("value")) {
                                sb.add(m.getName());
                                sb.add("=");
                            }
                            if (v instanceof Type) {
                                sb.add(CTypeI.of((Type) v));
                            } else if (v.getClass().isArray()) {
                                if (v instanceof Class[]) {
                                    Flow.flow((Class<?>[]) v).map(Class::getCanonicalName).join(sb, ",");
                                } else if (v instanceof int[]) {
                                    Flow.flow((int[]) v).join(sb,",");
                                }else {
                                    RU.errorUnsupported();
                                }
                            } else {
                                sb.add(v);
                            }
                        } catch (Exception e) {
                            throw new RuntimeException(e);
                        }
                    }
                    sb.add(")");
                }
                sb.add(" ");
            }
            sb.add(getCanonicalName());
            var pp = getParameters();
            if (pp != null && pp.length > 0) {
                sb.add("<");
                boolean first = true;
                for (var p : pp) {
                    if (first) {
                        first = false;
                    } else {
                        sb.add(", ");
                    }
                    sb.add(p.declareCode());
                }
                sb.add(">");
            }
            declareCode = sb.toString();
        }
        return declareCode;
    }

    @Override
    public String toId() {
        return getRaw2().getSimpleName() + hashCode();
    }

    @Override
    public Class<?> getRaw2() {
        return getRaw();
    }

    @Override
    public boolean instanceOf(CTypeI<?> t) {
        return instanceOf(t.toType());
    }

    @Override
    public boolean instanceOf(AnnotatedType t) {
        return instanceOf(t.getType());
    }

    @Override
    public @NotNull AnnotatedType toAnnotatedType() {
        return GenericTypeReflector.annotate(toType());
    }

    @Override
    public CTypeI<Object> getComponent() {
        return getComponent(0);
    }

    @Override
    public CTypeI<Object> getComponent(int i) {
        return null;
    }

    @Override
    public CTypeI<Object> getComponent(Class<?> base, int i) {
        var p = base.getTypeParameters()[i];
        return CTypeI.of(GenericTypeReflector.resolveType(p, toType()));
    }

    @Override
    public boolean isAnnotated(Class<? extends Annotation> aClass) {
        return getAnnotation(aClass) != null;
    }

    @Override
    public <A extends Annotation> A getAnnotation(Class<A> aClass) {
        for (var a : getAnnotations()) {
            if (a.annotationType() == aClass) {
                return aClass.cast(a);
            }
        }
        return null;
    }

    @Override
    public boolean isArray() {
        var t = toType();
        return (t instanceof Class<?> && ((Class<?>) t).isArray()) || t instanceof GenericArrayType;
    }

    @Override
    public boolean isNumber() {
        var r = getRaw2();
        return r == byte.class || r == short.class || r == int.class || r == long.class || r == Byte.class || r == Short.class || r == Integer.class || r == Long.class || r == float.class || r == Float.class || r == double.class || r == Double.class;
    }

    @Override
    public boolean isEnum() {
        return getRaw2().isEnum();
    }

    @Override
    public boolean isInterface() {
        return getRaw().isInterface();
    }

    @Override
    public TypeVariable<?>[] getGenericVariables() {
        return getRaw2().getTypeParameters();
    }

    @Override
    public Flow<Method> getMethods() {
        return Flow.flow(getRaw2().getDeclaredMethods());
    }

    @Override
    public Flow<Field> getAllFields() {
        List<Field> res = new ArrayList<>();
        CTypeI.getAllFields(getRaw2(), res);
        return Flow.flow(res);
    }

    @Override
    public Flow<Method> getAllMethods() {
        Set<Method> res = new HashSet<>();
        CTypeI.getAllMethods(getRaw2(), res);
        return Flow.flow(res);
    }

    @Override
    public Flow<Field> getFields() {
        return Flow.flow(getRaw2().getDeclaredFields());
    }

    @Override
    public Annotation[] getAnnotations() {
        return EMPTY_AR_A;
    }

    @Override
    public boolean isBoxedOrRawPrimitive() {
        return isBoxedPrimitive() || isPrimitive();
    }

    @Override
    public CTypeI<? extends T>[] getChildrenClasses() {
        return RU.cast(EMPTY_AR);
    }

    @Override
    public void getAllDependencies(Set<CTypeI<?>> res, boolean withTransient) {
    }

    @Override
    public Flow<CTypeI<? extends T>> getAllChildrenClasses() {
        return Flow.flow();
    }

    @Override
    public boolean isAbstract() {
        return Mods.isAbstract(getRaw2());
    }

    @Override
    public Flow<CTypeI<?>> getInterfaces() {
        var r = getRaw2();
        return Flow.flow(r.getAnnotatedInterfaces()).map(CTypeI::of);
    }

    @SuppressWarnings("unchecked")
    @Override
    public T makeAuto(Object... args) {
        var raw = getRaw2();
        Constructor<?> cRes = null;
        var cc = raw.getDeclaredConstructors();
        if (cc.length == 1) {
            var c = cc[0];
            if (c.getParameterCount() == args.length) {
                cRes = c;
            }
        } else {
            loop:
            for (var c : cc) {
                if (c.getParameterCount() != args.length) continue;
                if (args.length == 0) {
                    c.setAccessible(true);
                    cRes = c;
                    break;
                }
                for (int i = 0; i < c.getParameterTypes().length; i++) {
                    var t = c.getParameterTypes()[i];
                    if (!t.isAssignableFrom(args[i].getClass())) {
                        continue loop;
                    }
                }
                c.setAccessible(true);
                cRes = c;
                break;
            }
        }
        if (cRes == null) {
            throw new UnsupportedOperationException();
        }
        try {
            return (T) cRes.newInstance(args);
        } catch (Exception e) {
            return RU.error(e);
        }
    }

    @Override
    public VarHandle getFieldVarHandle(Field field) {
        try {
            field.setAccessible(true);
            var lookup2 = MethodHandles.privateLookupIn(getRaw2(), lookup);
            return lookup2.unreflectVarHandle(field);
        } catch (IllegalAccessException e) {
            return RU.error(e);
        }
    }

    @Override
    public VarHandle getFieldVarHandle(String name) {
        try {
            var raw = getRaw2();
            var f = raw.getDeclaredField(name);
            f.setAccessible(true);
            return getFieldVarHandle(f);
        } catch (NoSuchFieldException e) {
            return RU.error(e);
        }
    }

    @Override
    public ASupplier<?> maker() {
        throw new UnsupportedOperationException();
    }

    @Override
    public Constructor<?> getConstructor(Class<?>... args) {
        try {
            return getRaw2().getConstructor(args);
        } catch (Exception e) {
            return null;
        }
    }

    @Override
    public MethodHandle maker(Class<?>... args) {
        Constructor<?> c = getConstructor(args);
        assert c != null;
        try {
            return MethodHandles.lookup().unreflectConstructor(c);
        } catch (IllegalAccessException e) {
            return RU.error(e);
        }
    }

    @Override
    public Enum<?>[] getEnumValues() {
        return RU.cast(getRaw2().getEnumConstants());
    }

    @Override
    public boolean isAetherClass() {
        return getCanonicalName().contains("aether");
    }

    @Override
    public String getCanonicalName() {
        return getRaw2().getCanonicalName();
    }

    @Override
    public boolean isParametrizedClass() {
        return getRaw2().getTypeParameters().length > 0;
    }

    @Override
    public boolean isVariable() {
        return toType() instanceof TypeVariable<?>;
    }

    @Override
    public boolean isParameterizedType() {
        return toType() instanceof ParameterizedType;
    }

    @Override
    public String getPackage() {
        return getRaw2().getPackageName();
    }

    @Override
    public boolean isInstance(Object o) {
        return GenericTypeReflector.isSuperType(toType(), o.getClass());
    }


    @Override
    public boolean compareValues(T val1, T val2, CompareEvent consumer) {
        return Objects.equals(val1, val2);
    }

    @Override
    public CTypeI<T[]> toArray() {
        return CTypeI.buildArray(this);
    }

    @Override
    public <C> CTypeI<C> cast() {
        return RU.cast(this);
    }

    @Override
    public ClassLoader getClassLoader() {
        return getRaw2().getClassLoader();
    }

    @Override
    public T cast(Object o) {
        return RU.cast(o);
    }

    static {
        CACHE1 = new ConcurrentHashMap<>();
        permittedSubclassesEmpty = RU.cast(new CTypeI<?>[0]);
        lookup = MethodHandles.lookup();
        EMPTY_AR = RU.cast(new CTypeI<?>[0]);
        EMPTY_AR_A = new Annotation[0];
    }

    public static final class WType<T> extends CType<T> {
        private final TypeVariable<?> tt;

        public WType(TypeVariable<?> type) {
            super(type);
            this.tt = type;
        }

//        @Override
//        public String getRawSimpleName() {
//            return toString();
//        }
//
//        @Override
//        public String getRawSimpleNameWithParameters() {
//            return toString();
//        }
//
//        @Override
//        public String getCanonicalName() {
//            return toString();
//        }

        @Override
        public CTypeI<Object> getComponentByName(String name) {
            return null;
        }

        @Override
        public CTypeI<T> addAnnotations(Annotation... aa) {
            throw new UnsupportedOperationException();
        }

        @Override
        public Class<T> getRaw() {
            return null;
        }

        @Override
        public boolean instanceOf(Type t) {
            throw new UnsupportedOperationException();
        }

        @Override
        public @NotNull Type toType() {
            return tt;
        }

        @Override
        public boolean isPrimitive() {
            return false;
        }

        @Override
        public CTypeI<Object> resolveReturnType(Method m) {
            throw new UnsupportedOperationException();
        }

        @Override
        public CTypeI<Object>[] resolveTypeArgs(Method m) {
            throw new UnsupportedOperationException();
        }

        @Override
        public CTypeI<Object> resolveExtend(CTypeI<Object> inf) {
            throw new UnsupportedOperationException();
        }

        @Override
        public CTypeI<Object> resolveTypeField(Field f) {
            throw new UnsupportedOperationException();
        }

        @Override
        public boolean isBoxedPrimitive() {
            throw new UnsupportedOperationException();
        }

        @Override
        public CTypeI<T> toPrimitive() {
            throw new UnsupportedOperationException();
        }

        @Override
        public CTypeI<T> unbox() {
            throw new UnsupportedOperationException();
        }

        @Override
        public CTypeI<Object>[] getParameters() {
            throw new UnsupportedOperationException();
        }

        @Override
        public CType<Object> getParent() {
            throw new UnsupportedOperationException();
        }

        @Override
        public CTypeI<T> tryToBox() {
            throw new UnsupportedOperationException();
        }

        public String declareCode() {
            return tt.getName();
        }
    }

    private static abstract class BType<T> extends CType<T> {
        private final CTypeI<? extends T>[] permittedSubclasses;
        private int mutable;
        private CTypeI<? extends T>[] allPermittedSubclasses;
        private ASupplier<?> makerMh;

        public BType(Type type) {
            super(type);
            permittedSubclasses = RU.cast(permittedSubclassesEmpty);
        }

        @Override
        public void getAllDependencies(Set<CTypeI<?>> res, boolean withTransient) {//TODO methods
            if (isBoxedPrimitive()) {
                res.add(unbox());
            } else {
                if (res.add(this)) {
                    for (var f : getAllFields()) {
                        if (!withTransient && Mods.isTransient(f)) continue;
                        resolveTypeField(f).getAllDependencies(res, withTransient);
                    }
                }
            }
        }

        @Override
        public boolean compareValues(T val1, T val2, CompareEvent consumer) {
            boolean res = getParent() == null || getParent().compareValues(val1, val2, consumer);
            try {
                for (var f : getFields()) {
                    if (Mods.isFinal(f)) continue;
                    var ft = CTypeI.<T>of(f.getType());
                    T fv1 = RU.cast(f.get(val1));
                    T fv2 = RU.cast(f.get(val2));
                    if (fv1 == fv2) continue;
                    if (ft.isAetherClass()) {
                        res &= ft.compareValues(fv1, fv2, consumer);
                    } else {
                        if (!Objects.equals(fv1, fv2)) {
                            consumer.compareEvent(f, val1, val2, fv1, fv2);
                        }
                    }
                }
                return res;
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public boolean mutable() {
            switch (mutable) {
                case 0:
                    if (getParent() != null && getParent().mutable()) {
                        mutable = 1;
                        return true;
                    } else {
                        for (var f : getFields()) {
                            if (!Mods.isFinal(f) || resolveTypeField(f).mutable()) {
                                mutable = 1;
                                return true;
                            }
                        }
                        mutable = -1;
                        return false;
                    }
                case 1:
                    return true;
                case -1:
                    return false;
                default:
                    throw new UnsupportedOperationException();
            }
        }

        public CTypeI<? extends T>[] getChildrenClasses() {
            return permittedSubclasses;
        }

        @Override
        public ASupplier<?> maker() {
            if (makerMh == null) {
                if (isInterface()) {
                    makerMh = () -> {
                        throw new IllegalStateException();
                    };
                    return makerMh;
                }
                var raw = getRaw2();
                assert raw != null;
                Constructor<?> c;
                Class<?> parent = raw;
                while (parent != null && CTypeI.getDefaultConstructor(parent) == null) {
                    parent = parent.getSuperclass();
                }
                if (parent == raw) {
                    c = CTypeI.getDefaultConstructor(raw);
                } else {
                    assert parent != null : this;
                    c = ReflectionFactory.getReflectionFactory().newConstructorForSerialization(raw, CTypeI.getDefaultConstructor(parent));
                }
                assert c != null;
                c.setAccessible(true);
                makerMh = c::newInstance;
            }
            return makerMh;
        }

        @Override
        public Flow<CTypeI<? extends T>> getAllChildrenClasses() {
            if (allPermittedSubclasses != null) return Flow.flow(allPermittedSubclasses);
            if (permittedSubclasses == null || permittedSubclasses.length == 0) return Flow.flow();
            Flow<CTypeI<? extends T>> res = Flow.flow(permittedSubclasses);
            for (var c : permittedSubclasses) {
                res = res.addAll(c.getAllChildrenClasses());
            }
            res = res.sort((a, b) -> a.toId().compareTo(b.toId()));
            allPermittedSubclasses = res.toArray(CTypeI.class);
            return Flow.flow(allPermittedSubclasses);
        }
    }

    static final class AType<T> extends BType<T> {
        private final PType<T> type;
        private final AnnotatedType annotatedType;
        CTypeI<Object>[] parameters;
        private Annotation[] annotations;

        AType(AnnotatedType type) {
            super(type.getType());
            this.annotatedType = type;
            this.type = new PType<T>(type.getType());
        }

        @Override
        public CTypeI<?>[] resolveTypeArgs(Method m) {
            var aa = GenericTypeReflector.getParameterTypes(m, annotatedType);
            CTypeI<?>[] res = new CTypeI<?>[aa.length];
            for (int i = 0; i < res.length; i++) {
                res[i] = CTypeI.of(aa[i]);
            }
            return res;
        }

        @Override
        public CTypeI<?> resolveExtend(CTypeI<Object> inf) {
            return RU.resolveExtend(inf, RU.cast(this));
        }

        @Override
        public CTypeI<?> resolveTypeField(Field f) {
            return CTypeI.of(GenericTypeReflector.getFieldType(f, annotatedType)).addAnnotations(f.getAnnotations());
        }

        @Override
        public boolean instanceOf(Type t) {
            return GenericTypeReflector.isSuperType(t, annotatedType.getType());
        }

        @Override
        public @NotNull Type toType() {
            return annotatedType.getType();
        }

        @Override
        public @NotNull AnnotatedType toAnnotatedType() {
            return annotatedType;
        }

        @Override
        public CTypeI<Object> getComponentByName(String name) {
            if (annotatedType instanceof AnnotatedParameterizedType) {
                var aa = ((AnnotatedParameterizedType) annotatedType).getAnnotatedActualTypeArguments();
                if (aa == null) return null;
                var p = getGenericVariables();
                for (int i = 0; i < p.length; i++) {
                    if (p[i].getName().equals(name)) {
                        return CTypeI.of(aa[i]);
                    }
                }
            }
            return null;
        }

        @Override
        public CTypeI<Object> getComponent(int i) {
            if (i == 0 && annotatedType instanceof AnnotatedArrayType) {
                return CTypeI.of(((AnnotatedArrayType) annotatedType).getAnnotatedGenericComponentType());
            } else if (annotatedType instanceof AnnotatedParameterizedType) {
                var aa = ((AnnotatedParameterizedType) annotatedType).getAnnotatedActualTypeArguments();
                if (aa == null || aa.length <= i) return null;
                return CTypeI.of(aa[i]);
            }
            return null;
        }

        @Override
        public boolean isPrimitive() {
            var t = annotatedType.getType();
            return t instanceof Class<?> && ((Class<?>) t).isPrimitive();
        }

        @Override
        public Annotation[] getAnnotations() {
            if (annotations == null) {
                annotations = CTypeI.getAllAnnotations(getRaw()).filter(a -> a.annotationType().isAnnotationPresent(Inherited.class)).addAll(Flow.flow(annotatedType.getAnnotations())).toArray(Annotation.class);
            }
            return annotations;
        }

        @Override
        public CTypeI<Object> resolveReturnType(Method m) {
            return CTypeI.of(GenericTypeReflector.getReturnType(m, annotatedType));
        }

        public String toId() {
            return type.toId() + Math.abs(annotatedType.hashCode());
        }

        @Override
        public boolean isBoxedPrimitive() {
            return type.isBoxedPrimitive();
        }

        @Override
        public CTypeI<T> toPrimitive() {
            CTypeI<T> t = type.toPrimitive();
            if (t == type) return this;
            return CTypeI.of(annotatedType.getAnnotations(), t);
        }

        @Override
        public CTypeI<T> unbox() {
            var t = type.unbox();
            if (t == type) return this;
            return CTypeI.of(annotatedType.getAnnotations(), t);
        }

        @Override
        public CTypeI<Object>[] getParameters() {
            if (parameters == null) {
                if (annotatedType instanceof AnnotatedParameterizedType) {
                    parameters = RU.cast(new CTypeI<?>[((AnnotatedParameterizedType) annotatedType).getAnnotatedActualTypeArguments().length]);
                    for (int i = 0; i < parameters.length; i++) {
                        parameters[i] = CTypeI.of(((AnnotatedParameterizedType) annotatedType).getAnnotatedActualTypeArguments()[i]);
                    }
                } else {
                    parameters = RU.cast(new CTypeI<?>[0]);
                }
            }
            return parameters;
        }

        @Override
        public CTypeI<? super T> getParent() {
            return type.getParent();
        }

        @Override
        public Class<T> getRaw() {
            return type.getRaw();
        }

        @Override
        public CTypeI<T> tryToBox() {
            if (!isPrimitive()) return this;
            if (annotations.length == 0) {
                return type.tryToBox();
            }
            return CTypeI.of(annotations, type.tryToBox());
        }
    }

    static final class PType<T> extends BType<T> {
        private final Type type;
        String id;
        private CTypeI<Object>[] parameters;
        private Annotation[] annotations;

        public PType(Type type) {
            super(type);
            assert type != null;
            this.type = type;
        }

//        @Override
//        public boolean equals(Object o) {
//            if (this == o) return true;
//            if (o == null || getClass() != o.getClass()) return false;
//            PType<Object> pType = RU.cast(o);
//            return type.equals(pType.type);
//        }

        @Override
        public CTypeI<T> tryToBox() {
            var r = getRaw2();
            if (!r.isPrimitive()) return this;
            CTypeI<?> res = this;
            if (r == byte.class) {
                res = BYTE_BOX;
            } else if (r == boolean.class) {
                res = BOOLEAN_BOX;
            } else if (r == short.class) {
                res = SHORT_BOX;
            } else if (r == int.class) {
                res = INT_BOX;
            } else if (r == long.class) {
                res = LONG_BOX;
            } else if (r == float.class) {
                res = FLOAT_BOX;
            } else if (r == double.class) {
                res = DOUBLE_BOX;
            }
            return RU.cast(res);
        }

        @Override
        public boolean instanceOf(Type t) {
            return GenericTypeReflector.isSuperType(t, type);
        }

        @Override
        public CTypeI<Object> resolveExtend(CTypeI<Object> inf) {
            return RU.resolveExtend(inf, this);
        }

        @Override
        public CTypeI<Object> resolveTypeField(Field f) {
            return CTypeI.of(GenericTypeReflector.getFieldType(f, GenericTypeReflector.annotate(type))).addAnnotations(f.getAnnotations());
        }

        @Override
        public @NotNull Type toType() {
            return type;
        }

        @Override
        public CTypeI<Object> getComponentByName(String name) {
            if (type instanceof GenericArrayType) {
                return null;
            } else if (type instanceof Class<?> && ((Class<?>) type).isArray()) {
                return getComponent().getComponentByName(name);
            }
            if (type instanceof ParameterizedType) {
                var aa = ((ParameterizedType) type).getActualTypeArguments();
                if (aa == null || aa.length == 0) return null;
                var p = getGenericVariables();
                for (int i = 0; i < p.length; i++) {
                    if (p[i].getName().equals(name)) {
                        return CTypeI.of(aa[i]);
                    }
                }
            }
            return null;
        }

        @Override
        public CTypeI<Object> getComponent(int i) {
            if (i == 0) {
                if (type instanceof GenericArrayType) {
                    return CTypeI.of(((GenericArrayType) type).getGenericComponentType());
                } else if (type instanceof Class<?> && ((Class<?>) type).isArray()) {
                    return CTypeI.<Object>of(((Class<?>) type).getComponentType());
                }
            }
            if (type instanceof ParameterizedType) {
                var aa = ((ParameterizedType) type).getActualTypeArguments();
                if (aa == null || aa.length <= i) return null;
                return CTypeI.of(aa[i]);
            }
            return null;
        }

        @Override
        public boolean isPrimitive() {
            return type instanceof Class<?> && ((Class<?>) type).isPrimitive();
        }

        @Override
        public Annotation[] getAnnotations() {
            if (annotations == null) {
                annotations = CTypeI.getAllAnnotations(getRaw()).toArray(Annotation.class);
            }
            return annotations;
        }

        @Override
        public CTypeI<Object> resolveReturnType(Method m) {
            return CTypeI.of(GenericTypeReflector.getReturnType(m, GenericTypeReflector.annotate(type)));
        }

        @Override
        public CTypeI<Object>[] resolveTypeArgs(Method m) {
            var aa = GenericTypeReflector.getParameterTypes(m, GenericTypeReflector.annotate(type));
            CTypeI<Object>[] res = RU.cast(new CTypeI<?>[aa.length]);
            for (int i = 0; i < res.length; i++) {
                res[i] = CTypeI.of(aa[i]);
            }
            return res;
        }

        @Override
        public boolean isBoxedPrimitive() {
            if (type instanceof Class) {
                var c = (Class<?>) type;
                return c == Boolean.class || c == Byte.class || c == Short.class || c == Integer.class || c == Float.class || c == Double.class || c == Long.class || c == Character.class;
            }
            return false;
        }

        @Override
        public CTypeI<T> toPrimitive() {
            if (isPrimitive()) return this;
            if (type == Boolean.class) return RU.cast(CTypeI.BOOLEAN);
            if (type == Character.class) return RU.cast(CTypeI.CHAR);
            if (type == Byte.class) return RU.cast(CTypeI.BYTE);
            if (type == Short.class) return RU.cast(CTypeI.SHORT);
            if (type == Integer.class) return RU.cast(CTypeI.INT);
            if (type == Long.class) return RU.cast(CTypeI.LONG);
            if (type == Float.class) return RU.cast(CTypeI.FLOAT);
            if (type == Double.class) return RU.cast(CTypeI.DOUBLE);
            return this;
        }

        @Override
        public CTypeI<T> unbox() {
            return toPrimitive();
        }

        @Override
        public CTypeI<Object>[] getParameters() {
            if (parameters == null) {
                if (type instanceof ParameterizedType) {
                    parameters = RU.cast(new CTypeI<?>[((ParameterizedType) type).getActualTypeArguments().length]);
                    for (int i = 0; i < parameters.length; i++) {
                        parameters[i] = CTypeI.of(((ParameterizedType) type).getActualTypeArguments()[i]);
                    }
                } else if (type instanceof AnnotatedParameterizedType) {
                    parameters = RU.cast(new CTypeI<?>[((AnnotatedParameterizedType) type).getAnnotatedActualTypeArguments().length]);
                    for (int i = 0; i < parameters.length; i++) {
                        parameters[i] = CTypeI.of(((AnnotatedParameterizedType) type).getAnnotatedActualTypeArguments()[i]);
                    }
                } else {
                    parameters = RU.cast(new CTypeI<?>[0]);
                }
            }
            return parameters;
        }

        @Override
        public CTypeI<? super T> getParent() {
            var p = getRaw2();
            if (p.getAnnotatedSuperclass() == null) return null;
            return CTypeI.of(p.getAnnotatedSuperclass());
        }

        @Override
        public String toId() {
            if (id != null) return id;
            if (type instanceof Class<?>) {
                var tt = (Class<?>) type;
                if (tt.isArray()) {
                    id = "Array" + CTypeI.of(tt.getComponentType()).toId();
                    return id;
                } else if (tt.isEnum()) {
                    id = "Enum" + tt.getCanonicalName().replaceAll("[.$]", "_");
                    return id;
                } else {
                    id = tt.getCanonicalName().replaceAll("[.$]", "_");
                    return id;
                }
            } else {
                StringBuilder r = new StringBuilder();
                if (type instanceof ParameterizedType) {
                    var tt = (ParameterizedType) type;
                    r.append(CTypeI.of(tt.getRawType()).toId()).append("PP");
                    for (var p : tt.getActualTypeArguments()) {
                        r.append("P").append(CTypeI.of(p).toId());
                    }
                } else if (type instanceof WildcardType) {
                    var tt = (WildcardType) type;
                    for (var c : tt.getLowerBounds()) {
                        r.append(CTypeI.of(c).toId());
                    }
                    for (var c : tt.getUpperBounds()) {
                        r.append(CTypeI.of(c).toId());
                    }
                } else if (type instanceof GenericArrayType) {
                    var tt = (GenericArrayType) type;
                    r.append("Array").append(CTypeI.of(tt.getGenericComponentType()).toId());
                } else if (type instanceof TypeVariable) {
                    throw new RuntimeException("Not resolved type");
                } else {
                    throw new RuntimeException("Unknown type");
                }
                return r.toString();
            }
        }

        @Override
        @SuppressWarnings("unchecked")
        public Class<T> getRaw() {
            return (Class<T>) GenericTypeReflector.erase(type);
        }
    }
}

