package io.aether.utils;

import io.aether.logger.Log;
import io.aether.utils.futures.AFuture;
import io.aether.utils.interfaces.Destroyable;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

public class Destroyer implements Destroyable {
    public final String name;
    private final Queue<Destroyable> queue = new ConcurrentLinkedQueue<>();
    volatile AtomicReference<AFuture> destroyFuture = new AtomicReference<>();

    public Destroyer(String name) {
        this.name = name;
    }

    public boolean isDestroyed() {
        return destroyFuture.get() != null;
    }

    public void add(Destroyable destroyable) {
        queue.add(destroyable);
    }

    @Override
    public String toString() {
        return name;
    }

    @Override
    public AFuture destroy(boolean force) {
        AFuture res = AFuture.make();
        if (!destroyFuture.compareAndSet(null, res)) {
            return destroyFuture.get();
        }
        AtomicInteger counter = new AtomicInteger();
        while (true) {
            var e = queue.poll();
            if (e == null) break;
            counter.incrementAndGet();
            try {
                e.destroy(force).to(() -> {
                    if (counter.decrementAndGet() == 0) {
                        res.tryDone();
                    }
                }).timeout(5, () -> {
                    Log.warn("Timeout destroy: $unit", "unit", e);
                });
            } catch (Exception ex) {
                Log.warn("destroy exception", ex);
            }
        }
        if (counter.decrementAndGet() <= 0) {
            res.tryDone();
        }
        res.timeout(5, () -> {
            Log.warn("Timeout destroy all: $unit", "unit", this);
        });
        return res;
    }

    public void add(ScheduledFuture<?> os) {
        add(new Destroyable() {
            @Override
            public String toString() {
                return os.toString();
            }

            @Override
            public AFuture destroy(boolean force) {
                os.cancel(force);
                return AFuture.completed();
            }
        });
    }

    public void add(AutoCloseable os) {
        add(new Destroyable() {
            @Override
            public String toString() {
                return os.toString();
            }

            @Override
            public AFuture destroy(boolean force) {
                if (force) {
                    try {
                        os.close();
                    } catch (Exception e) {
                        Log.warn("destroy exception", e);
                    }
                    return AFuture.completed();
                } else {
                    try {
                        os.close();
                        return AFuture.completed();
                    } catch (Exception e) {
                        return AFuture.doThrow(e);
                    }
                }
            }
        });
    }
}
