package io.aether.utils;

import io.aether.logger.Log;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.JarURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

public class ModuleAutoRun {

    // The file extension used for module auto-run configuration files.
    // Modules should use unique names (e.g., 'mymodule.autorun') to avoid JAR conflicts.
    private static final String AUTO_RUN_FILE_EXTENSION = ".autorun";
    // A dummy resource name used to find the base URL of the JAR file in the classpath.
    private static final String DUMMY_RESOURCE_NAME = "io/aether/utils/ModuleAutoRun.class";


    static {
        // Scan the entire classpath to find all unique module autorun configuration files.
        // This new approach is designed to handle resources inside a single 'shadowJar'.
        Set<URL> autorunUrls = new HashSet<>();

        // 1. Try to find resources using the JAR scanning logic tailored for 'shadowJar'.
        try {
            autorunUrls.addAll(scanJarForAutorunFiles(AUTO_RUN_FILE_EXTENSION));
        } catch (Exception e) {
            Log.error("Error during JAR-based classpath scan.", e);
        }

        // 2. Fallback to the previous java.class.path scanning for directories or separate JARs.
        //    (This is often redundant with shadowJar, but good for completeness).
        if (autorunUrls.isEmpty()) {
            try {
                autorunUrls.addAll(scanClassPathForAutorunFiles(AUTO_RUN_FILE_EXTENSION));
            } catch (Exception e) {
                Log.error("Error during java.class.path fallback scan.", e);
            }
        }

        Log.info("Found " + autorunUrls.size() + " module autorun resources.");

        for (URL resourceUrl : autorunUrls) {
            String className = null;
            try (InputStream inputStream = resourceUrl.openStream();
                 BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream, StandardCharsets.UTF_8))) {

                // The first line of the resource file is expected to be the fully qualified class name.
                className = reader.readLine();
                if (className != null && !className.trim().isEmpty()) {
                    className = className.trim();
                    Log.info("Initializing auto-run module: " + className + " from " + resourceUrl);
                    // Load the class to trigger its static initializer block.
                    Class.forName(className);
                }
            } catch (Exception e) {
                // Catch Exception to ensure a single module failure does not stop the others.
                Log.error("Failed to load autorun class " + className + " from " + resourceUrl, e);
            }
        }
    }

    /**
     * Scans the current executing JAR file (e.g., a shadowJar) for resources
     * with the specified extension in the root. This is the recommended approach
     * for finding bundled resources when using single-JAR assembly tools like shadowJar.
     *
     * @param extension The file extension to search for (e.g., ".autorun").
     * @return A set of unique URLs pointing to the discovered resources.
     */
    private static Set<URL> scanJarForAutorunFiles(String extension) {
        Set<URL> resultUrls = new HashSet<>();
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();

        // Use a known class/resource to find the location of the containing JAR.
        URL classUrl = classLoader.getResource(DUMMY_RESOURCE_NAME);

        if (classUrl != null && "jar".equals(classUrl.getProtocol())) {
            try {
                // The URL will be in the format: jar:file:/path/to/jar.jar!/path/to/class
                JarURLConnection jarConnection = (JarURLConnection) classUrl.openConnection();
                JarFile jarFile = jarConnection.getJarFile();

                Enumeration<JarEntry> entries = jarFile.entries();
                while (entries.hasMoreElements()) {
                    JarEntry entry = entries.nextElement();
                    String name = entry.getName();

                    // Check for files in the root (no path separator '/') and matching the extension.
                    if (name.endsWith(extension) && name.indexOf('/') == -1) {
                        // Use ClassLoader to get the proper URL for reading the resource.
                        URL entryUrl = classLoader.getResource(name);
                        if (entryUrl != null) {
                            resultUrls.add(entryUrl);
                        }
                    }
                }
            } catch (Exception e) {
                Log.warn("Failed to scan the running JAR for resources.", e);
            }
        }

        return resultUrls;
    }


    /**
     * Fallback method to scan the elements listed in the 'java.class.path' system property.
     * This is useful for development environments (directories) or when modules are loaded
     * as separate JAR files, but is often insufficient for 'shadowJar' deployment.
     *
     * @param extension The file extension to search for (e.g., ".autorun").
     * @return A set of unique URLs pointing to the discovered resources.
     */
    private static Set<URL> scanClassPathForAutorunFiles(String extension) {
        Set<URL> resultUrls = new HashSet<>();
        try {
            ClassLoader classLoader = Thread.currentThread().getContextClassLoader();

            // Get all classpath elements (directories and JARs)
            String classpath = System.getProperty("java.class.path");
            String[] pathElements = classpath.split(System.getProperty("path.separator"));

            for (String element : pathElements) {
                File elementFile = new File(element);

                if (!elementFile.exists()) {
                    continue;
                }

                if (element.toLowerCase().endsWith(".jar")) {
                    // Handle JAR file: iterate over JarEntry objects.
                    try (JarFile jar = new JarFile(elementFile)) {
                        Enumeration<JarEntry> entries = jar.entries();
                        while (entries.hasMoreElements()) {
                            JarEntry entry = entries.nextElement();
                            String name = entry.getName();

                            // Check for files in the root (no path separator '/') and matching the extension.
                            if (name.endsWith(extension) && name.indexOf('/') == -1) {
                                // Get the URL via the ClassLoader to ensure a valid URL for openStream().
                                URL entryUrl = classLoader.getResource(name);
                                if (entryUrl != null) {
                                    resultUrls.add(entryUrl);
                                }
                            }
                        }
                    } catch (Exception e) {
                        Log.warn("Failed to scan JAR file: " + element, e);
                    }
                } else if (elementFile.isDirectory()) {
                    // Handle Directory: list files in the root directory.
                    File[] files = elementFile.listFiles((dir, name) -> name.toLowerCase().endsWith(extension));
                    if (files != null) {
                        for (File file : files) {
                            // Get the URL via the ClassLoader using the file name as the resource name.
                            URL fileUrl = classLoader.getResource(file.getName());
                            if (fileUrl != null) {
                                resultUrls.add(fileUrl);
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            Log.error("Error during full classpath scan for extension: " + extension, e);
            return Collections.emptySet();
        }
        return resultUrls;
    }

    /**
     * Placeholder initialization method.
     * @param moduleName The name of the module being initialized.
     */
    public static void initialize(String moduleName) {
//        Log.info("initialize: " + moduleName);
    }
}