package io.aether.utils;

import io.aether.logger.Log;
import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.interfaces.ARunnable;
import io.leangen.geantyref.GenericTypeReflector;
import io.leangen.geantyref.TypeFactory;
import it.unimi.dsi.fastutil.chars.CharConsumer;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntList;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import org.jetbrains.annotations.NotNull;

import java.io.*;
import java.lang.annotation.Annotation;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.lang.reflect.*;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;
import java.util.regex.Pattern;

/**
 * Utility class containing static methods for common operations such as JSON serialization,
 * URI parsing, byte conversion, and scheduling tasks. This class does not contain instance
 * fields or constructors.
 */
public class RU {
    public static final Executor EMPTY_EXECUTOR = Runnable::run;
    public static final Random RND = new Random();
    public static final SecureRandom SECURE_RANDOM = new SecureRandom();
    static final IntList sysLoadingList = new IntArrayList();
    private static final Pattern PATTERN_URI =
            Pattern.compile("((?<protocol>[\\w:]+)/)?((?<user>\\w+(:(?<password>\\w+))?)@)(?<host>[\\w.]+)(:(?<port>\\d))?(/(?<path>\\.*))?");
    private static final ScheduledExecutorService INSTANCE4 =
            Executors.newScheduledThreadPool(2, runnable -> {
                Thread th = new Thread(runnable);
                th.setName("INSTANCE4");
                th.setDaemon(true);
                return th;
            });
    private final static int[] EMPTY_INT_ARRAY = new int[0];
    private final static short[] EMPTY_SHORT_ARRAY = new short[0];
    private final static byte[] EMPTY_BYTE_ARRAY = new byte[0];

    static private final char[] base64_code = {
            '.', '/', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
            'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V',
            'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h',
            'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't',
            'u', 'v', 'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5',
            '6', '7', '8', '9'
    };
    private static final File HOME_DIR = new File(System.getProperty("user.home"));
    private static final File WORK_DIR = new File(System.getProperty("user.dir"));
    private static final AtomicLong contextIdCounter = new AtomicLong();
    static volatile int sysLoading;
    public static void loadLibrary(String libraryName) {
        String resourcePath = getResourcePath(libraryName);

        try (InputStream in = RU.class.getResourceAsStream(resourcePath)) {
            if (in == null) {
                throw new UnsatisfiedLinkError("Could not find native library: " + resourcePath);
            }

            Path tempDir = Files.createTempDirectory("native-lib-extract");
            tempDir.toFile().deleteOnExit();
            Path tempFile = tempDir.resolve("lib" + libraryName);

            Files.copy(in, tempFile, StandardCopyOption.REPLACE_EXISTING);

            System.load(tempFile.toAbsolutePath().toString());

        } catch (Exception e) {
            throw new UnsatisfiedLinkError("Failed to load native library: " + e.getMessage());
        }
    }

    private static String getResourcePath(String libraryName) {
        String osName = System.getProperty("os.name").toLowerCase();
        String extension;
        if (osName.contains("win")) {
            extension = ".dll";
        } else if (osName.contains("mac")) {
            extension = ".dylib";
        } else {
            extension = ".so";
        }
        return "/lib" + libraryName + extension;
    }

    /**
     * Static initializer that schedules a task to monitor system load.
     */
    static {
        INSTANCE4.scheduleAtFixedRate(new Runnable() {
            long cpu0;
            long nice0;
            long system0;
            long idle0;
            private boolean checkFirst;

            @Override
            public void run() {
                try (var fr = new BufferedReader(new FileReader("/proc/stat"))) {
                    var cpuLine = fr.readLine();
                    var s = cpuLine.split(" +");
                    var cpu1 = Long.parseLong(s[1]);
                    var nice1 = Long.parseLong(s[2]);
                    var system1 = Long.parseLong(s[3]);
                    var idle1 = Long.parseLong(s[4]);
                    var cpuDelta = cpu1 - cpu0;
                    var niceDelta = nice1 - nice0;
                    var systemDelta = system1 - system0;
                    var idleDelta = idle1 - idle0;
                    cpu0 = cpu1;
                    nice0 = nice1;
                    system0 = system1;
                    idle0 = idle1;
                    if (checkFirst) {
                        checkFirst = false;
                        return;
                    }
                    var total = cpuDelta + niceDelta + systemDelta + idleDelta;
                    var l = (int) (100.0 * (cpuDelta + niceDelta + systemDelta) / total);
                    sysLoadingList.add(l);
                    while (sysLoadingList.size() >= 6) {
                        sysLoadingList.removeInt(0);
                    }
                    sysLoading = Flow.flow(sysLoadingList).avg();
                } catch (IOException e) {
                    sysLoading = 0;
                }
            }
        }, 0, 1, TimeUnit.SECONDS);
    }

    /**
     * Converts an object to a JSON string.
     *
     * @param val The object to convert
     * @return A JSON string representation of the object
     */
    public static AString toJson(Object val) {
        var s = AString.of();
        toJson(s, val);
        return s;
    }
    public static byte[] sha1(byte[] input) {
        try {
            MessageDigest md = MessageDigest.getInstance("SHA-1");
            return md.digest(input);
        } catch (NoSuchAlgorithmException e) {
            Log.error(e);
            return error(e);
        }
    }
    /**
     * Converts an object to a JSON string, appending the result to the provided AString.
     *
     * @param s      The AString to append the result to
     * @param val     The object to convert
     */
    public static void toJson(AString s, Object val) {
        if (val == null) {
            s.add("null");
        } else if (val instanceof Map) {
            s.add("{");
            boolean first = true;
            for (var e : ((Map<?, ?>) val).entrySet()) {
                if (first) {
                    first = false;
                } else {
                    s.add(",");
                }
                s.add("\"").add(e.getKey()).add("\":");
                toJson(s, e.getValue());
            }
            s.add("}");
        } else if (val instanceof Iterable) {
            s.add("[");
            boolean first = true;
            for (Object o : (Iterable<?>) val) {
                if (first) {
                    first = false;
                } else {
                    s.add(",");
                }
                toJson(s, o);
            }
            s.add("]");
        } else if (val instanceof byte[]) {
            s.add("\"").add(val).add("\"");
        } else if (val.getClass().isArray()) {
            s.add("[");
            boolean first = true;
            var len = Array.getLength(val);
            for (int i = 0; i < len; i++) {
                if (first) {
                    first = false;
                } else {
                    s.add(",");
                }
                toJson(s, Array.get(val, i));
            }
            s.add("]");
        } else if (val instanceof String) {
            s.add("\"").add(val).add("\"");
        }
    }

    /**
     * Parses a string into a URI.
     *
     * @param s The string to parse
     * @return A URI object
     * @throws URISyntaxException If the string is not a valid URI
     */
    public static URI parseURI(String s) throws URISyntaxException {
        var m = PATTERN_URI.matcher(s);
        if (!m.find()) {
            throw new URISyntaxException(s, "URI is bad");
        }
        String auth = null;
        var user = m.group("user");
        var password = m.group("password");
        if (user != null) {
            auth = user;
            if (password != null) {
                auth += ":" + password;
            }
        }
        int port = 0;
        var portString = m.group("port");
        if (portString != null && !portString.isBlank()) {
            try {
                port = Integer.parseInt(portString);
            } catch (Exception e) {
                throw new URISyntaxException(s, "Port is bad: " + portString);
            }
        }
        try {
            return new URI(m.group("protocol"), auth, m.group("host"), port, m.group("path"), null, null);
        } catch (URISyntaxException e) {
            return RU.error(e);
        }
    }

    /**
     * Checks if a byte array starts with specified values.
     *
     * @param src   The byte array to check
     * @param vals  The expected values
     * @return true if the array starts with the values, false otherwise
     */
    public static boolean startWith(byte[] src, int... vals) {
        if (src == null) return false;
        if (src.length < vals.length) return false;
        for (int i = 0; i < vals.length; i++) {
            if (src[i] != vals[i]) return false;
        }
        return true;
    }

    /**
     * Encodes a byte array to Base64 using the specified character consumer.
     *
     * @param d The byte array to encode
     * @param rs The character consumer to store the result
     * @throws IllegalArgumentException If the input is invalid
     */
    public static void encodeBase64(byte[] d, CharConsumer rs)
            throws IllegalArgumentException {
        encodeBase64(d, d.length, rs);
    }

    /**
     * Encodes a byte array to Base64 using the specified length and character consumer.
     *
     * @param d The byte array to encode
     * @param len The number of bytes to encode
     * @param rs The character consumer to store the result
     * @throws IllegalArgumentException If the input is invalid
     */
    public static void encodeBase64(byte[] d, int len, CharConsumer rs)
            throws IllegalArgumentException {
        int off = 0;
        int c1, c2;
        if (len <= 0 || len > d.length)
            throw new IllegalArgumentException("Invalid len");
        while (off < len) {
            c1 = d[off++] & 0xff;
            rs.accept(base64_code[(c1 >> 2) & 0x3f]);
            c1 = (c1 & 0x03) << 4;
            if (off >= len) {
                rs.accept(base64_code[c1 & 0x3f]);
                break;
            }
            c2 = d[off++] & 0xff;
            c1 |= (c2 >> 4) & 0x0f;
            rs.accept(base64_code[c1 & 0x3f]);
            c1 = (c2 & 0x0f) << 2;
            if (off >= len) {
                rs.accept(base64_code[c1 & 0x3f]);
                break;
            }
            c2 = d[off++] & 0xff;
            c1 |= (c2 >> 6) & 0x03;
            rs.accept(base64_code[c1 & 0x3f]);
            rs.accept(base64_code[c2 & 0x3f]);
        }
    }

    /**
     * Encodes a byte array to Base64 using the specified StringBuilder.
     *
     * @param d The byte array to encode
     * @param rs The StringBuilder to store the result
     * @throws IllegalArgumentException If the input is invalid
     */
    public static void encodeBase64(byte[] d, StringBuilder rs)
            throws IllegalArgumentException {
        encodeBase64(d, d.length, rs);
    }

    /**
     * Encodes a byte array to Base64 using the specified length and StringBuilder.
     *
     * @param d The byte array to encode
     * @param len The number of bytes to encode
     * @param rs The StringBuilder to store the result
     * @throws IllegalArgumentException If the input is invalid
     */
    public static void encodeBase64(byte[] d, int len, StringBuilder rs)
            throws IllegalArgumentException {
        int off = 0;
        int c1, c2;
        if (len <= 0 || len > d.length)
            throw new IllegalArgumentException("Invalid len");
        while (off < len) {
            c1 = d[off++] & 0xff;
            rs.append(base64_code[(c1 >> 2) & 0x3f]);
            c1 = (c1 & 0x03) << 4;
            if (off >= len) {
                rs.append(base64_code[c1 & 0x3f]);
                break;
            }
            c2 = d[off++] & 0xff;
            c1 |= (c2 >> 4) & 0x0f;
            rs.append(base64_code[c1 & 0x3f]);
            c1 = (c2 & 0x0f) << 2;
            if (off >= len) {
                rs.append(base64_code[c1 & 0x3f]);
                break;
            }
            c2 = d[off++] & 0xff;
            c1 |= (c2 >> 6) & 0x03;
            rs.append(base64_code[c1 & 0x3f]);
            rs.append(base64_code[c2 & 0x3f]);
        }
    }

    /**
     * Schedules a task to run periodically with the specified period.
     *
     * @param periodMs The period in milliseconds
     * @param t The task to schedule
     * @return A ScheduledFuture representing the scheduled task
     */
    public static ScheduledFuture<?> schedule(long periodMs, ARunnable t) {
//		StackTraceElement[] st = Thread.currentThread().getStackTrace();
        StackTraceElement[] st = null;
        return INSTANCE4.schedule(Log.wrap(() -> {
            try {
                t.run2();
            } catch (Throwable e) {
                concatStackTrace(st, e);
            }
        }), periodMs, TimeUnit.MILLISECONDS);
    }

    public static ScheduledFuture<?> scheduleAtFixedRate(Destroyer resTo, Executor executor, int durationSeconds, ARunnable task) {
        var res = scheduleAtFixedRate(executor, durationSeconds, task);
        resTo.add(res);
        return res;
    }

    public static ScheduledFuture<?> scheduleAtFixedRate(Executor executor, int durationSeconds, ARunnable t) {
//		StackTraceElement[] st = Thread.currentThread().getStackTrace();
        StackTraceElement[] st = null;
        return INSTANCE4.scheduleAtFixedRate(() -> executor.execute(() -> {
            try {
                t.run2();
            } catch (Throwable e) {
                concatStackTrace(st, e);
            }
        }), 0, durationSeconds, TimeUnit.SECONDS);
    }

    public static ScheduledFuture<?> scheduleAtFixedRate(long period, ARunnable t) {
        return scheduleAtFixedRate(period, TimeUnit.MILLISECONDS, t);
    }

    public static ScheduledFuture<?> scheduleAtFixedRate(Destroyer resTo, long period, TimeUnit timeUnit, ARunnable t) {
        var res = scheduleAtFixedRate(period, timeUnit, t);
        resTo.add(res);
        return res;
    }

    public static ScheduledFuture<?> scheduleAtFixedRate(long period, TimeUnit timeUnit, ARunnable t) {
//		StackTraceElement[] st = Thread.currentThread().getStackTrace();
        StackTraceElement[] st = null;
        return INSTANCE4.scheduleAtFixedRate(Log.wrap(() -> {
            try {
                t.run2();
            } catch (Throwable e) {
                concatStackTrace(st, e);
            }
        }), 0, period, timeUnit);
    }

    public static CTypeI<Object> parseType(String s) {
        try {
            return parseType(s, new int[]{0});
        } catch (ClassNotFoundException e) {
            error(e);
            return null;
        }
    }

    public static <T extends Throwable> T filterFrontAndBackStackTrace(T e, Class<?>... skipClasses) {
        e.setStackTrace(filterFrontAndBackStackTrace(e.getStackTrace(), concatArrays(e.getClass(), skipClasses)));
        return e;
    }

    private static Type parseType0(String s) throws ClassNotFoundException {
        switch (s) {
            case "boolean":
                return boolean.class;
            case "char":
                return char.class;
            case "byte":
                return byte.class;
            case "short":
                return short.class;
            case "int":
                return int.class;
            case "long":
                return long.class;
            case "boolean[]":
                return boolean[].class;
            case "char[]":
                return char[].class;
            case "byte[]":
                return byte[].class;
            case "short[]":
                return short[].class;
            case "int[]":
                return int[].class;
            case "long[]":
                return long[].class;
            default:
                return Class.forName(s);
        }
    }

    public static StackTraceElement[] filterFrontAndBackStackTrace(StackTraceElement[] ss, Class<?>... skipClasses) {
        int top = 0;
        loop:
        for (int i = 0; i < ss.length; i++) {
            if (ss[i].getClassName().equals(Thread.class.getName())) continue;
            for (var s : skipClasses) {
                assert s != null;
                if (ss[i].getClassName().equals(s.getName())) continue loop;
            }
            top = i;
            break;
        }
        return filterStackTraceBack(Arrays.copyOfRange(ss, top, ss.length));
    }

    public static StackTraceElement[] filterStackTraceBack(StackTraceElement[] ss, Class<?>... skipClasses) {
        int bottom = ss.length;
        for (int i = ss.length - 1; i >= 0; i--) {
            if (ss[i].getClassName().contains("aether")) {
                bottom = i + 1;
                break;
            }
        }
        return Arrays.copyOfRange(ss, 0, bottom);
    }

    public static CTypeI<Object> parseType(String s, int[] i) throws ClassNotFoundException {
        int i1 = s.indexOf(',', i[0]);
        int i2 = s.indexOf('<', i[0]);
        int i3 = s.indexOf('>', i[0]);
        int min = s.length();
        if (i1 >= 0 && min > i1) min = i1;
        if (i2 >= 0 && min > i2) min = i2;
        if (i3 >= 0 && min > i3) min = i3;
        if (min == s.length()) {
            var ss = s.substring(i[0]).trim();
            if (ss.isBlank()) {
                i[0] = -1;
                return null;
            }
            var res = parseType0(ss);
            i[0] = -1;
            return CTypeI.of(res);
        }
        if (min == i1) {
            var ss = s.substring(i[0], i1).trim();
            if (ss.isBlank()) {
                i[0] = min + 1;
                return null;
            }
            var res = parseType0(ss);
            i[0] = i1 + 1;
            return CTypeI.of(res);
        }
        if (min == i2) {
            var res = parseType0(s.substring(i[0], i2).trim());
            i[0] = i2 + 1;
            List<Type> pp = new ObjectArrayList<>();
            while (true) {
                var t = parseType(s, i);
                if (t == null) {
                    if (pp.isEmpty()) return CTypeI.of(res);
                    return CTypeI.of(res, pp);
                } else {
                    pp.add(t.toType());
                }
            }
        }
        var ss = s.substring(i[0], i3).trim();
        i[0] = i3 + 1;
        if (ss.isBlank()) {
            return null;
        } else {
            return CTypeI.<Object>of(Class.forName(ss));
        }
    }

    public static Flow<File> getAllPaths() {
        return Flow.flow(System.getProperty("java.class.path").split(":"))
                .distinct()
                .map(File::new);
    }

    public static int sizeOf(Parameter p) {
        var t = p.getType();
        if (t == String.class) {
            return 2;
        } else if (t == byte.class) {
            return 1;
        } else if (t == short.class) {
            return 2;
        } else if (t == int.class) {
            return 4;
        } else if (t == long.class) {
            return 8;
        } else if (t == float.class) {
            return 4;
        } else if (t == double.class) {
            return 8;
        } else if (t == boolean.class) {
            return 1;
        }
        throw new UnsupportedOperationException(String.valueOf(t));
    }

    @SuppressWarnings("unchecked")
    public static <S, T> T[] mapArray(S[] src, Class<T> targetType, AFunction<S, T> mapper) {
        T[] res = (T[]) Array.newInstance(targetType, src.length);
        for (int i = 0; i < src.length; i++) {
            res[i] = mapper.apply(src[i]);
        }
        return res;
    }

    public static int ceilDiv(int x, int y) {
        final int q = x / y;
        // if the signs are the same and modulo not zero, round up
        if ((x ^ y) >= 0 && (q * y != x)) {
            return q + 1;
        }
        return q;
    }

    public static CTypeI<Object> resolveExtend(CTypeI<?> parent, CTypeI<?> owner) {
        return CTypeI.of(GenericTypeReflector.getExactSuperType(owner.toAnnotatedType(), parent.getRaw2()));
    }

    public static <T> T error(StackTraceElement[] st, Throwable e) {
        concatStackTrace(st, e);
        error(e);
        return null;
    }

    @SuppressWarnings("unchecked")
    public static <T, E extends Throwable> T error(Throwable e) throws E {
//        Log.error(e);
        throw (E) e;
    }

    public static int availableProcessors() {
        return Runtime.getRuntime().availableProcessors();
    }

    public static void sleep(long ms) {
        try {
            Thread.sleep(ms);
        } catch (InterruptedException e) {
            error(e);
        }
    }

    public static double getNetLoading() {
        try (var fr = new BufferedReader(new FileReader("/proc/stat"))) {
            return 0;
        } catch (Exception e) {
            return error(e);
        }
    }

    public static int getSystemLoading() {
        return sysLoading;
    }

    @SuppressWarnings("unchecked")
    public static <T> T[] concatArrays(T v, T[] vv) {
        if (vv.length == 0) {
            var res = (T[]) Array.newInstance(v.getClass(), 1);
            res[0] = v;
            return res;
        }
        T[] res = (T[]) Array.newInstance(
                vv.getClass().getComponentType(),
                vv.length + 1);
        System.arraycopy(vv, 0, res, 1, vv.length);
        res[0] = v;
        return res;
    }

    @SuppressWarnings("unchecked")
    public static <T> T cast(Object t) {
        return (T) t;
    }

    public static <T> T[] concatArrays(T[] v, T[] vv) {
        T[] res = Arrays.copyOf(vv, v.length + vv.length);
        System.arraycopy(vv, 0, res, v.length, vv.length);
        return res;
    }

    public static void concatStackTrace(StackTraceElement[] st, Throwable e) {
        if (st == null) return;
//        e.setStackTrace(Flow.flow(st).skip(2).addAllEls(e.getStackTrace()).toArray(StackTraceElement.class));
        e.setStackTrace(Flow.flow(e.getStackTrace()).addAll(Flow.flow(st).skip(2)).toArray(StackTraceElement.class));
    }

    public static <T> int updateMax(T owner, AtomicIntegerFieldUpdater<T> updater, int value) {
        int i;
        int nv;
        do {
            i = updater.get(owner);
            nv = Math.max(i, value);
        } while (!updater.compareAndSet(owner, i, nv));
        return nv;
    }

    public static <T> int updateMin(T owner, AtomicIntegerFieldUpdater<T> updater, int value) {
        int i;
        int nv;
        do {
            i = updater.get(owner);
            nv = Math.min(i, value);
        } while (!updater.compareAndSet(owner, i, nv));
        return nv;
    }

    public static <T> long updateMax(T owner, AtomicLongFieldUpdater<T> updater, long value) {
        long i;
        long nv;
        do {
            i = updater.get(owner);
            nv = Math.max(i, value);
        } while (!updater.compareAndSet(owner, i, nv));
        return nv;
    }

    public static <T> void readAll(Queue<T> q, AConsumer<T> o) {
        if (q == null) return;
        while (true) {
            var element = q.poll();
            if (element == null) return;
            o.accept(element);
        }
    }

    public static void runAll(Queue<ARunnable> q) {
        if (q == null) return;
        while (true) {
            var element = q.poll();
            if (element == null) return;
            try {
                element.run();
            } catch (Exception e) {
                Log.error(e);
            }
        }
    }

    public static String escape(String s) {
        return escape("\\$\"", s);
    }

    public static String escape(String chars, String s) {
        for (int i = 0; i < chars.length(); i++) {
            var c = chars.charAt(i);
            s = s.replace(Character.toString(c), "\\" + c);
        }
        return s;
    }

    public static CharSequence unescape(CharSequence s) {
        StringBuilder sb = null;
        int last = 0;
        for (int i = 0; i < s.length(); i++) {
            var ch = s.charAt(i);
            if (ch == '\\') {
                if (sb == null) {
                    sb = new StringBuilder(s.length() + 10);
                }
                sb.append(s, last, i);
                i++;
                last = i;
            }
        }
        if (sb == null) return s;
        if (last < s.length()) sb.append(s, last, s.length());
        return sb;
    }

    public static short[] convertBytesToShortArray(byte[] bytes) {
        if (bytes == null || bytes.length == 0) return EMPTY_SHORT_ARRAY;
        var res = new short[bytes.length / 2];
        for (int i = 0; i < res.length; i++) {
            int v = 0;
            for (int ii = 0; ii < 2; ii++) {
                v = v << 8 | Byte.toUnsignedInt(bytes[i * 2 + ii]);
            }
            res[i] = (short) v;
        }
        return res;
    }

    public static int[] convertBytesToIntArray(byte[] bytes) {
        if (bytes == null || bytes.length == 0) return EMPTY_INT_ARRAY;
        var res = new int[bytes.length / 4];
        for (int i = 0; i < res.length; i++) {
            int v = 0;
            for (int ii = 0; ii < 4; ii++) {
                v = v << 8 | Byte.toUnsignedInt(bytes[i * 4 + ii]);
            }
            res[i] = v;
        }
        return res;
    }

    public static byte[] convertShortToByteArray(int data) {
        return convertShortToByteArray((short) data);
    }

    public static byte[] convertShortToByteArray(short data) {
        return new byte[]{(byte) data, (byte) (data >> 8)};
    }

    public static byte[] convertShortToByteArray(short[] data) {
        if (data == null || data.length == 0) return EMPTY_BYTE_ARRAY;
        var res = new byte[data.length * 4];
        int p = 0;
        for (var v : data) {
            for (int i = 0; i < 2; i++) {
                res[p++] = (byte) (v >> 8 * (1 - i));
            }
        }
        return res;
    }

    public static byte[] convertIntToByteArray(int[] data) {
        if (data == null || data.length == 0) return EMPTY_BYTE_ARRAY;
        var res = new byte[data.length * 4];
        int p = 0;
        for (var v : data) {
            for (int i = 0; i < 4; i++) {
                res[p++] = (byte) (v >> 8 * (3 - i));
            }
        }
        return res;
    }

    public static long time() {
        return System.currentTimeMillis();
    }

    public static long bytesToLong(byte[] bytes, int offset) {
        return ((long) bytes[offset]     << 56) |
               ((long) (bytes[offset+1] & 0xFF) << 48) |
               ((long) (bytes[offset+2] & 0xFF) << 40) |
               ((long) (bytes[offset+3] & 0xFF) << 32) |
               ((long) (bytes[offset+4] & 0xFF) << 24) |
               ((long) (bytes[offset+5] & 0xFF) << 16) |
               ((long) (bytes[offset+6] & 0xFF) << 8)  |
               ((long) (bytes[offset+7] & 0xFF));
    }
    public static long bytesToLongLittleEndian(byte[] bytes, int offset) {
        return ((long) (bytes[offset+7] & 0xFF) << 56) |
               ((long) (bytes[offset+6] & 0xFF) << 48) |
               ((long) (bytes[offset+5] & 0xFF) << 40) |
               ((long) (bytes[offset+4] & 0xFF) << 32) |
               ((long) (bytes[offset+3] & 0xFF) << 24) |
               ((long) (bytes[offset+2] & 0xFF) << 16) |
               ((long) (bytes[offset+1] & 0xFF) << 8)  |
               ((long) (bytes[offset]   & 0xFF));
    }

    private static AnnotatedType arrayOf(AnnotatedType componentType, Annotation[] annotations) {
        return TypeFactory.arrayOf(componentType, annotations);
    }

    public static String[] splitWithQuotes(String s) {
        List<String> l = new ArrayList<>();
        StringBuilder sb = new StringBuilder();
        boolean inString = false;
        boolean escape = false;
        char q = '0';
        for (var c : s.toCharArray()) {
            if (inString) {
                if (escape) {
                    escape = false;
                    sb.append(c);
                } else if (c == '\\') {
                    escape = true;
                    sb.append(c);
                } else if (c == q) {
                    inString = false;
                    sb.append(c);
                    l.add(sb.toString());
                    sb.setLength(0);
                } else {
                    sb.append(c);
                }
            } else {
                if (c == '"' || c == '\'') {
                    inString = true;
                    q = c;
                    sb.append(c);
                } else if (c == ' ') {
                    l.add(sb.toString());
                    sb.setLength(0);
                } else {
                    sb.append(c);
                }
            }
        }
        return l.toArray(new String[0]);
    }

    public static boolean isDeclaredInClass(Field f, Class<?> rootClass) {
        assert rootClass != null;
        var c = rootClass;
        while (c != null) {
            if (f.getDeclaringClass() == c) return false;
            c = c.getSuperclass();
        }
        return rootClass.isAssignableFrom(f.getDeclaringClass());
    }

    public static void executeShellBashSudo(String password, String cmd) {
        executeShellBash("echo \"" + password + "\" | sudo -S " + cmd);
    }

    public static void executeShellBash(String cmd) {
        try {
            ProcessBuilder process = new ProcessBuilder("/usr/bin/bash", "-c", cmd);
            process.redirectErrorStream(true);
            Process proc = process.start();
            proc.waitFor();
            proc.getInputStream().transferTo(System.out);
            proc.getErrorStream().transferTo(System.out);
            proc.destroy();
        } catch (Exception e) {
            RU.error(e);
        }
    }

    public static void executeShell(String s) {
        try {
            var args = splitWithQuotes(s);
            ProcessBuilder process = new ProcessBuilder(args);
            process.redirectErrorStream(true);
            Process proc = process.start();
            proc.waitFor();
            proc.getInputStream().transferTo(System.out);
            proc.getErrorStream().transferTo(System.out);
            proc.destroy();
        } catch (Exception e) {
            RU.error(e);
        }
    }

    public static <T extends Enum<T>> Class<Enum<T>> castEnum(Class<?> raw) {
        return cast(raw);
    }

    public static <T> T[] removeElement(T[] a, int i) {
        if (a == null || a.length == 0) return a;
        assert i >= 0 && i < a.length;
        if (a.length == 1) {
            return cast(Array.newInstance(a.getClass().getComponentType(), 0));
        }
        if (i == 0) {
            return Arrays.copyOfRange(a, 1, a.length);
        } else if (i == a.length - 1) {
            return Arrays.copyOfRange(a, 0, a.length - 1);
        } else {
            T[] na = cast(Array.newInstance(a.getClass().getComponentType(), a.length - 1));
            System.arraycopy(a, 0, na, 0, i);
            System.arraycopy(a, 0, na, i + 1, a.length - i - 1);
            return na;
        }
    }

    public static MethodHandle unreflect(MethodHandles.Lookup lookup, Method method) {
        try {
            return lookup.unreflect(method);
        } catch (IllegalAccessException e) {
            try {
                var privateLookup = MethodHandles.privateLookupIn(method.getDeclaringClass(), MethodHandles.lookup());
                return privateLookup.unreflect(method);
            } catch (IllegalAccessException ex) {
                return RU.error(ex);
            }
        }
    }

    public static VarHandle unreflect(Field f) {
        return CTypeI.of(f.getDeclaringClass()).getFieldVarHandle(f);
    }

    public static Executor debugExecutor(Executor executor) {
        return new Executor() {
            @Override
            public void execute(@NotNull Runnable command) {
                var stack = Thread.currentThread().getStackTrace();
                executor.execute(() -> {
                    try {
                        command.run();
                    } catch (Throwable e) {
                        e.setStackTrace(
                                Flow.flow(e.getStackTrace()).skipLast(4)
                                        .addAll(Flow.flow(stack).skip(3)).toArray(StackTraceElement.class));
                        throw e;
                    }
                });
            }
        };
    }

    public static <T> T errorUnsupported() {
        return error(new UnsupportedOperationException());
    }

    public static File homeDir() {
        return HOME_DIR;
    }

    public static File workDir() {
        return WORK_DIR;
    }

    public static long newContextId() {
        return contextIdCounter.getAndIncrement();
    }

    public static File selfExecuteCmdJar() {
        var cmd = ProcessHandle.current().info().commandLine().orElse("");
        var res = cmd.replaceAll(".*?/java\\s+(-+\\S+\\s+)*(\\S+.jar).*", "$2");
        return new File(RU.workDir(), res);
    }

    public static String selfExecuteCmdWithoutArgs() {
        var cmd = ProcessHandle.current().info().commandLine().orElse("");
        var res = cmd.replaceAll(".*?/java\\s+(-+\\S+\\s+)*(\\S+.jar).*", "java $1$2");
        return res;
    }

    public static File tempFile() {
        try {
            return File.createTempFile("temp", "");
        } catch (IOException e) {
            return RU.error(e);
        }
    }

    public static void printStackTrace() {
        Log.debug(Flow.flow(Thread.currentThread().getStackTrace()).join("\n"));
    }
}
