package io.aether.utils;

import io.aether.logger.Log;
import io.aether.utils.interfaces.ASupplier;
import org.jetbrains.annotations.NotNull;

import java.util.concurrent.*;
import java.util.function.Supplier;

public class TimeoutChecker {

    private static final ScheduledExecutorService service = Executors.newSingleThreadScheduledExecutor(new ThreadFactory() {
        @Override
        public Thread newThread(@NotNull Runnable r) {
            Thread t=new Thread(r);
            t.setName("Timeout checker");
            t.setDaemon(true);
            return t;
        }
    });
    private final long ms;
    private final Object message;
    private volatile long startTime;
    private volatile ScheduledFuture<?> scTask;
    private volatile boolean successful;

    private TimeoutChecker(int seconds, Object message) {
        this(seconds*1000L,message);
    }

    private TimeoutChecker(long ms, Object message) {
        this.ms = ms;
        assert message != null;
        this.message = message;
    }

    private TimeoutChecker addTask() {
        startTime = RU.time();
        ScheduledFuture<?> scTask = service.schedule(() -> {
            if (successful) return;
            String msg = (message instanceof Supplier<?> ? String.valueOf(((Supplier<?>) message).get()) : String.valueOf(message));
            Log.warn("time $time > $limit: $msg",
                    "time", ((RU.time() - startTime)),
                    "limit", ms,
                    "msg", msg);
        }, ms, TimeUnit.MILLISECONDS);
        var t = this.scTask;
        if (t != null) {
            t.cancel(true);
        }
        this.scTask = scTask;
        return this;
    }

    public void done() {
        successful = true;
        var t = this.scTask;
        if (t != null) {
            t.cancel(true);
            scTask = null;
        }
    }

    public void next() {
        done();
        addTask();
    }

    public void resume() {
        addTask();
    }

    public static TimeoutChecker error5() {
        return error(5);
    }

    public static TimeoutChecker errorMs(long ms, ASupplier<Object> message) {
        return new TimeoutChecker(ms, message).addTask();
    }

    public static TimeoutChecker error(int seconds, ASupplier<Object> message) {
        return new TimeoutChecker(seconds, message).addTask();
    }

    public static TimeoutChecker error(int seconds, Object message) {
        return new TimeoutChecker(seconds, message).addTask();
    }

    public static TimeoutChecker error10() {
        return error(10);
    }

    public static TimeoutChecker error20() {
        return error(20);
    }

    public static TimeoutChecker error40() {
        return error(40);
    }

    public static TimeoutChecker error(int seconds) {
        return error(seconds, () -> "");
    }

    public static TimeoutChecker info(long timeout) {
        return new TimeoutChecker(timeout, null);
    }
}
