package io.aether.utils;

import io.aether.utils.flow.Flow;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

public class WeakIdenticalConcurrentHashMap<K, V> implements ConcurrentMap<K, V> {
    private final Map<MaskI<K>, MaskI<V>> body = new ConcurrentHashMap<>();
    private final ReferenceQueue<K> referenceQueue = new ReferenceQueue<>();

    @Override
    public V putIfAbsent(@NotNull K key, V value) {
        return unmask(body.putIfAbsent(new MaskSample<>(key), new MaskSample<>(value)));
    }

    @Override
    public boolean remove(@NotNull Object key, Object value) {
        return body.remove(new MaskSample<>(key), new MaskSample<>(value));
    }

    @Override
    public boolean replace(@NotNull K key, @NotNull V oldValue, @NotNull V newValue) {
        return body.replace(new MaskSample<>(key), new MaskSample<>(oldValue), new Mask<>(newValue));
    }

    @Override
    public V replace(@NotNull K key, @NotNull V value) {
        return unmask(body.replace(new MaskSample<>(key), new Mask<>(value)));
    }

    @Override
    public int size() {
        return body.size();
    }

    @Override
    public boolean isEmpty() {
        return body.isEmpty();
    }

    @Override
    public boolean containsKey(Object key) {
        return body.containsKey(new MaskSample<>(key));
    }

    @Override
    public boolean containsValue(Object value) {
        return body.containsValue(new MaskSample<>(value));
    }

    @Override
    public V get(Object key) {
        return unmask(body.get(new MaskSample<>(key)));
    }

    private void flushMem() {
        while (true) {
            Mask<K> v = RU.cast(referenceQueue.poll());
            if (v == null) break;
            body.remove(v);
        }
    }

    @Override
    public @Nullable V put(K key, V value) {
        flushMem();
        return unmask(body.put(new Mask<>(key, referenceQueue), new Mask<>(value)));
    }

    @Override
    public V remove(Object key) {
        return unmask(body.remove(new MaskSample<>(key)));
    }

    @Override
    public void putAll(@NotNull Map<? extends K, ? extends V> m) {
        for (var e : m.entrySet()) {
            put(e.getKey(), e.getValue());
        }
    }

    @Override
    public void clear() {
        body.clear();
    }

    @Override
    public @NotNull Set<K> keySet() {
        return new Set<K>() {
            @Override
            public int size() {
                return WeakIdenticalConcurrentHashMap.this.size();
            }

            @Override
            public boolean isEmpty() {
                return WeakIdenticalConcurrentHashMap.this.isEmpty();
            }

            @Override
            public boolean contains(Object o) {
                return WeakIdenticalConcurrentHashMap.this.containsKey(o);
            }

            @Override
            public @NotNull Iterator<K> iterator() {
                var it = body.keySet().iterator();
                return new Iterator<K>() {
                    K last;

                    @Override
                    public boolean hasNext() {
                        while (last == null && it.hasNext()) {
                            last = it.next().get();
                        }
                        return last != null;
                    }

                    @Override
                    public K next() {
                        var k = last;
                        last = null;
                        return k;
                    }
                };
            }

            @Override
            public @NotNull Object @NotNull [] toArray() {
                return Flow.flow(iterator()).toArray();
            }

            @Override
            public @NotNull <T> T @NotNull [] toArray(@NotNull T @NotNull [] a) {
                return Flow.flow(iterator()).toList().toArray(a);
            }

            @Override
            public boolean add(K k) {
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean remove(Object o) {
                return WeakIdenticalConcurrentHashMap.this.remove(new MaskSample<>(o)) != null;
            }

            @Override
            public boolean containsAll(@NotNull Collection<?> c) {
                for (var v : c) {
                    if (!contains(v)) return false;
                }
                return true;
            }

            @Override
            public boolean addAll(@NotNull Collection<? extends K> c) {
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean retainAll(@NotNull Collection<?> c) {
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean removeAll(@NotNull Collection<?> c) {
                var res = true;
                for (var e : c) {
                    res &= remove(e);
                }
                return res;
            }

            @Override
            public void clear() {
                body.clear();
            }
        };
    }

    @Override
    public @NotNull Collection<V> values() {
        return Flow.flow(entrySet()).map(Entry::getValue).toList();
    }

    @Override
    public @NotNull Set<Entry<K, V>> entrySet() {
        return new Set<>() {
            @Override
            public int size() {
                return body.size();
            }

            @Override
            public boolean isEmpty() {
                return body.isEmpty();
            }

            @Override
            public boolean contains(Object o) {
                if (!(o instanceof Entry)) return false;
                return WeakIdenticalConcurrentHashMap.this.containsKey(((Entry<?, ?>) o).getKey());
            }

            @Override
            public @NotNull Iterator<Entry<K, V>> iterator() {
                var it = body.entrySet().iterator();
                return new Iterator<Entry<K, V>>() {
                    Entry<K, V> last;

                    @Override
                    public boolean hasNext() {
                        while (last == null && it.hasNext()) {
                            var e = it.next();
                            var k = e.getKey().get();
                            if (k != null) {
                                last = new Entry<>() {
                                    @Override
                                    public K getKey() {
                                        return k;
                                    }

                                    @Override
                                    public V getValue() {
                                        return unmask(e.getValue());
                                    }

                                    @Override
                                    public V setValue(V value) {
                                        return unmask(e.setValue(new Mask<>(value)));
                                    }
                                };
                            }
                        }
                        return last != null;
                    }

                    @Override
                    public Entry<K, V> next() {
                        var l = last;
                        last = null;
                        return l;
                    }
                };
            }

            @Override
            public @NotNull Object @NotNull [] toArray() {
                return Flow.flow(iterator()).toArray();
            }

            @Override
            public @NotNull <T> T @NotNull [] toArray(@NotNull T @NotNull [] a) {
                return Flow.flow(iterator()).toList().toArray(a);
            }

            @Override
            public boolean add(Entry<K, V> entry) {
                return !Objects.equals(WeakIdenticalConcurrentHashMap.this.put(entry.getKey(), entry.getValue()), entry.getValue());
            }

            @Override
            public boolean remove(Object o) {
                if (!(o instanceof Entry)) return false;
                var e = RU.<Entry<K, V>>cast(o);
                return WeakIdenticalConcurrentHashMap.this.remove(e.getKey(), e.getValue());
            }

            @Override
            public boolean containsAll(@NotNull Collection<?> c) {
                for (var e : c) {
                    if (!contains(e)) return false;
                }
                return true;
            }

            @Override
            public boolean addAll(@NotNull Collection<? extends Entry<K, V>> c) {
                var res = true;
                for (var e : c) {
                    res &= add(e);
                }
                return res;
            }

            @Override
            public boolean retainAll(@NotNull Collection<?> c) {
                throw new UnsupportedOperationException();
            }

            @Override
            public boolean removeAll(@NotNull Collection<?> c) {
                var res = true;
                for (var e : c) {
                    res &= remove(e);
                }
                return res;
            }

            @Override
            public void clear() {
                body.clear();
            }
        };
    }

    private static <E> E unmask(MaskI<E> r) {
        if (r == null) return null;
        return r.get();
    }

    private interface MaskI<K> {
        K get();
    }

    private static class MaskSample<K> implements MaskI<K> {
        K val;
        int hash;

        public MaskSample(K val) {
            hash = val.hashCode();
            this.val = val;
        }

        @Override
        public int hashCode() {
            return hash;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == this) return true;
            var k = RU.<Mask<K>>cast(obj);
            return get() == k.get();
        }

        @Override
        public K get() {
            return val;
        }
    }

    private static class Mask<K> extends WeakReference<K> implements MaskI<K> {
        final int hash;

        public Mask(K referent, ReferenceQueue<K> referenceQueue) {
            super(referent, referenceQueue);
            hash = referent.hashCode();
        }

        public Mask(K referent) {
            super(referent);
            hash = referent.hashCode();
        }

        @Override
        public int hashCode() {
            return hash;
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == this) return true;
            var k = RU.<Mask<K>>cast(obj);
            return get() == k.get();
        }
    }
}
