package io.aether.utils.dataio;


import io.aether.utils.HexUtils;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;

public class ByteBufferDataIO implements DataIO{
    private final ByteBuffer buffer;
    private boolean isWriting; // Флаг, обозначающий текущий режим работы

    public ByteBufferDataIO(int capacity) {
        this.buffer = ByteBuffer.allocate(capacity).order(ByteOrder.LITTLE_ENDIAN);
        this.isWriting = true; // Начальный режим - запись
    }

    @Override
    public int getSizeForWrite() {
        return buffer.remaining();
    }

    @Override
    public boolean isWritable() {
        return buffer.hasRemaining();
    }

    @Override
    public void clear() {
        buffer.clear();
        this.isWriting = true; // После очистки возвращаемся к режиму записи
    }

    @Override
    public int getSizeForRead() {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        return buffer.position();
    }

    private void flipBuffer() {
        buffer.flip();
        isWriting = !isWriting; // Инвертируем флаг после переключения режима
    }

    @Override
    public int read(byte[] b, int offset, int len) {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        var remaining = buffer.remaining();
        int l = Math.min(Math.min(len, b.length), remaining);
        if (l > 0) {
            buffer.get(b, offset, l);
        }
        return l; // Return the number of bytes actually read
    }

    @Override
    public void write(byte [] b) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        if (b.length > buffer.remaining()) {
            throw new IllegalStateException();
        }
        buffer.put(b);
    }

    @Override
    public int write(byte [] b, int off, int len) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        var l = Math.min(len, buffer.remaining());
        buffer.put(b, off, l);
        return l;
    }

    @Override
    public void writeShort(int v) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        if (buffer.remaining() < 2) throw new IllegalStateException();
        buffer.putShort((short) v);
    }

    @Override
    public short readShort() {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        if (buffer.position() + 2 > buffer.limit()) throw new RuntimeException();
        return buffer.getShort();
    }

    @Override
    public void writeInt(int v) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        if (buffer.remaining() < 4) throw new IllegalStateException();
        buffer.putInt(v);
    }

    @Override
    public int readInt() {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        if (buffer.position() + 4 > buffer.limit()) throw new RuntimeException();
        return buffer.getInt();
    }

    @Override
    public void writeLong(long v) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        if (buffer.remaining() < 8) throw new IllegalStateException();
        buffer.putLong(v);
    }

    @Override
    public long readLong() {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        if (buffer.position() + 8 > buffer.limit()) throw new RuntimeException();
        return buffer.getLong();
    }

    @Override
    public void writeFloat(float v) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        if (buffer.remaining() < 4) throw new IllegalStateException();
        buffer.putFloat(v);
    }

    @Override
    public float readFloat() {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        if (buffer.position() + 4 > buffer.limit()) throw new RuntimeException();
        return buffer.getFloat();
    }

    @Override
    public void writeDouble(double v) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        if (buffer.remaining() < 8) throw new IllegalStateException();
        buffer.putDouble(v);
    }

    @Override
    public double readDouble() {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        if (buffer.position() + 8 > buffer.limit()) throw new RuntimeException();
        return buffer.getDouble();
    }

    @Override
    public void writeBoolean(boolean v) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        writeByte(v ? 1 : 0);
    }

    @Override
    public boolean readBoolean() {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        return readUByte() != 0;
    }

    @Override
    public byte[] toArray() {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        if (buffer.position() == 0 && buffer.limit() == buffer.capacity()) return buffer.array();
        var array = new byte[buffer.position()];
        System.arraycopy(buffer.array(), 0, array, 0, buffer.position());
        return array;
    }

    @Override
    public void skipBytes(int n) {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        if (n > buffer.remaining())
            throw new IllegalStateException("Not enough bytes to skip");
        buffer.position(buffer.position() + n);
    }

    @Override
    public boolean isEmpty() {
        return getSizeForRead() == 0;
    }

    @Override
    public void skipAllBytes() {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        buffer.position(buffer.limit());
    }

    @Override
    public int indexOf(int limit, byte val) {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        for (int i = buffer.position(); i < Math.min(limit, buffer.limit()); i++) {
            if (buffer.get(i) == val) return i;
        }
        return -1;
    }

    @Override
    public DataIO readSubData(int length) {
        if (isWriting) {
            flipBuffer(); // Переключение на режим чтения, если необходимо
        }
        var res = new ByteBufferDataIO(length);
        for (int i = 0; i < length && buffer.hasRemaining(); i++) {
            res.writeByte(buffer.get());
        }
        return res;
    }

    @Override
    public void writeHexBytes(String hex) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        byte[] bytes = HexUtils.hexToBytes(hex);
        if (bytes.length > buffer.remaining()) throw new IllegalStateException();
        buffer.put(bytes);
    }

    @Override
    public void write(DataInOut data) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        var r = write(data.data, data.readPos, data.writePos - data.readPos);
        assert r == data.getSizeForRead();
        data.readPos = 0;
        data.writePos = 0;
    }

    @Override
    public void write(DataInOutStatic data) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        var r = write(data.getData(), data.getReadPos(), data.getWritePos() - data.getReadPos());
        assert r == data.getSizeForRead();
        data.setReadPos(0);
        data.setWritePos(0);
    }

    @Override
    public void write(DataIn data) {
        if (!isWriting) {
            flipBuffer(); // Переключение на режим записи, если необходимо
        }
        write(data.toArray());
    }

    public int readUByte() {
        return buffer.get();
    }

    public void writeByte(int b) {
        buffer.put((byte) (b & 0xFF));
    }
}