package io.aether.utils.dataio;


import io.aether.utils.HexUtils;
import org.jetbrains.annotations.NotNull;

import java.util.Arrays;

public class DataInOut implements DataIO {
    public byte[] data;
    public int writePos;
    public int readPos;

    public DataInOut(byte[] data, int len) {
        this.data = data;
        writePos = len;
    }

    @Override
    public byte[] toArray() {
        if (data.length == getSizeForRead()) {
            var d = data;
            data = new byte[d.length];
            clear();
            return d;
        }else{
            var r = Arrays.copyOfRange(data, readPos, writePos);
            clear();
            return r;
        }
    }

    public DataInOut() {
        this(100);
    }

    public DataInOut(int initSize) {
        data = new byte[initSize];
    }

    public DataInOut(byte[] data) {
        this.data = data;
        writePos = data.length;
    }

    @Override
    public String toString() {
        return HexUtils.toHexString(data, readPos, writePos);
    }

    @Override
    public void write(DataIn data) {
        var needEnd = writePos + data.getSizeForRead();
        if (needEnd >= this.data.length) {
            this.data = Arrays.copyOf(this.data, needEnd * 2);
        }
        data.read(this.data, writePos, data.getSizeForRead());
        writePos = needEnd;
    }

    public byte[] getData() {
        return data;
    }

    public void setData(byte[] data) {
        this.data = data;
        writePos = data.length;
        readPos = 0;
    }

    public int getWritePos() {
        return writePos;
    }

    public void setWritePos(int writePos) {
        this.writePos = writePos;
    }

    public int getReadPos() {
        return readPos;
    }

    public void setReadPos(int readPos) {
        this.readPos = readPos;
    }

    public void clear() {
        writePos = 0;
        readPos = 0;
    }

    @Override
    public int readInt() {
        int res = 0;
        for (int i = 0; i < 4; i++) {
            res = res | (readUByte0() << (8 * i));
        }
        trim();
        return res;
    }

    @Override
    public long readLong() {
        long res = 0;
        for (int i = 0; i < 8; ++i) {
            res = res | (((long) readUByte0()) << (8 * i));
        }
        trim();
        return res;
    }

    @Override
    public int readUShort() {
        int res = 0;
        for (int i = 0; i < 2; i++) {
            res = res | ((readUByte0() & 255) << (8 * i));
        }
        trim();
        return res;
    }

    public byte[] toArrayCopy() {
        return Arrays.copyOfRange(data, readPos, writePos);
    }

    @Override
    public int getSizeForRead() {
        return writePos - readPos;
    }

    @Override
    public int read(byte[] b, int offset, int len) {
        var a = getSizeForRead();
        int l = Math.min(Math.min(len, b.length), a);
        if (l > 0) {
            System.arraycopy(data, readPos, b, offset, l);
            readPos += l;
        }
        trim();
        return l;
    }

    public void trim() {
        if (writePos == readPos) {
            writePos = 0;
            readPos = 0;
        }
    }

    private int readUByte0() {
        return Byte.toUnsignedInt(data[readPos++]);
    }

    @Override
    public void skipAllBytes() {
        writePos = 0;
        readPos = 0;
    }

    @Override
    public int indexOf(int limit, byte val) {
        for (int i = readPos; i < writePos; i++) {
            if (data[i] == val) return i;
        }
        return -1;
    }

    @Override
    public void skipBytes(int n) {
        readPos += n;
        assert readPos <= writePos;
        trim();
    }

    @Override
    public void write(byte @NotNull [] b) {
        checkSize(b.length);
        System.arraycopy(b, 0, data, writePos, b.length);
        writePos += b.length;
    }

    @Override
    public int write(byte @NotNull [] b, int off, int len) {
        checkSize(len);
        System.arraycopy(b, off, data, writePos, len);
        writePos += len;
        return len;
    }

    @Override
    public int readUByte() {
        assert readPos < writePos;
        var res = readUByte0();
        trim();
        return res;
    }

    @Override
    public void writeByte(int v) {
        if (writePos >= data.length) {
            data = Arrays.copyOf(data, data.length * 2);
        }
        data[writePos++] = (byte) v;
    }

    public void checkSize(int size) {
        if (data.length - writePos < size) {
            data = Arrays.copyOf(data, (int) ((writePos + size) * 1.3));
        }
    }
}
