package io.aether.utils.flow;

import io.aether.utils.AString;
import io.aether.utils.RU;
import io.aether.utils.interfaces.*;
import it.unimi.dsi.fastutil.ints.*;
import it.unimi.dsi.fastutil.longs.*;
import it.unimi.dsi.fastutil.objects.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Array;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * A fluent interface for processing sequences of elements supporting chained method calls.
 * Combines features of {@link Iterator} and {@link Iterable} with additional stream-like operations.
 *
 * @param <T> the type of elements in this flow
 */
public interface Flow<T> extends Iterator<T>, Iterable<T> {
    /** Shared random instance used for operations like shuffle() and random() */
    Random RANDOM = new Random();

    /**
     * Returns an iterator over elements of type {@code T}.
     * @return an Iterator
     */
    @NotNull
    @Override
    default Iterator<T> iterator() {
        return this;
    }

    /**
     * Returns whether any elements of this flow match the provided predicate.
     * @param p the predicate to apply to elements
     * @return true if any elements match the predicate, false otherwise
     */
    default boolean anyMatch(APredicate<T> p) {
        while (hasNext()) {
            if (p.test(next())) return true;
        }
        return false;
    }

    /**
     * Returns whether no elements of this flow match the provided predicate.
     * @param p the predicate to apply to elements
     * @return true if no elements match the predicate, false otherwise
     */
    default boolean noneMatch(APredicate<T> p) {
        while (hasNext()) {
            if (p.test(next())) return false;
        }
        return true;
    }

    /**
     * Appends an element to the end of this flow.
     * @param value the element to append
     * @return a new flow containing all elements of this flow plus the appended element
     */
    default Flow<T> add(T value) {
        final var oit = this;
        return new Flow<>() {
            boolean parent = true;
            private boolean flag = true;

            @Override
            public boolean hasNext() {
                if (parent && oit.hasNext()) {
                    return true;
                }
                parent = false;
                return flag;
            }

            @Override
            public T next() {
                if (parent) {
                    return oit.next();
                }
                flag = false;
                return value;
            }
        };
    }

    /**
     * Returns whether no elements of this flow are equal to the given object.
     * @param p the object to compare against
     * @return true if no elements equal the object, false otherwise
     */
    default boolean noneMatchObj(Object p) {
        while (hasNext()) {
            if (Objects.equals(p, next())) return false;
        }
        return true;
    }

    /**
     * Returns whether any element of this flow equals the given object.
     * @param p the object to compare against
     * @return true if any element equals the object, false otherwise
     */
    default boolean anyMatchObj(T p) {
        while (hasNext()) {
            if (Objects.equals(p, next())) return true;
        }
        return false;
    }

    /**
     * Returns the minimum element according to the provided comparable function.
     * @param comparable function that extracts the comparable key
     * @return Optional containing the minimum element, or empty if flow is empty
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    default Optional<T> min(AFunction<T, Comparable> comparable) {
        return min((a, b) -> comparable.apply(a).compareTo(comparable.apply(b)));
    }

    /**
     * Returns whether all elements of this flow match the provided predicate.
     * @param p the predicate to apply to elements
     * @return true if all elements match the predicate or flow is empty, false otherwise
     */
    default boolean allMatch(APredicate<T> p) {
        while (hasNext()) {
            if (!p.test(next())) return false;
        }
        return true;
    }

    /**
     * Maps elements to their string representations.
     * @return a new flow of string elements
     */
    default Flow<String> mapToString() {
        return map(Object::toString);
    }

    /**
     * Collects elements into an array of the specified type.
     * @param arrayType the component type of the array
     * @return an array containing all elements of this flow
     */
    @NotNull
    default T[] toArray(@NotNull Class<? super T> arrayType) {
        T[] res = RU.cast(Array.newInstance(arrayType, 10));
        int i = 0;
        while (hasNext()) {
            if (i == res.length) {
                res = Arrays.copyOf(res, (int) (res.length * 1.5));
            }
            res[i++] = next();
        }
        return Arrays.copyOf(res, i);
    }

    /**
     * Sorts elements according to the provided comparator.
     * @param comparator the comparator to determine element order
     * @return a new sorted flow
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    @NotNull
    default Flow<T> sort(@NotNull Comparator<T> comparator) {
        Object[] arr = toArray();
        Arrays.sort(arr, (Comparator<Object>) comparator);
        return (Flow) flow(arr);
    }

    /**
     * Maps each element to an iterable and flattens the results.
     * @param f function that maps elements to iterables
     * @param <E> the type of elements in the resulting flow
     * @return a new flow of flattened elements
     */
    @NotNull
    default <E> Flow<E> flatMap1(@NotNull AFunctionO2Iterable<T, E> f) {
        return flatMap((e, c) -> {
            var ii = f.apply(e);
            if (ii != null) {
                for (var v : ii) {
                    c.accept(v);
                }
            }
        });
    }

    /**
     * Maps non-null elements to their runtime class.
     * @return a new flow of class objects
     */
    @NotNull
    @SuppressWarnings("unchecked")
    default Flow<Class<? extends T>> mapToType() {
        return filter(Objects::nonNull).map(v -> (Class<? extends T>) v.getClass());
    }

    /**
     * Casts elements to the specified type.
     * @param t the target type class object
     * @param <E> the target type
     * @return a flow of casted elements
     */
    @SuppressWarnings("unchecked")
    default <E> Flow<E> mapToType(Class<E> t) {
        return (Flow<E>) this;
    }

    /**
     * Maps elements using the provided function.
     * @param f the mapping function
     * @param <E> the type of elements in the resulting flow
     * @return a new flow of mapped elements
     */
    @NotNull
    default <E> Flow<E> map(@NotNull AFunction<T, E> f) {
        final var self = this;
        return new Flow<>() {
            @Override
            public boolean hasNext() {
                return self.hasNext();
            }

            @Override
            public E next() {
                return f.apply(self.next());
            }
        };
    }

    /**
     * Executes an action when the flow is exhausted.
     * @param t the action to execute
     * @return a new flow with the done handler
     */
    default Flow<T> onDone(Runnable t) {
        var self = this;
        return new Flow<>() {
            @Override
            public boolean hasNext() {
                return self.hasNext();
            }

            @Override
            public T next() {
                var res = self.next();
                if (!hasNext()) {
                    t.run();
                }
                return res;
            }
        };
    }

    /**
     * Returns a random element from the flow.
     * @return a random element or null if flow is empty
     */
    default T random() {
        var l = toList();
        if (l.isEmpty()) return null;
        return l.get(RANDOM.nextInt(l.size()));
    }

    /**
     * Maps elements to long values.
     * @param f the mapping function
     * @return a new flow of long values
     */
    @NotNull
    default FlowLong mapToLong(@NotNull AFunctionO2L<T> f) {
        final var self = this;
        return new FlowLong() {
            @Override
            public boolean hasNext() {
                return self.hasNext();
            }

            @Override
            public long nextLong() {
                return f.apply(self.next());
            }
        };
    }

    /**
     * Splits elements into three collections based on comparator results.
     * @param ifTrue collection for elements with positive comparison
     * @param ifFalse collection for elements with negative comparison
     * @param ifCenter collection for elements with zero comparison
     * @param p the splitting comparator
     */
    default void split(Collection<T> ifTrue, Collection<T> ifFalse, Collection<T> ifCenter, SplitComparator<T> p) {
        while (hasNext()) {
            var v = next();
            var i = p.test(v);
            if (i > 0) {
                ifTrue.add(v);
            } else if (i < 0) {
                ifFalse.add(v);
            } else {
                ifCenter.add(v);
            }
        }
    }

    /**
     * Maps elements to boolean values.
     * @param f the mapping function
     * @return a new flow of boolean values
     */
    @NotNull
    default FlowBoolean mapToBoolean(@NotNull AFunctionO2Bool<T> f) {
        final var self = this;
        return new FlowBoolean() {
            @Override
            public boolean hasNext() {
                return self.hasNext();
            }

            @Override
            public boolean nextBoolean() {
                return f.apply(self.next());
            }
        };
    }

    /**
     * Maps elements to integer values.
     * @param f the mapping function
     * @return a new flow of integer values
     */
    @NotNull
    default FlowInt mapToInt(@NotNull AFunctionO2I<T> f) {
        final var self = this;
        return new FlowInt() {
            @Override
            public boolean hasNext() {
                return self.hasNext();
            }

            @Override
            public int nextInt() {
                return f.apply(self.next());
            }
        };
    }

    @NotNull
    default FlowShort mapToShort(@NotNull AFunctionO2S<T> f) {
        final var self = this;
        return new FlowShort() {
            @Override
            public boolean hasNext() {
                return self.hasNext();
            }

            @Override
            public short nextShort() {
                return f.apply(self.next());
            }
        };
    }

    /**
     * Applies an action to each element while passing through the elements.
     * @param c the action to apply
     * @return a new flow with the applied action
     */
    @NotNull
    default Flow<T> apply(AConsumer<T> c) {
        final var self = this;
        return new Flow<>() {
            @Override
            public boolean hasNext() {
                return self.hasNext();
            }

            @Override
            public T next() {
                var v = self.next();
                c.accept(v);
                return v;
            }
        };
    }

    /**
     * Filters elements by type.
     * @param predicate the type to filter by
     * @param <E> the target type
     * @return a new flow containing only elements of the specified type
     */
    @SuppressWarnings("unchecked")
    @NotNull
    default <E> Flow<E> filterByType(Class<E> predicate) {
        return (Flow<E>) filter(predicate::isInstance);
    }

    /**
     * Filters out elements of the specified type.
     * @param predicate the type to exclude
     * @return a new flow without elements of the specified type
     */
    @NotNull
    default Flow<T> filterNotType(Class<?> predicate) {
        return filterNot(predicate::isInstance);
    }

    /**
     * Appends multiple elements to the end of this flow.
     * @param values the elements to append
     * @param <E> the type of elements to append
     * @return a new flow with appended elements
     */
    @SuppressWarnings("unchecked")
    default <E extends T> Flow<T> addAllEls(E... values) {
        final var oit = this;
        return new Flow<>() {
            boolean parent = true;
            private int index;

            @Override
            public boolean hasNext() {
                if (parent && oit.hasNext()) {
                    return true;
                }
                parent = false;
                return index < values.length;
            }

            @Override
            public T next() {
                if (parent) {
                    return oit.next();
                }
                return values[index++];
            }
        };
    }

    /**
     * Appends all elements from another flow.
     * @param values the flow to append
     * @return a new flow combining elements from both flows
     */
    default Flow<T> addAll(Flow<? extends T> values) {
        return addAll((Iterator<? extends T>) values);
    }

    /**
     * Appends all elements from an iterator.
     * @param values the iterator to append
     * @return a new flow combining elements from both sources
     */
    default Flow<T> addAll(Iterator<? extends T> values) {
        if (!values.hasNext()) return this;
        final var oit = this;
        return new Flow<>() {
            boolean parent = true;

            @Override
            public boolean hasNext() {
                if (parent && oit.hasNext()) {
                    return true;
                }
                parent = false;
                return values.hasNext();
            }

            @Override
            public T next() {
                if (parent) {
                    return oit.next();
                }
                return values.next();
            }
        };
    }

    /**
     * Appends all elements from an iterable.
     * @param values the iterable to append
     * @return a new flow combining elements from both sources
     */
    default Flow<T> addAll(Iterable<? extends T> values) {
        return addAll(values.iterator());
    }

    /**
     * Maps each element to an array and flattens the results.
     * @param f function that maps elements to arrays
     * @param <E> the type of elements in the resulting flow
     * @return a new flow of flattened elements
     */
    @NotNull
    default <E> Flow<E> flatMap2(@NotNull AFunctionO2Array<T, E> f) {
        return flatMap((e, c) -> {
            var ii = f.apply(e);
            if (ii != null) {
                for (var v : ii) {
                    c.accept(v);
                }
            }
        });
    }

    /**
     * Maps each element to a stream and flattens the results.
     * @param f function that maps elements to streams
     * @param <E> the type of elements in the resulting flow
     * @return a new flow of flattened elements
     */
    @NotNull
    default <E> Flow<E> flatMap3(@NotNull AFunctionO2Stream<T, E> f) {
        var self = this;
        return new Flow<>() {
            Flow<E> cur;

            @Override
            public boolean hasNext() {
                while (cur == null || !cur.hasNext()) {
                    if (cur != null) {
                        cur = null;
                    }
                    if (!self.hasNext()) return false;
                    cur = f.apply(self.next());
                }
                return true;
            }

            @Override
            public E next() {
                return cur.next();
            }
        };
    }

    /**
     * Sorts elements according to the provided comparator.
     * @param comparator the comparator to determine element order
     * @return a new sorted flow
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    @NotNull
    default Flow<T> sort(@NotNull AComparator<T> comparator) {
        Object[] arr = toArray();
        Arrays.sort(arr, (Comparator<Object>) comparator);
        return (Flow) flow(arr);
    }

    /**
     * Consumes all elements in the flow.
     */
    default void run() {
        while (hasNext()) {
            next();
        }
    }

    /**
     * Performs an action for each element in the flow.
     * @param c the action to perform
     */
    default void foreach(AConsumer<T> c) {
        while (hasNext()) {
            c.accept(next());
        }
    }

    /**
     * Handles errors by applying a fallback function.
     * @param f the error handling function
     * @return a new flow with error handling
     */
    default Flow<T> ifError(AFunction<Throwable, T> f) {
        var self = this;
        return new Flow<>() {
            @Override
            public boolean hasNext() {
                return self.hasNext();
            }

            @Override
            public T next() {
                try {
                    return self.next();
                } catch (Throwable e) {
                    return f.apply(e);
                }
            }
        };
    }

    /**
     * Maps each element to an iterator and flattens the results.
     * @param f function that maps elements to iterators
     * @param <E> the type of elements in the resulting flow
     * @return a new flow of flattened elements
     */
    @NotNull
    default <E> Flow<E> flatMap4(@NotNull AFunctionO2Iterator<T, E> f) {
        return flatMap((e, c) -> {
            var ii = f.apply(e);
            if (ii != null) {
                while (ii.hasNext()) {
                    c.accept(ii.next());
                }
            }
        });
    }

    /**
     * Reverses the order of elements in the flow.
     * @return a new flow with elements in reverse order
     */
    @SuppressWarnings("unchecked")
    default Flow<T> reverse() {
        return (Flow<T>) flow(toArray());
    }

    /**
     * Maps each element to multiple elements using a consumer-based approach.
     * @param f the flat mapping function
     * @param <E> the type of elements in the resulting flow
     * @return a new flow of flattened elements
     */
    @NotNull
    default <E> Flow<E> flatMap(@NotNull ABiConsumer<T, AConsumer<E>> f) {
        final var oit = this;
        ObjectArrayList<E> list = new ObjectArrayList<>();
        final AConsumer<E> cc = list::add;
        return new Flow<>() {
            int pos;

            @Override
            public boolean hasNext() {
                while (list.isEmpty()) {
                    if (!oit.hasNext()) return false;
                    f.accept(oit.next(), cc);
                }
                return true;
            }

            @Override
            public E next() {
                var res = list.get(pos++);
                if (pos == list.size()) {
                    pos = 0;
                    list.clear();
                }
                return res;
            }
        };
    }

    /**
     * Collects elements into a set and applies an action to it.
     * @param c the action to apply to the set
     * @return the original flow
     */
    @NotNull
    default Flow<T> setTo(AConsumer<Set<T>> c) {
        var set = toSet();
        c.accept(set);
        return this;
    }

    /**
     * Collects elements into an object array.
     * @return an array containing all elements
     */
    @NotNull
    default Object[] toArray() {
        Object[] res = new Object[10];
        int i = 0;
        while (hasNext()) {
            if (i == res.length) {
                res = Arrays.copyOf(res, (int) (res.length * 1.5));
            }
            res[i++] = next();
        }
        return Arrays.copyOf(res, i);
    }

    /**
     * Collects elements into a collection.
     * @param collection the target collection
     * @param <E> the collection type
     * @return the populated collection
     */
    default <E extends Collection<T>> E toCollection(@NotNull E collection) {
        while (hasNext()) {
            collection.add(next());
        }
        return collection;
    }

    /**
     * Removes elements from a collection.
     * @param collection the collection to remove from
     * @param <E> the collection type
     * @return the modified collection
     */
    default <E extends Collection<T>> E removeFromCollection(@NotNull E collection) {
        while (hasNext()) {
            collection.remove(next());
        }
        return collection;
    }

    /**
     * Filters elements based on a predicate.
     * @param predicate the filtering predicate
     * @return a new flow containing only matching elements
     */
    @NotNull
    default Flow<T> filter(@Nullable APredicate<T> predicate) {
        if (predicate == null) return this;
        var oit = this;
        return new Flow<>() {
            T last;
            boolean hasNext;

            @Override
            public boolean hasNext() {
                if (hasNext) return true;
                while (oit.hasNext()) {
                    last = oit.next();
                    if (predicate.test(last)) {
                        hasNext = true;
                        return true;
                    }
                }
                return false;
            }

            @Override
            public T next() {
                assert hasNext;
                hasNext = false;
                return last;
            }
        };
    }

    /**
     * Limits the flow to a maximum number of elements.
     * @param i the maximum number of elements
     * @return a new truncated flow
     */
    default Flow<T> limit(int i) {
        var self = this;
        return new Flow<>() {
            int cc;

            @Override
            public boolean hasNext() {
                return cc < i && self.hasNext();
            }

            @Override
            public T next() {
                cc++;
                return self.next();
            }
        };
    }

    /**
     * Filters out null elements.
     * @return a new flow without null elements
     */
    @NotNull
    default Flow<T> filterNotNull() {
        return filter(Objects::nonNull);
    }

    /**
     * Filters out elements equal to the specified value.
     * @param value the value to exclude
     * @return a new flow without matching elements
     */
    @NotNull
    default Flow<T> filterExclude(T value) {
        return filterNot(v -> Objects.equals(v, value));
    }

    /**
     * Filters elements that don't match the predicate.
     * @param predicate the filtering predicate
     * @return a new flow containing only non-matching elements
     */
    @NotNull
    default Flow<T> filterNot(@Nullable APredicate<T> predicate) {
        if (predicate == null) return this;
        var oit = this;
        return new Flow<>() {
            T last;
            boolean hasNext;

            @Override
            public boolean hasNext() {
                if (hasNext) return true;
                while (oit.hasNext()) {
                    last = oit.next();
                    if (!predicate.test(last)) {
                        hasNext = true;
                        return true;
                    }
                }
                return false;
            }

            @Override
            public T next() {
                assert hasNext;
                hasNext = false;
                return last;
            }
        };
    }

    /**
     * Executes an action if the flow is empty.
     * @param task the action to execute
     * @return a new flow with the empty handler
     */
    @NotNull
    default Flow<T> ifEmpty(ARunnable task) {
        if (task == null) return this;
        var oit = this;
        return new Flow<>() {
            boolean first = true;

            @Override
            public boolean hasNext() {
                var res = oit.hasNext();
                if (!res && first) {
                    task.run();
                }
                first = false;
                return res;
            }

            @Override
            public T next() {
                return oit.next();
            }
        };
    }

    /**
     * Joins elements into a StringBuilder with comma delimiters.
     * @param sb the StringBuilder to append to
     */
    default void joinD(StringBuilder sb) {
        join(AString.of(sb), ", ");
    }

    /**
     * Joins elements into an AString with comma delimiters.
     * @param sb the AString to append to
     */
    default void joinD(AString sb) {
        join(sb, ", ");
    }

    /**
     * Casts elements to the specified type.
     * @param type the target type class
     * @param <E> the target type
     * @return a flow of casted elements
     */
    @SuppressWarnings("unchecked")
    default <E> Flow<E> cast(@NotNull Class<E> type) {
        return (Flow<E>) this;
    }

    /**
     * Throws an exception if any element matches the condition.
     * @param conditions the condition predicate
     * @return a flow with error checking
     */
    default Flow<T> errorIf(APredicate<T> conditions) {
        return apply(v -> {
            if (conditions.test(v)) {
                throw new RuntimeException(String.valueOf(v));
            }
        });
    }

    /**
     * Throws the specified exception if the flow is empty.
     * @param error the exception to throw
     * @return a flow with empty checking
     */
    @NotNull
    default Flow<T> ifEmpty(Exception error) {
        return ifEmpty(() -> {
            throw error;
        });
    }

    /**
     * Randomly shuffles the elements in the flow.
     * @return a new flow with shuffled elements
     */
    @NotNull
    default Flow<T> shuffle() {
        var l = toList();
        Collections.shuffle(l);
        return flow(l);
    }

    /**
     * Ignores exceptions of the specified type.
     * @param ee the exception type to ignore
     * @return a flow with exception handling
     */
    @NotNull
    default Flow<T> ignoreError(Class<? extends Exception> ee) {
        if (ee == null) return this;
        var oit = this;
        return new Flow<>() {
            T last;
            boolean hasNext;

            @Override
            public boolean hasNext() {
                if (hasNext) return true;
                while (oit.hasNext()) {
                    try {
                        last = oit.next();
                        hasNext = true;
                        return true;
                    } catch (Exception ex) {
                        if (!ee.isInstance(ex)) {
                            RU.error(ex);
                        }
                    }
                }
                return false;
            }

            @Override
            public T next() {
                assert hasNext;
                hasNext = false;
                return last;
            }
        };
    }

    /**
     * Collects elements into a list.
     * @return a new list containing all elements
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    @NotNull
    default List<T> toList() {
        return new ObjectArrayList(toArray());
    }

    /**
     * Converts the flow to an iterable.
     * @return an iterable view of the flow
     */
    @NotNull
    default Iterable<T> toIterable() {
        return () -> this;
    }

    /**
     * Collects elements into an immutable list.
     * @return an immutable list containing all elements
     */
    @NotNull
    @SuppressWarnings({"unchecked"})
    default ObjectImmutableList<T> toImmutableList() {
        return (ObjectImmutableList<T>) ObjectImmutableList.of(toArray());
    }

    /**
     * Collects elements into a set.
     * @return a new set containing all elements
     */
    @NotNull
    @SuppressWarnings({"unchecked", "rawtypes"})
    default Set<T> toSet() {
        return new ObjectOpenHashSet(toArray());
    }

    /**
     * Returns whether the flow contains any elements.
     * @return true if flow has elements, false otherwise
     */
    default boolean isNotEmpty() {
        return hasNext();
    }

    /**
     * Returns the maximum element according to the provided comparable function.
     * @param comparable function that extracts the comparable key
     * @return Optional containing the maximum element, or empty if flow is empty
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    default Optional<T> max(AFunction<T, Comparable> comparable) {
        return max((a, b) -> comparable.apply(a).compareTo(comparable.apply(b)));
    }

    /**
     * Performs an action for each element with an additional argument.
     * @param arg the additional argument
     * @param consumer the action to perform
     * @param <T2> the type of the additional argument
     */
    default <T2> void to(T2 arg, ABiConsumer<T2, T> consumer) {
        while (hasNext()) {
            consumer.accept(arg, next());
        }
    }

    /**
     * Performs an action for each element.
     * @param consumer the action to perform
     */
    default void to(AConsumer<T> consumer) {
        while (hasNext()) {
            consumer.accept(next());
        }
    }

    /**
     * Converts the flow to a supplier.
     * @return a supplier that provides elements sequentially
     */
    @NotNull
    default Supplier<T> toSupplier() {
        return () -> {
            if (hasNext()) {
                return next();
            }
            return null;
        };
    }

    /**
     * Collects elements into a map using a key extraction function.
     * @param keyFactory function to extract map keys from elements
     * @param <K> the key type
     * @return a new map with elements as values
     */
    @NotNull
    default <K> Map<K, T> toMapExtractKey(AFunction<T, K> keyFactory) {
        return toMapExtractKey(new Object2ObjectOpenHashMap<>(), keyFactory);
    }

    /**
     * Collects elements into a concurrent map using a key extraction function.
     * @param keyFactory function to extract map keys from elements
     * @param <K> the key type
     * @return a new concurrent map with elements as values
     */
    @NotNull
    default <K> Map<K, T> toCMapExtractKey(AFunction<T, K> keyFactory) {
        return toMapExtractKey(new ConcurrentHashMap<>(), keyFactory);
    }

    /**
     * Collects elements into a map using a key extraction function.
     * @param map the target map
     * @param keyFactory function to extract map keys from elements
     * @param <K> the key type
     * @return the populated map
     */
    @NotNull
    default <K> Map<K, T> toMapExtractKey(Map<K, T> map, AFunction<T, K> keyFactory) {
        while (hasNext()) {
            var val = next();
            map.put(keyFactory.apply(val), val);
        }
        return map;
    }

    /**
     * Collects elements into a map using key and value extraction functions.
     * @param keyFactory function to extract map keys from elements
     * @param valFactory function to extract map values from elements
     * @param <K> the key type
     * @param <V> the value type
     * @return a new map with extracted keys and values
     */
    @NotNull
    default <K, V> Map<K, V> toMap(AFunction<T, K> keyFactory, AFunction<T, V> valFactory) {
        return toMap(new Object2ObjectOpenHashMap<>(), keyFactory, valFactory);
    }

    /**
     * Collects elements into a map using key and value extraction functions.
     * @param map the target map
     * @param keyFactory function to extract map keys from elements
     * @param valFactory function to extract map values from elements
     * @param <K> the key type
     * @param <V> the value type
     * @return the populated map
     */
    @NotNull
    default <K, V> Map<K, V> toMap(Map<K, V> map, AFunction<T, K> keyFactory, AFunction<T, V> valFactory) {
        while (hasNext()) {
            var val = next();
            map.put(keyFactory.apply(val), valFactory.apply(val));
        }
        return map;
    }

    /**
     * Collects elements into an int-to-object map.
     * @param keyFactory function to extract integer keys from elements
     * @param valFactory function to extract values from elements
     * @param <V> the value type
     * @return a new int-to-object map
     */
    @NotNull
    default <V> Int2ObjectMap<V> toMapI2O(AFunctionO2I<T> keyFactory, AFunction<T, V> valFactory) {
        Int2ObjectOpenHashMap<V> res = new Int2ObjectOpenHashMap<>();
        while (hasNext()) {
            var val = next();
            res.put(keyFactory.apply(val), valFactory.apply(val));
        }
        return res;
    }

    /**
     * Collects elements into an object-to-int map.
     * @param keyFactory function to extract keys from elements
     * @param valFactory function to extract integer values from elements
     * @param <K> the key type
     * @return a new object-to-int map
     */
    @NotNull
    default <K> Object2IntMap<K> toMapO2I(AFunction<T, K> keyFactory, AFunctionO2I<T> valFactory) {
        Object2IntOpenHashMap<K> res = new Object2IntOpenHashMap<>();
        while (hasNext()) {
            var val = next();
            res.put(keyFactory.apply(val), valFactory.apply(val));
        }
        return res;
    }

    /**
     * Collects elements into a long-to-object map.
     * @param keyFactory function to extract long keys from elements
     * @param valFactory function to extract values from elements
     * @param <V> the value type
     * @return a new long-to-object map
     */
    @NotNull
    default <V> Long2ObjectMap<V> toMapL2O(AFunctionO2L<T> keyFactory, AFunction<T, V> valFactory) {
        Long2ObjectOpenHashMap<V> res = new Long2ObjectOpenHashMap<>();
        while (hasNext()) {
            var val = next();
            res.put(keyFactory.apply(val), valFactory.apply(val));
        }
        return res;
    }

    /**
     * Joins elements into a string with comma delimiters.
     * @return the joined string
     */
    default String joinD() {
        StringBuilder sb = new StringBuilder();
        joinD(sb);
        return sb.toString();
    }

    /**
     * Joins elements into a string with the specified delimiter.
     * @param delimer the delimiter string
     * @return the joined string
     */
    default String join(String delimer) {
        var sb = AString.of();
        join(sb, delimer);
        return sb.toString();
    }

    /**
     * Joins elements into a string with newline delimiters.
     * @return the joined string
     */
    default String joinN() {
        return join("\n");
    }

    /**
     * Joins elements into an AString with the specified delimiter.
     * @param sb the AString to append to
     * @param delimer the delimiter string
     */
    default void join(AString sb, String delimer) {
        boolean first = true;
        while (hasNext()) {
            if (first) {
                first = false;
            } else {
                sb.add(delimer);
            }
            sb.add(next());
        }
    }

    /**
     * Throws an exception using the provided runnable if any element matches the condition.
     * @param conditions the condition predicate
     * @param throwCode the exception throwing runnable
     * @return a flow with error checking
     */
    default Flow<T> errorIf(APredicate<T> conditions, Runnable throwCode) {
        return apply(v -> {
            if (conditions.test(v)) {
                throwCode.run();
            }
        });
    }

    /**
     * Throws an exception using the provided consumer if any element matches the condition.
     * @param conditions the condition predicate
     * @param throwCode the exception throwing consumer
     * @return a flow with error checking
     */
    default Flow<T> errorIf(APredicate<T> conditions, Consumer<T> throwCode) {
        return apply(v -> {
            if (conditions.test(v)) {
                throwCode.accept(v);
            }
        });
    }

    /**
     * Throws an exception of the specified type if any element matches the condition.
     * @param conditions the condition predicate
     * @param throwCode the exception class to throw
     * @return a flow with error checking
     */
    default Flow<T> errorIf(APredicate<T> conditions, Class<? extends Throwable> throwCode) {
        return apply(v -> {
            if (conditions.test(v)) {
                throw throwCode.getConstructor().newInstance();
            }
        });
    }

    /**
     * Returns a flow with distinct elements.
     * @return a new flow with duplicates removed
     */
    default Flow<T> distinct() {
        return filter(new APredicate<>() {
            private final Set<T> old = new ObjectOpenHashSet<>();

            @Override
            public boolean test2(T value) {
                return old.add(value);
            }
        });
    }

    /**
     * Joins elements into a string with the specified delimiter and element preparer.
     * @param preparer function to prepare elements for joining
     * @param delimer the delimiter string
     * @return the joined string
     */
    default String join(AFunction<T, Object> preparer, String delimer) {
        StringBuilder sb = new StringBuilder();
        boolean first = true;
        while (hasNext()) {
            if (first) {
                first = false;
            } else {
                sb.append(delimer);
            }
            sb.append(preparer.apply(next()));
        }
        return sb.toString();
    }

    /**
     * Returns the first element or null if empty.
     * @return the first element or null
     */
    default T getFirstOrNull() {
        if (hasNext()) return next();
        return null;
    }

    /**
     * Returns the first element or throws if empty.
     * @return the first element
     * @throws NoSuchElementException if flow is empty
     */
    default T getFirst() {
        if (hasNext()) return next();
        throw new NoSuchElementException();
    }

    /**
     * Returns the first matching element or throws if none match.
     * @param p the filtering predicate
     * @return the first matching element
     * @throws NoSuchElementException if no elements match
     */
    default T getFirst(APredicate<T> p) {
        return filter(p).getFirst();
    }

    /**
     * Returns the first matching element or null if none match.
     * @param p the filtering predicate
     * @return the first matching element or null
     */
    default T getFirstOrNull(APredicate<T> p) {
        return filter(p).getFirstOrNull();
    }

    /**
     * Joins elements into a string without delimiters.
     * @return the joined string
     */
    default String join() {
        StringBuilder sb = new StringBuilder();
        while (hasNext()) {
            sb.append(next());
        }
        return sb.toString();
    }

    /**
     * Joins elements into a string with delimiters and surrounding strings.
     * @param delim the delimiter string
     * @param prefix the prefix string
     * @param postfix the postfix string
     * @return the joined string
     */
    default String join(String delim, String prefix, String postfix) {
        StringBuilder sb = new StringBuilder(prefix);
        boolean first = true;
        while (hasNext()) {
            if (first) {
                first = false;
            } else {
                sb.append(delim);
            }
            sb.append(next());
        }
        sb.append(postfix);
        return sb.toString();
    }

    /**
     * Performs an action for each unique pair of elements (excluding self-pairs).
     * @param consumer the action to perform for each pair
     */
    @SuppressWarnings("unchecked")
    default void crossSelf(ABiConsumer<T, T> consumer) {
        var ar = toArray();
        for (var a : ar) {
            for (var b : ar) {
                if (a == b) continue;
                consumer.accept((T) a, (T) b);
            }
        }
    }

    /**
     * An empty flow singleton instance.
     */
    Flow<?> EMPTY = new FlowCompleted<>() {
        @Override
        public int count() {
            return 0;
        }

        @Override
        public Object random() {
            return null;
        }

        @Override
        public @NotNull List<Object> toList() {
            return new ObjectArrayList<>(0);
        }

        @Override
        public @NotNull Set<Object> toSet() {
            return ObjectSet.of();
        }

        @Override
        public String join() {
            return "";
        }

        @Override
        public void to(AConsumer<Object> consumer) {
        }

        @Override
        public @NotNull <K, V> Map<K, V> toMap(AFunction<Object, K> keyFactory, AFunction<Object, V> valFactory) {
            return Object2ObjectMaps.emptyMap();
        }

        @Override
        @NotNull
        public Object[] toArray(@NotNull Class<? super Object> arrayType) {
            return RU.cast(Array.newInstance(arrayType, 0));
        }

        @Override
        public @NotNull Object[] toArray() {
            return new Object[0];
        }

        @Override
        public @NotNull Iterator<Object> iterator() {
            return RU.cast(ObjectIterators.EMPTY_ITERATOR);
        }

        @Override
        public boolean anyMatch(APredicate<Object> p) {
            return false;
        }

        @Override
        public boolean noneMatch(APredicate<Object> p) {
            return true;
        }

        @Override
        public boolean noneMatchObj(Object p) {
            return true;
        }

        @Override
        public boolean anyMatchObj(Object p) {
            return false;
        }

        @Override
        public Flow<Object> skip(int count) {
            return this;
        }

        @Override
        public Flow<Object> limit(int i) {
            return this;
        }

        @Override
        public Object getFirstOrNull(APredicate<Object> p) {
            return null;
        }

        @Override
        public void run() {
        }

        @Override
        public Flow<Object> ifError(AFunction<Throwable, Object> f) {
            return this;
        }

        @Override
        public String joinD() {
            return "";
        }

        @Override
        public String joinN() {
            return "";
        }

        @Override
        public Flow<Object> reverse() {
            return this;
        }

        @Override
        public @NotNull Flow<Object> filterNotNull() {
            return this;
        }

        @Override
        public boolean isEmpty() {
            return true;
        }

        @Override
        public void forEach(Consumer<? super Object> action) {
        }

        @Override
        public void forEachRemaining(Consumer<? super Object> action) {
        }

        @Override
        public boolean allMatch(APredicate<Object> p) {
            return true;
        }

        @Override
        @SuppressWarnings("unchecked")
        public @NotNull <E> Flow<E> map(@NotNull AFunction<Object, E> f) {
            return (Flow<E>) this;
        }

        @Override
        public @NotNull Flow<Object> apply(AConsumer<Object> c) {
            return this;
        }

        @Override
        public Flow<Object> addAllEls(Object... values) {
            return flow(values);
        }

        @Override
        @SuppressWarnings("unchecked")
        public @NotNull <E> Flow<E> flatMap1(@NotNull AFunctionO2Iterable<Object, E> f) {
            return (Flow<E>) this;
        }

        @Override
        @SuppressWarnings("unchecked")
        public @NotNull <E> Flow<E> flatMap2(@NotNull AFunctionO2Array<Object, E> f) {
            return (Flow<E>) this;
        }

        @Override
        public void foreach(AConsumer<Object> c) {
        }

        @Override
        @SuppressWarnings("unchecked")
        public @NotNull <E> Flow<E> flatMap3(@NotNull AFunctionO2Stream<Object, E> f) {
            return (Flow<E>) this;
        }

        @Override
        @SuppressWarnings("unchecked")
        public @NotNull <E> Flow<E> flatMap4(@NotNull AFunctionO2Iterator<Object, E> f) {
            return (Flow<E>) this;
        }

        @Override
        @SuppressWarnings("unchecked")
        public @NotNull <E> Flow<E> flatMap(@NotNull ABiConsumer<Object, AConsumer<E>> f) {
            return (Flow<E>) this;
        }

        @Override
        public <E extends Collection<Object>> E toCollection(@NotNull E collection) {
            return collection;
        }

        @Override
        public @NotNull Flow<Object> filter(@Nullable APredicate<Object> predicate) {
            return this;
        }

        @Override
        public @NotNull Flow<Object> filterNot(@Nullable APredicate<Object> predicate) {
            return this;
        }

        @Override
        @SuppressWarnings("unchecked")
        public @NotNull <E> Flow<E> filterByType(@NotNull Class<E> type) {
            return (Flow<E>) this;
        }

        @Override
        public @NotNull Flow<Object> ifEmpty(Exception error) {
            RU.error(error);
            throw new RuntimeException(error);
        }

        @Override
        public @NotNull Flow<Object> ifEmpty(ARunnable task) {
            task.run();
            return this;
        }

        @Override
        public @NotNull Flow<Object> ignoreError(Class<? extends Exception> ee) {
            return this;
        }

        @Override
        public @NotNull Flow<Object> sort(@NotNull AComparator<Object> comparator) {
            return this;
        }

        @Override
        public @NotNull Flow<Object> sort(@NotNull Comparator<Object> comparator) {
            return this;
        }

        @Override
        public @NotNull Iterable<Object> toIterable() {
            return ObjectList.of();
        }

        @Override
        public @NotNull Supplier<Object> toSupplier() {
            return () -> null;
        }

        @Override
        public @NotNull <V> Int2ObjectMap<V> toMapI2O(AFunctionO2I<Object> keyFactory, AFunction<Object, V> valFactory) {
            return Int2ObjectMaps.emptyMap();
        }

        @Override
        public @NotNull <K> Object2IntMap<K> toMapO2I(AFunction<Object, K> keyFactory, AFunctionO2I<Object> valFactory) {
            return Object2IntMaps.emptyMap();
        }

        @Override
        public @NotNull <V> Long2ObjectMap<V> toMapL2O(AFunctionO2L<Object> keyFactory, AFunction<Object, V> valFactory) {
            return Long2ObjectMaps.emptyMap();
        }

        @Override
        public Flow<Object> distinct() {
            return this;
        }

        @Override
        public String join(String delimer) {
            return "";
        }

        @Override
        public String join(AFunction<Object, Object> preparer, String delimer) {
            return "";
        }

        @Override
        public Object getFirstOrNull() {
            return null;
        }

        @Override
        public Object getFirst() {
            throw new NoSuchElementException();
        }

        @Override
        public Object getFirst(APredicate<Object> p) {
            throw new NoSuchElementException();
        }

        @Override
        public String join(String delim, String prefix, String postfix) {
            return "";
        }

        @Override
        public void crossSelf(ABiConsumer<Object, Object> consumer) {
        }

        @Override
        public void crossSelf(ABiPredicate<Object, Object> p, ABiConsumer<Object, Object> consumer) {
        }

        @Override
        public boolean hasNext() {
            return false;
        }

        @Override
        public Object next() {
            return null;
        }
    };

    /**
     * Performs an action for each unique pair of elements that satisfy the predicate.
     * @param p the pair predicate
     * @param consumer the action to perform for matching pairs
     */
    @SuppressWarnings("unchecked")
    default void crossSelf(ABiPredicate<T, T> p, ABiConsumer<T, T> consumer) {
        var ar = toArray();
        for (var a : ar) {
            for (var b : ar) {
                if (a == b || !p.test((T) a, (T) b)) continue;
                consumer.accept((T) a, (T) b);
            }
        }
    }

    /**
     * Returns the minimum element according to the provided comparator.
     * @param comparator the comparator to determine order
     * @return Optional containing the minimum element, or empty if flow is empty
     */
    default Optional<T> min(AComparator<T> comparator) {
        T min = null;
        while (hasNext()) {
            var v = next();
            if (min == null) {
                min = v;
            } else {
                if (comparator.compare(min, v) > 0) {
                    min = v;
                }
            }
        }
        return Optional.ofNullable(min);
    }

    /**
     * Returns the maximum element according to the provided comparator.
     * @param comparator the comparator to determine order
     * @return Optional containing the maximum element, or empty if flow is empty
     */
    default Optional<T> max(AComparator<T> comparator) {
        T max = null;
        while (hasNext()) {
            var v = next();
            if (max == null) {
                max = v;
            } else {
                if (comparator.compare(max, v) < 0) {
                    max = v;
                }
            }
        }
        return Optional.ofNullable(max);
    }

    /**
     * Returns whether the flow is empty.
     * @return true if flow has no elements, false otherwise
     */
    default boolean isEmpty() {
        return !hasNext();
    }

    /**
     * Skips the last N elements of the flow.
     * @param count the number of elements to skip
     * @return a new flow without the last N elements
     */
    default Flow<T> skipLast(int count) {
        var a = toArray();
        if (a.length < count) return Flow.flow();
        a = Arrays.copyOfRange(a, 0, a.length - count);
        return RU.cast(flow(a));
    }

    /**
     * Skips the first N elements of the flow.
     * @param count the number of elements to skip
     * @return a new flow without the first N elements
     */
    default Flow<T> skip(int count) {
        if (count == 0) return this;
        var self = this;
        return new Flow<>() {
            private boolean skipped = false;

            private void skip() {
                if (!skipped) {
                    int i = 0;
                    while (i < count && self.hasNext()) {
                        self.next();
                        i++;
                    }
                    skipped = true;
                }
            }

            @Override
            public boolean hasNext() {
                skip();
                return self.hasNext();
            }

            @Override
            public T next() {
                skip();
                return self.next();
            }
        };
    }

    /**
     * Counts the number of elements in the flow.
     * @return the element count
     */
    default int count() {
        int res = 0;
        while (hasNext()) {
            res++;
            next();
        }
        return res;
    }

    /**
     * Applies a function to the entire flow.
     * @param f the function to apply
     * @param <R> the result type
     * @return the function result
     */
    default <R> R allMap(AFunction<Flow<T>, R> f) {
        return f.apply(this);
    }

    /**
     * Joins elements into a StringBuilder.
     * @param sb the StringBuilder to append to
     */
    default void join(StringBuilder sb) {
        while (hasNext()) {
            sb.append(next());
        }
    }

    /**
     * Creates a flow from another flow (identity function).
     * @param stream the source flow
     * @param <T> the element type
     * @return the same flow
     */
    @NotNull
    static <T> Flow<T> flow(@NotNull Flow<T> stream) {
        return stream;
    }

    /**
     * Creates a flow from string values.
     * @param ss the string values
     * @return a new string flow
     */
    @NotNull
    static Flow<String> flow(@NotNull String... ss) {
        return RU.cast(flow((Object[]) ss));
    }

    /**
     * Creates a flow from array values.
     * @param ss the array values
     * @param <T> the element type
     * @return a new flow
     */
    @SafeVarargs
    static <T> Flow<T> flowArray(@NotNull T... ss) {
        return RU.cast(flow(ss));
    }

    /**
     * Creates a flow from an array.
     * @param array the source array
     * @param <T> the element type
     * @return a new flow
     */
    @NotNull
    static <T> Flow<T> flow(@NotNull T[] array) {
        if (array == null || array.length == 0) return flow();
        return new FlowCompleted<>() {
            int pos;

            @Override
            public int count() {
                return array.length;
            }

            @Override
            public T random() {
                return array[RANDOM.nextInt(array.length)];
            }

            @Override
            public boolean hasNext() {
                return pos < array.length;
            }

            @Override
            public T next() {
                return array[pos++];
            }

            @Override
            public @NotNull Object[] toArray() {
                return array;
            }

            @Override
            public void foreach(AConsumer<T> c) {
                for (var v : array) {
                    c.accept(v);
                }
            }

            @Override
            public @NotNull List<T> toList() {
                return new ObjectArrayList<>(array);
            }

            @Override
            public @NotNull Supplier<T> toSupplier() {
                return new ASupplier<>() {
                    int pos;

                    @Override
                    public T get2() {
                        if (pos == array.length) return null;
                        return array[pos++];
                    }
                };
            }

            @Override
            @NotNull
            public T[] toArray(@NotNull Class<? super T> arrayType) {
                T[] res = RU.cast(Array.newInstance(arrayType, array.length));
                System.arraycopy(array, 0, res, 0, array.length);
                return res;
            }
        };
    }

    /**
     * Creates a long flow from a long array.
     * @param array the source array
     * @return a new long flow
     */
    @NotNull
    static FlowLong flow(long @NotNull [] array) {
        return FlowLong.of(array);
    }

    /**
     * Creates an integer flow representing a range.
     * @param number the range end (exclusive)
     * @return a new integer flow
     */
    static FlowInt flow(int number) {
        return new FlowCompletedInt() {
            int c = 0;

            @Override
            public int count() {
                return number;
            }

            @Override
            public int nextInt() {
                return c++;
            }

            @Override
            public boolean hasNext() {
                return c < number;
            }
        };
    }

    /**
     * Creates an integer flow from an int array.
     * @param array the source array
     * @return a new integer flow
     */
    static FlowInt flow(int @NotNull [] array) {
        return FlowInt.of(array);
    }

    /**
     * Creates a short flow from a short array.
     * @param array the source array
     * @return a new short flow
     */
    static FlowShort flow(short @NotNull [] array) {
        return FlowShort.of(array);
    }

    /**
     * Creates a byte flow from a byte array.
     * @param array the source array
     * @return a new byte flow
     */
    @NotNull
    static FlowByte flow(byte @NotNull [] array) {
        return FlowByte.of(array);
    }

    /**
     * Creates a flow from an iterable.
     * @param iterable the source iterable
     * @param <T> the element type
     * @return a new flow
     */
    @NotNull
    static <T> Flow<T> flow(@NotNull Iterable<T> iterable) {
        if (iterable instanceof Set) return flow((Set<T>) iterable);
        if (iterable instanceof List) return flow((List<T>) iterable);
        if (iterable instanceof Collection) return flow((Collection<T>) iterable);
        return flow(iterable.iterator());
    }

    /**
     * Creates a long flow from a long iterable.
     * @param iterable the source iterable
     * @return a new long flow
     */
    @NotNull
    static FlowLong flow(@NotNull LongIterable iterable) {
        return FlowLong.of(iterable);
    }

    /**
     * Creates an integer flow from an int iterable.
     * @param iterable the source iterable
     * @return a new integer flow
     */
    @NotNull
    static FlowInt flow(@NotNull IntIterable iterable) {
        return FlowInt.of(iterable);
    }

    /**
     * Creates a long flow from a long set.
     * @param collection the source set
     * @return a new long flow
     */
    @NotNull
    static FlowLong flow(@NotNull LongSet collection) {
        return FlowLong.of(collection);
    }

    /**
     * Creates a flow from a set.
     * @param collection the source set
     * @param <T> the element type
     * @return a new flow
     */
    @NotNull
    static <T> Flow<T> flow(@NotNull Set<T> collection) {
        if (collection.isEmpty()) return flow();
        var it = collection.iterator();
        return new FlowCompleted<>() {
            @Override
            public int count() {
                return collection.size();
            }

            @Override
            public boolean hasNext() {
                return it.hasNext();
            }

            @Override
            public T next() {
                return it.next();
            }

            @Override
            public @NotNull Set<T> toSet() {
                return collection;
            }
        };
    }

    /**
     * Creates a long flow from a long list.
     * @param collection the source list
     * @return a new long flow
     */
    @NotNull
    static FlowLong flow(@NotNull LongList collection) {
        return FlowLong.of(collection);
    }

    /**
     * Creates an integer flow from an int list.
     * @param collection the source list
     * @return a new integer flow
     */
    @NotNull
    static FlowInt flow(@NotNull IntList collection) {
        return FlowInt.of(collection);
    }

    /**
     * Creates a flow from a list.
     * @param collection the source list
     * @param <T> the element type
     * @return a new flow
     */
    @NotNull
    static <T> Flow<T> flow(@NotNull List<T> collection) {
        if (collection.isEmpty()) return flow();
        var it = collection.iterator();
        return new FlowCompleted<>() {
            @Override
            public int count() {
                return collection.size();
            }

            @Override
            public boolean hasNext() {
                return it.hasNext();
            }

            @Override
            public T next() {
                return it.next();
            }

            @Override
            public @NotNull List<T> toList() {
                return new ObjectArrayList<>(collection);
            }
        };
    }

    /**
     * Creates a flow from a collection.
     * @param collection the source collection
     * @param <T> the element type
     * @return a new flow
     */
    @NotNull
    static <T> Flow<T> flow(@NotNull Collection<T> collection) {
        if (collection.isEmpty()) return flow();
        var it = collection.iterator();
        return new FlowCompleted<>() {
            @Override
            public int count() {
                return collection.size();
            }

            @Override
            public boolean hasNext() {
                return it.hasNext();
            }

            @Override
            public T next() {
                return it.next();
            }

            @Override
            public @NotNull List<T> toList() {
                return new ObjectArrayList<>(collection);
            }

            @Override
            public @NotNull Set<T> toSet() {
                return new ObjectOpenHashSet<>(collection);
            }
        };
    }

    /**
     * Creates a flow from a supplier.
     * @param supplier the element supplier
     * @param <T> the element type
     * @return a new flow
     */
    @NotNull
    static <T> Flow<T> flow(@NotNull ASupplier<T> supplier) {
        return new Flow<>() {
            private T last;
            private boolean end;

            @Override
            public boolean hasNext() {
                if (end) return false;
                if (last != null) return true;
                last = supplier.get();
                if (last == null) {
                    end = true;
                    return false;
                }
                return true;
            }

            @Override
            public T next() {
                assert last != null;
                return last;
            }
        };
    }

    /**
     * Creates a long flow from a long iterator.
     * @param iterator the source iterator
     * @return a new long flow
     */
    @NotNull
    static FlowLong flow(@NotNull LongIterator iterator) {
        return FlowLong.of(iterator);
    }

    /**
     * Creates a flow from an iterator.
     * @param iterator the source iterator
     * @param <T> the element type
     * @return a new flow
     */
    @NotNull
    static <T> Flow<T> flow(@NotNull Iterator<T> iterator) {
        if (!iterator.hasNext()) return flow();
        return new Flow<>() {
            @Override
            public boolean hasNext() {
                return iterator.hasNext();
            }

            @Override
            public T next() {
                return iterator.next();
            }
        };
    }

    /**
     * Creates a flow from a map's entry set.
     * @param map the source map
     * @param <K> the key type
     * @param <V> the value type
     * @return a new flow of map entries
     */
    @NotNull
    static <K, V> Flow<Map.Entry<K, V>> flow(@NotNull Map<K, V> map) {
        return flow(map.entrySet());
    }

    /**
     * Creates an empty flow.
     * @param <T> the element type
     * @return an empty flow
     */
    @SuppressWarnings("unchecked")
    static <T> Flow<T> flow() {
        return (Flow<T>) EMPTY;
    }

    /**
     * Functional interface for splitting elements into three categories.
     * @param <T> the element type
     */
    interface SplitComparator<T> {
        /**
         * Tests an element and returns a comparison result.
         * @param value the element to test
         * @return positive, negative, or zero to indicate category
         */
        int test(T value);
    }
}