package io.aether.utils.futures;

import io.aether.utils.RU;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.interfaces.ARunnable;
import org.jetbrains.annotations.NotNull;

import java.util.concurrent.Executor;

/**
 * Base interface defining the common life cycle, status, and chaining methods
 * for all asynchronous operations (AFuture and ARFuture).
 * @param <Self> The concrete future type (AFuture or ARFuture) for fluent chaining.
 */
public interface AFutureBase<Self extends AFutureBase<Self>> {
    /**
     * Registers a listener to be executed if the future completes with a cancellation.
     * @param l The runnable task to execute on cancellation.
     */
    Self onCancel(ARunnable l);
    boolean updateStatus(Object result);
    /**
     * Registers a listener to be executed if the future completes with a cancellation.
     * @param l The consumer that accepts the future instance on cancellation.
     */
    Self onCancel(AConsumer<Self> l);

    /**
     * Registers a listener to be executed if the future completes with an error.
     * @param l The consumer that accepts the Throwable error.
     * @return This future instance for chaining.
     */
    Self onError(AConsumer<Throwable> l);

    /**
     * Checks if the future has reached a final state (Done, Error, or Canceled).
     * @return true if the status is final.
     */
    boolean isFinalStatus();

    /**
     * Checks if the final status is an error.
     * @return true if the future completed with an error.
     */
    boolean isError();

    /**
     * Completes the future with the given error.
     * @param e The Throwable error.
     * @return This future instance for chaining.
     */
    Self error(Throwable e);

    /**
     * Gets the error Throwable if the future is in an error state.
     * @return The error or null.
     */
    Throwable getError();

    /**
     * Sets the error Throwable. Same as {@link #error(Throwable)}.
     * @param e The Throwable error.
     */
    void setError(Throwable e);

    /**
     * Checks if the future has not reached the Done state.
     * @return true if the future is pending or in an Error/Canceled state.
     */
    boolean isNotDone();

    /**
     * Checks if the future has reached the Done state (completed successfully).
     * @return true if completed successfully.
     */
    boolean isDone();

    /**
     * Checks if the future has been canceled.
     * @return true if canceled.
     */
    boolean isCanceled();

    /**
     * Cancels the future.
     */
    void cancel();

    /**
     * Sets a timeout duration in seconds, after which the future is automatically marked as an error.
     * @param seconds Timeout duration.
     * @param text Error message text.
     * @return This future instance for chaining.
     */
    Self timeoutError(int seconds, String text);

    /**
     * Sets a timeout duration in seconds, executing a task if the timeout occurs.
     * @param seconds Timeout duration.
     * @param task The task to run on timeout.
     * @return This future instance for chaining.
     */
    Self timeout(int seconds, ARunnable task);

    /**
     * Sets a timeout duration in milliseconds, executing a task if the timeout occurs.
     * @param ms Timeout duration in milliseconds.
     * @param task The task to run on timeout.
     * @return This future instance for chaining.
     */
    Self timeoutMs(long ms, ARunnable task);

    /**
     * Sets a timeout duration in milliseconds, executing a consumer with this future if the timeout occurs.
     * @param ms Timeout duration in milliseconds.
     * @param task The consumer to run on timeout.
     * @return This future instance for chaining.
     */
    Self timeoutMs(long ms, AConsumer<Self> task);

    /**
     * Adds a listener to the future's completion status.
     * @param l The consumer that accepts the future instance upon finalization.
     * @return true if the listener was successfully added (and the future was not yet finalized).
     */
    boolean addListener(AConsumer<Self> l);
    default Self addListener2(AConsumer<Self> l){
        addListener(l);
        return RU.cast(this);
    }

    /**
     * Executes a runnable task on the provided executor upon successful completion.
     * @param executor The executor to run the task on.
     * @param t The runnable task.
     * @return This future instance for chaining.
     */
    Self to(Executor executor, ARunnable t);

    /**
     * Executes a runnable task upon successful completion in the current context.
     * @param t The runnable task.
     * @return This future instance for chaining.
     */
    Self to(ARunnable t);

    /**
     * Gets the raw result object. Intended for internal implementation use.
     * @return The raw result object (which may be NULL, CANCEL_VALUE, or Throwable).
     */
    Object getNowRaw();

    boolean tryError(@NotNull Throwable error);

    Self tryCancel();
}