package io.aether.utils.futures;

import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.interfaces.ARunnable;
import io.aether.utils.interfaces.ASupplier;
import it.unimi.dsi.fastutil.objects.ObjectList;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Iterator;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Internal implementation of the AFuture interface.
 * This class is package-private and should only be instantiated
 * or accessed via the static factory methods in the AFuture interface.
 */
final class AFutureImpl extends AFutureBaseImpl<AFuture> implements AFuture {
    // Static instances are now final members of the implementation class.
    static final AFuture DONED;
    static final AFuture CANCELED;

    private static final Object DONE_VALUE = new Object() {
        @Override
        public String toString() {
            return "DONE";
        }
    };

    AFutureImpl(Throwable e) {
        setError(e);
    }

    AFutureImpl() {
    }

    @Override
    public String toString() {
        return "AFuture(" + (isDone() ? "done:" : (isError() ? "error:" : (isCanceled() ? "canceled" : ""))) + result + ")";
    }

    @Override
    public AFuture and(AFuture f) {
        return all(this, f);
    }

    @Override
    public AFuture apply(ARunnable t) {
        AFuture res = new AFutureImpl();
        to(() -> {
            t.run();
            res.done();
        });
        return res;
    }

    @Override
    public void done() {
        if (!updateStatus(DONE_VALUE)) {
            throw new IllegalStateException();
        }
    }

    @Override
    public boolean tryDone() {
        return updateStatus(DONE_VALUE);
    }

    @Override
    public <T> ARFuture<T> mapRFuture(ASupplier<T> t) {
        ARFuture<T> res = new ARFutureImpl<>();
        addListener((f) ->{
            if(f.isDone()){
                res.done(t.get());
            }else if(f.isError()){
                res.error(f.getError());
            }else if(f.isCanceled()){
                res.cancel();
            }
        });
        return res;
    }

    @Override
    @NotNull
    public AFuture to(@NotNull AFuture f) {
        f.addListener(c -> updateStatus(c.getNowRaw()));
        addListener(c -> f.updateStatus(c.getNowRaw()));
        return this;
    }

    static {
        DONED = new AFutureImpl();
        DONED.updateStatus(DONE_VALUE);

        CANCELED = new AFutureImpl();
        CANCELED.cancel();
    }

    // ==================== STATIC IMPLEMENTATIONS ====================

    static AFuture run(Executor executor, AConsumer<AFuture> task) {
        AFuture f = new AFutureImpl();
        executor.execute(() -> {
            try {
                task.accept(f);
                if (!f.isCanceled()) {
                    f.done();
                }
            } catch (Throwable e) {
                f.setError(e);
            }
        });
        return f;
    }

    static AFuture run(Executor executor, ARunnable task) {
        AFuture f = new AFutureImpl();
        executor.execute(() -> {
            try {
                if (!f.isCanceled()) {
                    task.run();
                }
                if (!f.isCanceled()) {
                    f.done();
                }
            } catch (Throwable e) {
                f.setError(e);
            }
        });
        return f;
    }

    static AFuture any(Collection<AFuture> ff, AConsumer<AFuture> other) {
        if (ff.size() == 1) return ff.iterator().next();
        for (var a : ff) {
            if (a.isDone()) {
                return a;
            }
        }
        AFuture res = new AFutureImpl();
        for (var a : ff) {
            a.addListener(c -> {
                if (!res.isDone() && c.isFinalStatus()) {
                    if (((AFutureImpl)res).updateStatus(((AFutureBaseImpl<?>)c).result)) {
                        for (var aa : ff) {
                            if (aa == c) continue;
                            other.accept(aa);
                        }
                    }
                }
            });
        }
        return res;
    }

    static AFuture all(AFuture... ff) {
        if (ff.length == 0) return AFuture.of();
        return all(ObjectList.of(ff).iterator());
    }

    static AFuture all(Collection<AFuture> ff) {
        if (ff.isEmpty()) return AFuture.of();
        return all(ff.iterator());
    }

    static AFuture all(Iterator<AFuture> ff) {
        AFuture res = new AFutureImpl();
        AFuture[] fff = Flow.flow(ff).toArray(AFuture.class);
        if (fff.length == 0) {
            return AFuture.of();
        }

        // The counter tracks only the UNRESOLVED futures.
        AtomicInteger counter = new AtomicInteger(0);

        AConsumer<AFuture> listener = t -> {
            // If the resulting future is already finalized (error/cancel), just return
            if (res.isFinalStatus()) return;

            if (t.isDone()) {
                // If a future is successfully completed, decrement the counter.
                if (counter.decrementAndGet() == 0) res.done();
            } else if (t.isError()) {
                // If an error occurs, immediately fail the resulting future.
                res.setError(t.getError());
            } else if (t.isCanceled()) {
                // If cancelled, immediately cancel the resulting future.
                res.cancel();
            }
        };

        boolean allDone = true;
        for (var aFuture : fff) {
            // Must cast aFuture back to AFutureImpl to access the protected implementation methods if needed.
            AFutureBaseImpl<?> implFuture = (AFutureBaseImpl<?>) aFuture;

            if (implFuture.isFinalStatus()) {
                // Check status of already finalized futures
                if (implFuture.isError() || implFuture.isCanceled()) {
                    // If an error or cancellation is found, finalize and return immediately.
                    res.updateStatus(implFuture.getNowRaw());
                    return res;
                }
                // If successfully done, allDone remains true, and we skip adding listener/incrementing counter.
            } else {
                // An unresolved future is found.
                allDone = false;
                counter.incrementAndGet(); // Increment the counter only for unresolved futures.
                aFuture.addListener(listener);
            }
        }

        if (allDone) {
            // If the loop finished and all futures were already successfully done, complete res.
            res.done();
        }
        return res;
    }
}