package io.aether.utils.futures;

import io.aether.logger.Log;
import io.aether.utils.RU;
import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.*;
import io.aether.utils.tuples.Tuple2;
import io.aether.utils.tuples.Tuple3;
import org.jetbrains.annotations.NotNull;

import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.lang.reflect.Array;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Internal implementation of the ARFuture interface.
 * This class is package-private and should only be instantiated
 * or accessed via the static factory methods in the ARFuture interface.
 *
 * @param <T> The type of the result value.
 */
class ARFutureImpl<T> extends AFutureBaseImpl<ARFuture<T>> implements ARFuture<T> {
    // Static instances are now final members of the implementation class.
    static final ARFuture<Boolean> TRUE = new ARFutureImpl<>(true);
    static final ARFuture<Boolean> FALSE = new ARFutureImpl<>(false);
    private static final AConsumer<ARFuture<?>> CANCEL = ARFuture::cancel;
    private static final VarHandle AA = MethodHandles.arrayElementVarHandle(Object[].class);

    ARFutureImpl() {
    }

    // Constructor used by static factory methods (ARFuture.of(value))
    @SuppressWarnings("unchecked")
    ARFutureImpl(T value) {
        result = value;
        tasks = RU.cast(NOP_TASK);
    }

    @Override
    public ARFuture<T> to(@NotNull AConsumer<T> task, int timeout, ARunnable onTimeout) {
        to(task);
        return timeout(timeout, onTimeout);
    }

    @Override
    public ARFutureWithFlag<T> toWithFlag() {
        var res = new ARFutureWithFlag<T>();
        to(res);
        return res;
    }

    @Override
    public String toString() {
        return "ARFuture(" + (isDone() ? "done:" : (isError() ? "error:" : (isCanceled() ? "canceled" : ""))) + result + ")";
    }

    @Override
    public <T2> ARFuture<Tuple2<T, T2>> and(ARFuture<T2> f) {
        return ARFuture.all(this, f);
    }

    @Override
    public ARFuture<T> to(AConsumer<T> onDone, AConsumer<Throwable> onError) {
        to(onDone);
        onError(onError);
        return this;
    }

    @Override
    @SuppressWarnings("unchecked")
    public ARFuture<T> to(AConsumer<T> onDone) {
        assert onDone != null;
        if (isDone()) {
            var r = result;
            if (r == NULL) r = null;
            onDone.accept((T) r);
            return this;
        }
        addListener(f -> {
            if (f.isDone()) {
                var r = result;
                if (r == NULL) r = null;
                onDone.accept((T) r);
            }
        });
        return this;
    }

    @Override
    public void done(T value) {
        if (!updateStatus(value)) {
            throw new IllegalStateException();
        }
    }

    @Override
    public boolean tryDone(T value) {
        return updateStatus(value);
    }

    @Override
    @SuppressWarnings("unchecked")
    public T get() {
        waitSuccessful();
        var r = result;
        if (r == NULL) r = null;
        if (isError()) {
            RU.error((Throwable) r);
        } else if (isCanceled()) {
            throw new RuntimeException();
        }
        return (T) r;
    }

    public T get(int timeout) {
        return get((long) timeout);
    }

    public T getSeconds(int timout) {
        return get((long) timout * 1000);
    }

    @Override
    @SuppressWarnings("unchecked")
    public T get(long timout) {
        if (!waitSuccessful(timout)) {
            RU.error(RU.filterFrontAndBackStackTrace(new TimeoutException(), ARFuture.class));
        }
        var r = result;
        if (r == NULL) r = null;
        if (isError()) {
            RU.error((Throwable) r);
        }
        if (isCanceled()) {
            throw new RuntimeException();
        }
        return (T) r;
    }

    @Override
    public T getNow() {
        return getNowElse(null);
    }

    @Override
    @SuppressWarnings("unchecked")
    public T getNowElse(T elseValue) {
        if (isDone()) {
            var r = result;
            if (r == NULL) r = null;
            return (T) r;
        } else {
            return elseValue;
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    public <E> ARFuture<E> mapSafe(@NotNull AFunction<T, E> f) {
        ARFuture<E> res = new ARFutureImpl<>();
        addListener(c -> {
            if (c.isDone()) {
                var r = get();
                // Executor is intentionally hardcoded to Executor
                ((Executor) r).execute(() -> {
                    try {
                        res.done(f.apply(r));
                    } catch (Throwable e) {
                        res.error(e);
                    }
                });
            } else if (c.isError()) {
                RU.error((Throwable) c.getNowRaw());
            }
        });
        return res;
    }

    @Override
    @NotNull
    public AFuture mapToFuture(@NotNull AConsumer<T> f) {
        return map((v) -> {
            f.accept(v);
            return null;
        }).toFuture();
    }

    @Override
    @NotNull
    @SuppressWarnings("unchecked")
    public <E> ARFuture<E> mapRFuture(@NotNull AFunction<T, ARFuture<E>> f) {
        ARFuture<E> res = new ARFutureImpl<>();
        addListener(c -> {
            if (c.isDone()) {
                try {
                    var r = c.getNowRaw();
                    if (r == NULL) r = null;
                    f.apply(RU.cast(r)).to(res);
                } catch (Throwable e) {
                    res.error(e);
                }
            } else if (c.isError()) {
                res.error(c.getError());
            } else if (c.isCanceled()) {
                res.cancel();
            }
        });
        return res;
    }

    @Override
    @NotNull
    public <E> ARFuture<E> mapWL(@NotNull AFunction<T, E> f) {
        return map(Log.wrap(f));
    }

    @Override
    @NotNull
    @SuppressWarnings("unchecked")
    public <E> ARFuture<E> map(@NotNull AFunction<T, E> f) {
        ARFuture<E> res = new ARFutureImpl<>();
        addListener(c -> {
            if (c.isDone()) {
                try {
                    var r = c.getNowRaw();
                    if (r == NULL) r = null;
                    res.done(f.apply(RU.cast(r)));
                } catch (Throwable e) {
                    res.error(e);
                }
            } else if (c.isError()) {
                res.error(c.getError());
            } else if (c.isCanceled()) {
                res.cancel();
            }
        });
        return res;
    }

    @Override
    public ARFuture<T> apply(ARunnable c) {
        return apply(v -> c.run());
    }

    @Override
    public ARFuture<T> apply(AConsumer<T> c) {
        ARFuture<T> res = new ARFutureImpl<>();
        to((d) -> {
            c.accept(d);
            res.done(d);
        });
        return res;
    }

    public <E> ARFuture<E> decompose(Class<E> t) {
        return decompose();
    }

    @SuppressWarnings("unchecked")
    private <E> ARFuture<E> decompose() {
        ARFuture<E> res = new ARFutureImpl<>();
        addListener(v -> {
            if (v.isDone()) {
                var f = (ARFuture<E>) v.getNowRaw();
                res.updateStatus(((AFutureBaseImpl<?>) f).result);
            } else {
                res.updateStatus(v.getNowRaw());
            }
        });
        return res;
    }

    @Override
    public ARFuture<T> to(@NotNull ARFuture<T> f) {
        f.onCancel(this::cancel);
        addListener(c -> f.updateStatus(getNowRaw()));
        return this;
    }

    @Override
    public ARFuture<T> to(@NotNull AFuture f) {
        f.onCancel(this::cancel);
        addListener(c -> {
            if (c.isDone()) {
                f.done();
            } else if (c.isCanceled()) {
                f.cancel();
            } else if (c.isError()) {
                f.error(c.getError());
            }
        });
        return this;
    }

    @Override
    public AFuture toFuture() {
        AFuture res = new AFutureImpl();
        to(res);
        return res;
    }

    // ==================== STATIC IMPLEMENTATIONS ====================

    static <T> ARFuture<T> of(T value) {
        if (value instanceof Boolean) {
            return RU.cast((Boolean) value ? TRUE : FALSE);
        }
        return new ARFutureImpl<>(value);
    }

    static <T> ARFuture<T> canceled() {
        var f = new ARFutureImpl<T>();
        f.cancel();
        return f;
    }

    static <T> ARFuture<T> ofThrow(Throwable throwable) {
        ARFuture<T> res = new ARFutureImpl<>();
        res.error(throwable);
        return res;
    }

    static <T> ARFuture<T> run2(Executor executor, ACallable<ARFuture<T>> task) {
        ARFuture<T> f = new ARFutureImpl<>();
        executor.execute(() -> {
            try {
                var res = task.call();
                res.to(f);
            } catch (Throwable e) {
                f.error(e);
            }
        });
        return f;
    }

    static <T> ARFuture<T> run(Executor executor, ACallable<T> task) {
        ARFuture<T> f = new ARFutureImpl<>();
        executor.execute(() -> {
            try {
                var res = task.call();
                if (!f.isCanceled()) {
                    f.done(res);
                }
            } catch (Throwable e) {
                f.error(e);
            }
        });
        return f;
    }

    static <T> ARFuture<T> anyAndCancel(Flow<ARFuture<T>> ff) {
        return any(ff.toList(), ARFuture::cancel);
    }

    static <T> ARFuture<T> any(Collection<ARFuture<T>> ff) {
        return any(ff, (a) -> {
        });
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    static <T> ARFuture<T> any(Collection<ARFuture<T>> ff, AConsumer<ARFuture<T>> other) {
        if (ff.size() == 1) return ff.iterator().next();
        for (var a : ff) {
            if (a.isDone()) {
                return a;
            }
        }
        ARFuture<T> res = new ARFutureImpl<>();
        for (var a : ff) {
            a.addListener(c -> {
                if (!res.isDone() && c.isFinalStatus()) {
                    if (c.isDone() || c.isError()) {
                        if (res.updateStatus(c.getNowRaw())) {
                            for (var aa : ff) {
                                if (aa == c) continue;
                                other.accept(aa);
                            }
                        }
                    } else if (c.isCanceled()) {
                        if (res.updateStatus(AFutureBaseImpl.CANCEL_VALUE)) {
                            for (var aa : ff) {
                                if (aa == c) continue;
                                other.accept(aa);
                            }
                        }
                    }
                }
            });
        }
        return res;
    }

    @SuppressWarnings("unchecked")
    static <T> ARFuture<T[]> all(Class<T> elType, Flow<ARFuture<T>> ff) {
        return all(elType, ff.toList());
    }

    @SafeVarargs
    static <T> ARFuture<T[]> all(Class<T> elType, ARFuture<T>... ff) {
        return all(elType, List.of(ff));
    }

    @SuppressWarnings("unchecked")
    static <T> ARFuture<T[]> all(Class<T> elType, Collection<ARFuture<T>> ff) {
        var aa = Flow.flow(ff).toArray(ARFuture.class);
        if (aa.length == 0) {
            return ARFuture.of(RU.cast(Array.newInstance(elType, 0)));
        }
        if (aa.length == 1) {
            return aa[0].map(v -> {
                var res = RU.<T[]>cast(Array.newInstance(elType, 1));
                res[0] = v;
                return res;
            });
        }
        T[] ar = RU.cast(Array.newInstance(elType, aa.length));
        final ARFuture<T[]> res = new ARFutureImpl<>();
        AtomicInteger counter = new AtomicInteger(1);
        int i = 0;
        for (var e : aa) {
            int finalI = i;
            counter.incrementAndGet();
            e.addListener(c -> {
                if (res.isFinalStatus()) {
                    return;
                }
                if (c.isDone()) {
                    AA.setVolatile(ar, finalI, c.getNow());
                } else if (c.isError()) {
                    res.error(c.getError());
                } else if (c.isCanceled()) {
                    res.cancel();
                }
                if (counter.decrementAndGet() == 0) res.done(RU.cast(ar));
            });
            i++;
        }
        if (counter.decrementAndGet() == 0) {
            res.done(RU.cast(ar));
        }
        res.to(v -> {
            for (var e : v) {
                if (e == null) throw new IllegalStateException();
            }
        });
        return res;
    }

    @SuppressWarnings("unchecked")
    static <T> ARFuture<List<T>> all(@NotNull List<ARFuture<T>> list) {
        return RU.cast(all(Object.class, RU.<Collection<ARFuture<Object>>>cast(list)).map(List::of));
    }

    @SuppressWarnings("unchecked")
    static <T1, T2> ARFuture<Tuple2<T1, T2>> all(ARFuture<T1> f1, ARFuture<T2> f2) {
        Object[] ar = new Object[2];
        final ARFuture<Tuple2<T1, T2>> res = new ARFutureImpl<>();
        AtomicInteger counter = new AtomicInteger(2);
        f1.addListener(c -> {
            if (res.isFinalStatus()) {
                return;
            }
            var cc = counter.decrementAndGet();
            if (c.isDone()) {
                AA.setVolatile(ar, 0, c.getNow());
            } else if (c.isError()) {
                res.error(c.getError());
            } else if (c.isCanceled()) {
                res.cancel();
            }
            if (cc == 0) res.done(new Tuple2<>(RU.cast(ar[0]), RU.cast(ar[1])));
        });
        f2.addListener(c -> {
            if (res.isFinalStatus()) {
                return;
            }
            var cc = counter.decrementAndGet();
            if (c.isDone()) {
                AA.setVolatile(ar, 1, c.getNow());
            } else if (c.isError()) {
                res.error(c.getError());
            } else if (c.isCanceled()) {
                res.cancel();
            }
            if (cc == 0) res.done(new Tuple2<>(RU.cast(ar[0]), RU.cast(ar[1])));
        });
        return res;
    }

    @SuppressWarnings("unchecked")
    static <T1, T2, T3> ARFuture<Tuple3<T1, T2, T3>> all(ARFuture<T1> f1, ARFuture<T2> f2, ARFuture<T3> f3) {
        Object[] ar = new Object[3];
        final ARFuture<Tuple3<T1, T2, T3>> res = new ARFutureImpl<>();
        AtomicInteger counter = new AtomicInteger(3);
        f1.addListener(c -> {
            if (res.isFinalStatus()) {
                return;
            }
            var cc = counter.decrementAndGet();
            if (c.isDone()) {
                AA.setVolatile(ar, 0, c.getNow());
            } else if (c.isError()) {
                res.error(c.getError());
            } else if (c.isCanceled()) {
                res.cancel();
            }
            if (cc == 0) res.done(new Tuple3<>(RU.cast(ar[0]), RU.cast(ar[1]), RU.cast(ar[2])));
        });
        f2.addListener(c -> {
            if (res.isFinalStatus()) {
                return;
            }
            var cc = counter.decrementAndGet();
            if (c.isDone()) {
                AA.setVolatile(ar, 1, c.getNow());
            } else if (c.isError()) {
                res.error(c.getError());
            } else if (c.isCanceled()) {
                res.cancel();
            }
            if (cc == 0) res.done(new Tuple3<>(RU.cast(ar[0]), RU.cast(ar[1]), RU.cast(ar[2])));
        });
        f3.addListener(c -> {
            if (res.isFinalStatus()) {
                return;
            }
            var cc = counter.decrementAndGet();
            if (c.isDone()) {
                AA.setVolatile(ar, 2, c.getNow());
            } else if (c.isError()) {
                res.error(c.getError());
            } else if (c.isCanceled()) {
                res.cancel();
            }
            if (cc == 0) res.done(new Tuple3<>(RU.cast(ar[0]), RU.cast(ar[1]), RU.cast(ar[2])));
        });
        return res;
    }

    static <T1, T2> ARFuture<Tuple2<T1, T2>> all(ARFuture<T1> f1, ARFuture<T2> f2, ABiConsumer<T1, T2> task) {
        return all(f1, f2).to(vv -> task.accept(vv.val1(), vv.val2()));
    }

    static <T1, T2, T3> ARFuture<Tuple3<T1, T2, T3>> all(ARFuture<T1> f1, ARFuture<T2> f2, ARFuture<T3> f3, A3Consumer<T1, T2, T3> task) {
        return all(f1, f2, f3).to(vv -> {
            task.accept(vv.val1(), vv.val2(), vv.val3());
        });
    }

    static <T1, T2, R> ARFuture<R> map(ARFuture<T1> f1, ARFuture<T2> f2, A2Function<T1, T2, R> task) {
        return all(f1, f2).map(vv -> task.apply(vv.val1(), vv.val2()));
    }

    static <T1, T2, T3, R> ARFuture<R> map(ARFuture<T1> f1, ARFuture<T2> f2, ARFuture<T3> f3, A3Function<T1, T2, T3, R> task) {
        return all(f1, f2, f3).map(vv -> task.apply(vv.val1(), vv.val2(), vv.val3()));
    }

}