package io.aether.utils.rcollections;

import io.aether.utils.ConcurrentHashMapWithDefault;
import io.aether.utils.ConcurrentHashSet;
import io.aether.utils.RU;
import io.aether.utils.WeakConcurrentHashMap;
import io.aether.utils.futures.AFuture;
import io.aether.utils.futures.ARFutureWithFlag;
import io.aether.utils.slots.EventConsumer;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Array;
import java.util.Map;
import java.util.Set;

/**
 * Implementation of BMap that manages request throttling and exposure of pending requests
 * for integration with a batching/flushing mechanism (like in AetherCloudClient),
 * supporting multiple concurrent request senders.
 *
 * @param <K> The type of the key.
 * @param <V> The type of the value to be retrieved asynchronously.
 */
public class BMapImpl<K, V> implements BMap<K, V> {
    /**
     * Stores all keys that are currently requested by any sender and haven't been resolved.
     */
    final Set<K> allRequests = new ConcurrentHashSet<>();
    final EventConsumer<Update<K, V>> valueUpdate = new EventConsumer<>();
    /**
     * Tracks requests per sender object. Uses a WeakConcurrentHashMap to allow senders to be garbage collected.
     */
    final Map<Object, Sender> senders = new WeakConcurrentHashMap<>();

    /**
     * Internal storage for all data and the associated future.
     */
    final Map<K, ARFutureWithFlag<V>> data;

    /**
     * Constructs a new BMapImpl.
     *
     * @param initialCapacity The initial capacity for the internal map.
     * @param name A descriptive name (not used in this implementation).
     * @param timeoutMs The request timeout duration in milliseconds (not used in this implementation).
     */
    public BMapImpl(int initialCapacity, String name, long timeoutMs) {
        data = new ConcurrentHashMapWithDefault<>(k -> {
            ARFutureWithFlag<V> future = new ARFutureWithFlag<>();

            future.to(v -> {
                removeRequest(k);
                valueUpdate.fire(new Update<>(k, v, null));
            });

            addRequest(k);

            return future;
        });
    }

    /**
     * Removes the key from the global request pool and all sender-specific pools.
     * This is called when the future is successfully resolved or explicitly removed/errored.
     * @param key The key to remove.
     * @return true if the key was present in the global pool, false otherwise.
     */
    private boolean removeRequest(K key) {
        var res = allRequests.remove(key);
        for (var s : senders.values()) {
            s.requests.remove(key);
        }
        return res;
    }

    /**
     * Adds the key to the global request pool and all existing sender-specific pools.
     * This ensures any new or existing sender will see this key as pending.
     * @param key The key to add.
     */
    void addRequest(K key) {
        allRequests.add(key);
        for (var s : senders.values()) {
            s.requests.add(key);
        }
    }

    @Override
    public @NotNull ARFutureWithFlag<V> getFuture(@NotNull K key) {
        return data.get(key);
    }

    @Override
    public @NotNull Set<K> getPendingRequests() {
        return allRequests;
    }

    @Override
    public void putResolved(@NotNull K key, @NotNull V value) {
        data.get(key).tryDone(value);
    }

    @Override
    public void putError(@NotNull K key, @NotNull Throwable error) {
        ARFutureWithFlag<V> future = data.get(key);
        future.tryError(error);
        removeRequest(key);
    }

    @Override
    public EventConsumer<Update<K, V>> forValueUpdate() {
        return valueUpdate;
    }

    /**
     * Helper to retrieve or create a Sender object associated with the network sender.
     * @param k The sender object.
     * @return The Sender instance.
     */
    Sender getSender(Object k) {
        return senders.computeIfAbsent(k, kk -> new Sender());
    }

    @Override
    public K[] getRequestsFor(Class<K> elementType, Object sender) {
        return getSender(sender).extract(elementType);
    }

    @Override
    public boolean isRequests() {
        return !allRequests.isEmpty();
    }

    @Override
    public boolean isRequestsFor(@NotNull Object sender) {
        return !getSender(sender).requests.isEmpty();
    }

    @Override
    public AFuture destroy(boolean force) {
        return AFuture.completed();
    }

    @Override
    public EventConsumer<Update<K, ARFutureWithFlag<V>>> forUpdate() {
        return new EventConsumer<>();
    }

    @Override
    public EventConsumer<Entry<K, ARFutureWithFlag<V>>> forRemove() {
        return new EventConsumer<>();
    }

    @Override
    public int size() {
        return data.size();
    }

    @Override
    public boolean isEmpty() {
        return data.isEmpty();
    }

    @Override
    public boolean containsKey(Object key) {
        return data.containsKey(key);
    }

    @Override
    public boolean containsValue(Object value) {
        throw new UnsupportedOperationException("containsValue is not supported on future maps");
    }

    @Override
    public ARFutureWithFlag<V> get(Object key) {
        return getFuture(RU.cast(key));
    }

    @Override
    public @Nullable ARFutureWithFlag<V> put(K key, ARFutureWithFlag<V> value) {
        throw new UnsupportedOperationException("Cannot put ARFutureWithFlag directly to BMapImpl");
    }

    @Override
    public ARFutureWithFlag<V> remove(Object key) {
        removeRequest(RU.cast(key));
        return data.remove(key);
    }

    @Override
    public @NotNull RSet<K> keySet() {
        return RSet.of(data.keySet());
    }

    @Override
    public @NotNull RCollection<ARFutureWithFlag<V>> values() {
        return RCollection.of(data.values());
    }

    @Override
    public @NotNull RSet<Entry<K, ARFutureWithFlag<V>>> entrySet() {
        return RSet.of(data.entrySet());
    }

    /**
     * Represents a single object responsible for sending network requests (a 'flush' mechanism).
     */
    class Sender {
        /**
         * The set of keys this specific sender needs to fetch.
         */
        final Set<K> requests = new ConcurrentHashSet<>();

        {
            requests.addAll(allRequests);
        }

        /**
         * Extracts all pending keys for this sender, clears the internal set, and returns the array.
         * This is the 'flush' operation for this sender.
         *
         * @param elementType The Class of the key type for array creation.
         * @return An array of keys (K[]) that are queued for network submission by this sender.
         */
        public K[] extract(Class<K> elementType) {
            ObjectArrayList<K> r = new ObjectArrayList<>(requests);
            requests.clear();
            return r.toArray(RU.<K[]>cast(Array.newInstance(elementType, r.size())));
        }
    }
}