package io.aether.utils.rcollections;

import io.aether.utils.AString;
import io.aether.utils.ConcurrentHashMapWithDefault;
import io.aether.utils.RU;
import io.aether.utils.futures.ARFuture;
import io.aether.utils.futures.ARFutureWithFlag;
import io.aether.utils.interfaces.ABiFunction;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.slots.EventConsumer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.AbstractMap;
import java.util.Iterator;
import java.util.Map;

public interface RMap<K, V> extends Map<K, V> {

    EventConsumer<Update<K, V>> forUpdate();

    EventConsumer<Entry<K, V>> forRemove();

    default RFMap<K, V> mapToFutures() {
        var self = this;
        Map<K, ARFutureWithFlag<V>> map2 = new ConcurrentHashMapWithDefault<>(k -> {
            var res = new ARFutureWithFlag<V>();
            res.to(v -> {
                if (res.tryRequest()) {
                    self.put(k, v);
                }
            });
            return res;
        });
        return new RFMap<>() {
            final EventConsumer<Update<K, ARFutureWithFlag<V>>> forUpdate = new EventConsumer<>();
            final EventConsumer<Entry<K, ARFutureWithFlag<V>>> forRemove = new EventConsumer<>();

            {
                for (var e : self.entrySet()) {
                    if (e.getValue() != null) {
                        var f = new ARFutureWithFlag<V>();
                        f.done(e.getValue());
                        f.tryRequest();
                        map2.put(e.getKey(), f);
                    }
                }
                self.forUpdate().add(u -> {
                    var f = map2.get(u.key);
                    f.tryRequest();
                    f.tryDone(u.newValue);
                });
                self.forRemove().add(u -> {
                    var f = map2.get(u.getKey());
                    f.cancel();
                    forRemove.fire(new AbstractMap.SimpleEntry<>(u.getKey(), f));
                });
            }

            @Override
            public EventConsumer<Update<K, ARFutureWithFlag<V>>> forUpdate() {
                return forUpdate;
            }

            @Override
            public EventConsumer<Entry<K, ARFutureWithFlag<V>>> forRemove() {
                return forRemove;
            }

            @Override
            public @NotNull RSet<K> keySet() {
                return RSet.of(map2.keySet());
            }

            @Override
            public @NotNull RCollection<ARFutureWithFlag<V>> values() {
                return RCollection.of(RU.cast(map2.values()));
            }

            @Override
            public @NotNull RSet<Entry<K, ARFutureWithFlag<V>>> entrySet() {
                return RSet.of(RU.cast(map2.entrySet()));
            }

            @Override
            public int size() {
                return map2.size();
            }

            @Override
            public boolean isEmpty() {
                return map2.isEmpty();
            }

            @Override
            public boolean containsKey(Object key) {
                return map2.containsKey(key);
            }

            @Override
            public boolean containsValue(Object value) {
                return map2.containsValue(value);
            }

            @Override
            public ARFutureWithFlag<V> get(Object key) {
                return map2.get(key);
            }

            @Override
            public @Nullable ARFutureWithFlag<V> put(K key, ARFutureWithFlag<V> value) {
                return map2.compute(key, (k, v) -> {
                    if (v != null) {
                        value.to(v);
                        return v;
                    } else {
                        return RU.cast(value);
                    }
                });
            }

            @Override
            public ARFutureWithFlag<V> remove(Object key) {
                var res = map2.remove(key);
                if (res != null) {
                    res.to(() -> self.remove(key));
                }
                return res;
            }
        };
    }

    default RSet<V> mapToValues(AFunction<V, K> keyGetter) {
        var self = this;
        return new RSet<V>() {
            final EventConsumer<V> fAdd = new EventConsumer<>();
            final EventConsumer<V> fRemove = new EventConsumer<>();

            @Override
            public void toString(AString sb) {
                boolean first = true;
                for (var e : this) {
                    if (first) {
                        first = false;
                    } else {
                        sb.add(", ");
                    }
                    sb.add(e);
                }
            }

            @Override
            public String toString() {
                return toString2();
            }

            @Override
            public EventConsumer<V> forAdd() {
                return fAdd;
            }

            @Override
            public EventConsumer<V> forRemove() {
                return fRemove;
            }

            @Override
            public int size() {
                return self.size();
            }

            @Override
            public boolean isEmpty() {
                return self.isEmpty();
            }

            @Override
            public boolean contains(Object o) {
                return self.containsValue(RU.<V>cast(o));
            }

            @Override
            public @NotNull Iterator<V> iterator() {
                return self.values().iterator();
            }

            @Override
            public @NotNull Object @NotNull [] toArray() {
                return self.values().toArray();
            }

            @Override
            public @NotNull <T> T @NotNull [] toArray(@NotNull T @NotNull [] a) {
                return self.values().toArray(a);
            }

            @Override
            public boolean add(V v) {
                var old = self.put(keyGetter.apply(v), v);
                return old != v;
            }

            @Override
            public boolean remove(Object o) {
                return self.remove(keyGetter.apply(RU.cast(o)), RU.<V>cast(o));
            }
        };
    }

    default void link(RMap<K, V> other) {
        forUpdate().add(e -> other.put(e.key, e.newValue));
        forRemove().add(e -> other.remove(e.getKey()));
        other.forUpdate().add(e -> put(e.key, e.newValue));
        other.forRemove().add(e -> remove(e.getKey()));
        other.putAll(this);
        this.putAll(other);
    }

    default <V2> RMap<K, V2> mapVal(AFunction<V, V2> v1ToV2, AFunction<V2, V> v2ToV1) {
        return map(AFunction.stub(), AFunction.stub(), v1ToV2, v2ToV1);
    }

    default <V2> RMap<K, V2> mapVal(AFunction<V, V2> v1ToV2) {
        return map(AFunction.stub(), AFunction.stub(), v -> {
            if (v == null) return null;
            return v1ToV2.apply(v);
        }, c -> {
            throw new UnsupportedOperationException();
        });
    }

    default <K2, V2> RMap<K2, V2> map(AFunction<K, K2> k1ToK2, AFunction<K2, K> k2ToK1, AFunction<V, V2> v1ToV2, AFunction<V2, V> v2ToV1) {
        var self = this;
        return new RMap<>() {
            final EventConsumer<Update<K2, V2>> forUpdate = new EventConsumer<>();
            final EventConsumer<Entry<K2, V2>> forRemove = new EventConsumer<>();

            {
                self.forUpdate().add(u -> {
                    forUpdate.fire(new Update<>(k1ToK2.apply(u.key), v1ToV2.apply(u.newValue), v1ToV2.apply(u.oldValue)));
                });
                self.forRemove().add(u -> {
                    forRemove.fire(new AbstractMap.SimpleEntry<>(k1ToK2.apply(u.getKey()), v1ToV2.apply(u.getValue())) {
                        @Override
                        public V2 setValue(V2 value) {
                            return v1ToV2.apply(u.setValue(v2ToV1.apply(value)));
                        }
                    });
                });
            }

            @Override
            public EventConsumer<Update<K2, V2>> forUpdate() {
                return forUpdate;
            }

            @Override
            public EventConsumer<Entry<K2, V2>> forRemove() {
                return forRemove;
            }

            @Override
            public @NotNull RSet<K2> keySet() {
                return self.keySet().map(k1ToK2, k2ToK1);
            }

            @Override
            public @NotNull RCollection<V2> values() {
                return self.values().map(v1ToV2, v -> {
                    throw new UnsupportedOperationException();
                });
            }

            @Override
            public @NotNull RSet<Entry<K2, V2>> entrySet() {
                return self.entrySet().map(new AFunction<Entry<K, V>, Entry<K2, V2>>() {
                    @Override
                    public Entry<K2, V2> apply2(Entry<K, V> entry) throws Throwable {
                        return new Entry<>() {
                            @Override
                            public K2 getKey() {
                                return k1ToK2.apply(entry.getKey());
                            }

                            @Override
                            public V2 getValue() {
                                return v1ToV2.apply(entry.getValue());
                            }

                            @Override
                            public V2 setValue(V2 value) {
                                var old = entry.setValue(v2ToV1.apply(value));
                                return v1ToV2.apply(old);
                            }
                        };
                    }
                }, new AFunction<Entry<K2, V2>, Entry<K, V>>() {
                    @Override
                    public Entry<K, V> apply2(Entry<K2, V2> e) throws Throwable {
                        return new Entry<>() {
                            @Override
                            public K getKey() {
                                return k2ToK1.apply(e.getKey());
                            }

                            @Override
                            public V getValue() {
                                return v2ToV1.apply(e.getValue());
                            }

                            @Override
                            public V setValue(V value) {
                                return v2ToV1.apply(e.setValue(v1ToV2.apply(value)));
                            }
                        };
                    }
                });
            }

            @Override
            public int size() {
                return self.size();
            }

            @Override
            public boolean isEmpty() {
                return self.isEmpty();
            }

            @Override
            public boolean containsKey(Object key) {
                return self.containsKey(k2ToK1.apply(RU.cast(key)));
            }

            @Override
            public boolean containsValue(Object value) {
                return self.containsValue(v2ToV1.apply(RU.cast(value)));
            }

            @Override
            public V2 get(Object key) {
                return v1ToV2.apply(self.get(k2ToK1.apply(RU.cast(key))));
            }

            @Override
            public @Nullable V2 put(K2 key, V2 value) {
                return v1ToV2.apply(self.put(k2ToK1.apply(key), v2ToV1.apply(value)));
            }

            @Override
            public V2 remove(Object key) {
                return v1ToV2.apply(self.remove(k2ToK1.apply(RU.cast(key))));
            }
        };
    }

    default <K2> RMap<K2, V> mapKey(ABiFunction<K, V, K2> k1ToK2) {
        var res = RCol.<K2, V>map();
        forRemove().add(e -> {
            var k2 = k1ToK2.apply(e.getKey(), e.getValue());
            res.remove(k2);
        });
        forUpdate().add(e -> {
            var k2 = k1ToK2.apply(e.key, e.newValue);
            res.put(k2, e.newValue);
        });
        return res;
    }

    @Override
    default void putAll(@NotNull Map<? extends K, ? extends V> m) {
        for (var e : m.entrySet()) {
            put(e.getKey(), e.getValue());
        }
    }

    @Override
    default void clear() {
        var it = entrySet().iterator();
        while (it.hasNext()) {
            it.next();
            it.remove();
        }
    }

    @Override
    @NotNull RSet<K> keySet();

    @Override
    @NotNull RCollection<V> values();

    @Override
    @NotNull RSet<Entry<K, V>> entrySet();

    static <K, V> RMap<K, V> of(Map<K, V> src) {
        if (src instanceof RMap) return RU.cast(src);
        return new RMapBySrc<>(src);
    }

    class Update<K, V> {
        public final K key;
        public final V newValue;
        public final V oldValue;

        public Update(K key, V newValue, V oldValue) {
            this.key = key;
            this.newValue = newValue;
            this.oldValue = oldValue;
        }

        public Update(Entry<K, V> e) {
            key = e.getKey();
            newValue = e.getValue();
            oldValue = e.getValue();
        }
    }
}
