package io.aether.utils.rcollections;

import io.aether.utils.AString;
import io.aether.utils.RU;
import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.slots.EventConsumer;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Iterator;
import java.util.Queue;

public interface RQueue<T> extends RCollection<T>, Queue<T> {

    @Override
    default boolean retainAll(@NotNull Collection<?> c) {
        return RCollection.super.retainAll(c);
    }

    @Override
    default void clear() {
        RCollection.super.clear();
    }

    @Override
    default boolean addAll(@NotNull Collection<? extends T> c) {
        return RCollection.super.addAll(c);
    }

    @Override
    default boolean containsAll(@NotNull Collection<?> c) {
        return RCollection.super.containsAll(c);
    }

    @Override
    default boolean removeAll(@NotNull Collection<?> c) {
        return RCollection.super.removeAll(c);
    }

    default <T2> void link(RQueue<T2> other, AFunction<T, T2> f, AFunction<T2, T> back) {
        forAdd().add(v -> other.add(f.apply(v)));
        forRemove().add(v -> other.remove(f.apply(v)));
        other.forAdd().add(v -> add(back.apply(v)));
        other.forRemove().add(v -> remove(back.apply(v)));
        for (var e : other) {
            add(back.apply(e));
        }
        for (var e : this) {
            other.add(f.apply(e));
        }
    }

    default <T2> @NotNull RQueue<T2> map(AFunction<T, T2> f, AFunction<T2, T> back) {
        var self = this;
        return new RQueue<>() {
            final EventConsumer<T2> forAdd = new EventConsumer<>();
            final EventConsumer<T2> forRemove = new EventConsumer<>();

            @Override
            public void toString(AString sb) {
                sb.add("queue ").add(size()).add(" (");
                boolean first=true;
                for(var e:this){
                    if(first){
                        first=false;
                    }else{
                        sb.add(", ");
                    }
                    sb.add(e);
                }
                sb.add(")");
            }

            @Override
            public String toString() {
                return toString2();
            }

            @Override
            public boolean offer(T2 t2) {
                return self.offer(back.apply(t2));
            }

            @Override
            public T2 remove() {
                return f.apply(self.remove());
            }

            @Override
            public T2 poll() {
                var v=self.poll();
                if(v==null)return null;
                return f.apply(v);
            }

            @Override
            public T2 element() {
                return f.apply(self.element());
            }

            @Override
            public T2 peek() {
                return f.apply(self.peek());
            }

            @Override
            public EventConsumer<T2> forAdd() {
                return forAdd;
            }

            @Override
            public EventConsumer<T2> forRemove() {
                return forRemove;
            }

            @Override
            public int size() {
                return self.size();
            }

            @Override
            public boolean isEmpty() {
                return self.isEmpty();
            }

            @Override
            public boolean contains(Object o) {
                return self.contains(back.apply(RU.cast(o)));
            }

            @Override
            public @NotNull Iterator<T2> iterator() {
                var it = self.iterator();
                return new Iterator<>() {
                    @Override
                    public boolean hasNext() {
                        return it.hasNext();
                    }

                    @Override
                    public T2 next() {
                        return f.apply(it.next());
                    }
                };
            }

            @Override
            public @NotNull Object @NotNull [] toArray() {
                return Flow.flow(self).map(f).toArray();
            }

            @Override
            public @NotNull <T5> T5 @NotNull [] toArray(@NotNull T5 @NotNull [] a) {
                return Flow.flow(self).map(f).toList().toArray(a);
            }

            @Override
            public boolean add(T2 t2) {
                return self.add(back.apply(t2));
            }

            @Override
            public boolean remove(Object o) {
                return self.remove(back.apply(RU.cast(o)));
            }

        };
    }

    static <E> RQueue<E> of(Queue<E> src) {
        if (src instanceof RQueue) return RU.cast(src);
        return new RQueueBySrc<>(src);
    }

}

