package io.aether.utils.slots;

import io.aether.utils.CTypeI;
import io.aether.utils.TaskRun;
import io.aether.utils.futures.AFuture;
import io.aether.utils.interfaces.ARunnable;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import it.unimi.dsi.fastutil.objects.ObjectSets;

import java.lang.invoke.VarHandle;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;

public class EventSourceRunnable {
    private static final VarHandle OFFSET_ENABLED = CTypeI.of(EventSourceRunnable.class).getFieldVarHandle("enabled");
    private static final Executor DEF = Runnable::run;
    private final Queue<TaskRun> tasks = new ConcurrentLinkedQueue<>();
    private final Set<Runnable> tasksPermanent = ObjectSets.synchronize(new ObjectOpenHashSet<>());
    private volatile boolean enabled;

    public AFuture toFuture() {
        if (isEnabled()) return AFuture.completed();
        AFuture f = AFuture.make();
        run(f::done);
        return f;
    }

    public void enabled(boolean f) {
        if (f) enable();
        else disable();
    }

    public boolean enable() {
        if (!OFFSET_ENABLED.compareAndSet(this, false, true)) return false;
        for (var task : tasksPermanent) {
            if (!enabled) break;
            task.run();
        }
        runTasks();
        return true;
    }

    public void disable() {
        enabled = false;
    }

    private void runTasks() {
        if (!enabled) return;
        while (true) {
            var task = tasks.poll();
            if (task == null) break;
            task.run();
        }
    }

    public void run(Executor executor, ARunnable task) {
        if (isEnabled()) {
            executor.execute(task);
        } else {
            TaskRun t = new TaskRun() {
                @Override
                public String toString() {
                    return task.toString();
                }

                @Override
                protected void runTask() {
                    executor.execute(task);
                }
            };
            tasks.add(t);
            runTasks();
        }
    }

    public void run(ARunnable task) {
        if (isEnabled()) {
            task.run();
        } else {
            run(DEF, task);
        }
    }

    public void runPermanent(ARunnable task) {
        tasksPermanent.add(task);
        if (enabled) {
            task.run();
        }
    }

    public boolean isEnabled() {
        return enabled;
    }

    public void reset() {
        enabled = false;
        tasks.clear();
        tasksPermanent.clear();
    }
}
