package io.aether.logger;

import io.aether.utils.ToString;
import io.aether.utils.interfaces.*;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.concurrent.Executor;

/**
 * Aether Logger - powerful logging system with context support, filtering and formatting.
 *
 * Main features:
 * - Logging with levels (TRACE, DEBUG, INFO, WARN, ERROR)
 * - Contextual logging with automatic lifecycle management
 * - Log filtering by various criteria
 * - Colored console output support
 * - Wrappers for code execution in logging context
 * - Thread safety
 */
public class Log {
    public static final boolean TRACE = true;
    public static final String TIME = "logTime";
    public static final String LEVEL = "logLevel";
    public static final String SYSTEM_COMPONENT = "SystemComponent";
    public static final String MSG = "logMessage";
    public static final String EXCEPTION_STR = "exception";

    /**
     * Global logger filter for all log messages
     */
    public static final LogFilter filter = LogImpl.filter;
    public static final DateFormat DATE_FORMAT = new SimpleDateFormat("MM:dd:HH:mm:ss.SSSS");

    /**
     * Global enabled/disabled state of the logger
     */
    public static boolean ENABLED = true;

    // ==================== CONTEXT MANAGEMENT ====================

    /**
     * Stores a named node for later retrieval and modification
     * @param name Unique name for the node
     * @param n LNode to store
     */
    public static void storeNode(String name, LNode n) {
        LogImpl.storeNode(name, n);
    }

    /**
     * Sets a value in a previously stored named node
     * @param nameNode Name of the stored node
     * @param key Key to set
     * @param value Value to set
     */
    public static void setStored(String nameNode, String key, Object value) {
        LogImpl.setStored(nameNode, key, value);
    }

    /**
     * Gets the current context stack for the current thread
     * @return Deque of LNodes representing the context stack
     */
    public static java.util.Deque<LNode> getStack() {
        return LogImpl.getStack();
    }

    /**
     * Creates a new logging context by combining current context with additional data
     * @param data Key-value pairs for the context
     * @return Combined LNode context
     */
    public static LNode createContext(Object... data) {
        return LogImpl.createContext(data);
    }

    /**
     * Creates a new logging context by combining current context with additional nodes
     * @param data LNodes to combine with current context
     * @return Combined LNode context
     */
    public static LNode createContext(LNode... data) {
        return LogImpl.createContext(data);
    }

    /**
     * Creates a context from the current stack only
     * @return Current context LNode
     */
    public static LNode createContext() {
        return LogImpl.createContext();
    }

    /**
     * Creates an auto-closeable context that will be automatically popped when closed
     * @param data Key-value pairs for the context
     * @return AutoCloseable context that should be used in try-with-resources
     */
    public static LogAutoClose context(Object... data) {
        return LogImpl.context(data);
    }

    /**
     * Creates a context with a parent node and additional data
     * @param parent Parent LNode
     * @param addData Additional key-value pairs
     * @return AutoCloseable context
     */
    public static LogAutoClose context(LNode parent, Object... addData) {
        return LogImpl.context(parent, addData);
    }

    /**
     * Creates a context from multiple LNodes
     * @param nn Array of LNodes to form the context
     * @return AutoCloseable context
     */
    public static LogAutoClose context(LNode... nn) {
        return LogImpl.context(nn);
    }

    /**
     * Creates a context from a single LNode
     * @param n LNode to use as context
     * @return AutoCloseable context
     */
    public static LogAutoClose context(LNode n) {
        return LogImpl.context(n);
    }

    /**
     * Pushes a node onto the context stack
     * @param n LNode to push
     */
    public static void push(LNode n) {
        LogImpl.push(n);
    }

    /**
     * Gets the current context from the stack
     * @return Current context LNode (combination of all nodes in stack)
     */
    public static LNode get() {
        return LogImpl.get();
    }

    /**
     * Pops a node from the context stack
     * @param n Expected node to pop (for validation)
     */
    public static void pop(LNode n) {
        LogImpl.pop(n);
    }

    // ==================== LOGGING METHODS ====================

    /**
     * Logs a custom LNode
     * @param node LNode to log
     */
    public static void log(LNode node) {
        LogImpl.log(node);
    }

    /**
     * Logs a TRACE level message with formatted object
     * @param msg Message implementing ToString interface
     * @param data Additional key-value pairs
     * @return Created LNode
     */
    public static LNode trace(ToString msg, Object... data) {
        return LogImpl.trace(msg, data);
    }

    /**
     * Logs a TRACE level message with string and LNode data
     * @param msg Message string
     * @param data LNodes with additional data
     * @return Created LNode
     */
    public static LNode trace(String msg, LNode... data) {
        return LogImpl.trace(msg, data);
    }

    /**
     * Logs a TRACE level message with string and single LNode
     * @param msg Message string
     * @param data LNode with additional data
     * @return Created LNode
     */
    public static LNode trace(String msg, LNode data) {
        return LogImpl.trace(msg, data);
    }

    /**
     * Logs a TRACE level message with string and key-value pairs
     * @param msg Message string
     * @param data Key-value pairs
     * @return Created LNode
     */
    public static LNode trace(String msg, Object... data) {
        return LogImpl.trace(msg, data);
    }

    /**
     * Logs a DEBUG level message with formatted object
     * @param msg Message implementing ToString interface
     * @param data Additional key-value pairs
     * @return Created LNode
     */
    public static LNode debug(ToString msg, Object... data) {
        return LogImpl.debug(msg, data);
    }

    /**
     * Logs a DEBUG level message with string and key-value pairs
     * @param msg Message string
     * @param data Key-value pairs
     * @return Created LNode
     */
    public static LNode debug(String msg, Object... data) {
        return LogImpl.debug(msg, data);
    }

    /**
     * Logs a DEBUG level message with string and LNode data
     * @param msg Message string
     * @param data LNodes with additional data
     * @return Created LNode
     */
    public static LNode debug(String msg, LNode... data) {
        return LogImpl.debug(msg, data);
    }

    /**
     * Logs a DEBUG level message with string and single LNode
     * @param msg Message string
     * @param data LNode with additional data
     * @return Created LNode
     */
    public static LNode debug(String msg, LNode data) {
        return LogImpl.debug(msg, data);
    }

    /**
     * Logs an INFO level message with formatted object
     * @param msg Message implementing ToString interface
     * @param data Additional key-value pairs
     * @return Created LNode
     */
    public static LNode info(ToString msg, Object... data) {
        return LogImpl.info(msg, data);
    }

    /**
     * Logs an INFO level message with string and key-value pairs
     * @param msg Message string
     * @param data Key-value pairs
     * @return Created LNode
     */
    public static LNode info(String msg, Object... data) {
        return LogImpl.info(msg, data);
    }

    /**
     * Logs an INFO level message with string and LNode data
     * @param msg Message string
     * @param data LNodes with additional data
     * @return Created LNode
     */
    public static LNode info(String msg, LNode... data) {
        return LogImpl.info(msg, data);
    }

    /**
     * Logs an INFO level message with string and single LNode
     * @param msg Message string
     * @param data LNode with additional data
     * @return Created LNode
     */
    public static LNode info(String msg, LNode data) {
        return LogImpl.info(msg, data);
    }

    /**
     * Logs a WARN level message with formatted object
     * @param msg Message implementing ToString interface
     * @param data Additional key-value pairs
     * @return Created LNode
     */
    public static LNode warn(ToString msg, Object... data) {
        return LogImpl.warn(msg, data);
    }

    /**
     * Logs a WARN level message with string and key-value pairs
     * @param msg Message string
     * @param data Key-value pairs
     * @return Created LNode
     */
    public static LNode warn(String msg, Object... data) {
        return LogImpl.warn(msg, data);
    }

    /**
     * Logs a WARN level message with string and LNode data
     * @param msg Message string
     * @param data LNodes with additional data
     * @return Created LNode
     */
    public static LNode warn(String msg, LNode... data) {
        return LogImpl.warn(msg, data);
    }

    /**
     * Logs a WARN level message with string and single LNode
     * @param msg Message string
     * @param data LNode with additional data
     * @return Created LNode
     */
    public static LNode warn(String msg, LNode data) {
        return LogImpl.warn(msg, data);
    }

    /**
     * Logs a WARN level message with exception
     * @param msg Message string
     * @param throwable Exception to log
     * @param data Additional key-value pairs
     * @return Created LNode
     */
    public static LNode warn(String msg, Throwable throwable, Object... data) {
        return LogImpl.warn(msg, throwable, data);
    }

    /**
     * Logs an ERROR level message with exception
     * @param msg Message string
     * @param throwable Exception to log
     * @param data Additional key-value pairs
     * @return Created LNode
     */
    public static LNode error(String msg, Throwable throwable, Object... data) {
        return LogImpl.error(msg, throwable, data);
    }

    /**
     * Logs an ERROR level message without exception
     * @param msg Message string
     * @param data Additional key-value pairs
     * @return Created LNode
     */
    public static LNode error(String msg, Object... data) {
        return LogImpl.error(msg, data);
    }

    /**
     * Logs an ERROR level message from exception
     * @param throwable Exception to log
     * @param data Additional key-value pairs
     * @return Created LNode
     */
    public static LNode error(Throwable throwable, Object... data) {
        return LogImpl.error(throwable, data);
    }

    // ==================== FILTER MANAGEMENT ====================

    /**
     * Adds a negative filter (excludes logs that match the predicate)
     * @param p Predicate to exclude matching logs
     */
    public static void addFilterNot(APredicate<LNode> p) {
        LogImpl.addFilterNot(p);
    }

    /**
     * Adds a positive filter (includes only logs that match the predicate)
     * @param p Predicate to include matching logs
     */
    public static void addFilter(APredicate<LNode> p) {
        LogImpl.addFilter(p);
    }

    // ==================== OUTPUT FORMATTING ====================

    /**
     * Creates a colored console printer with default filter
     * @return LogPrinter instance for colored console output
     */
    public static LogPrinter printConsoleColored() {
        return LogImpl.printConsoleColored();
    }

    /**
     * Creates a colored console printer with custom filter
     * @param filter Custom filter for the printer
     * @return LogPrinter instance for colored console output
     */
    public static LogPrinter printConsoleColored(LogFilter filter) {
        return LogImpl.printConsoleColored(filter);
    }

    /**
     * Creates a plain console printer with custom filter
     * @param filter Custom filter for the printer
     * @return LogPrinter instance for plain console output
     */
    public static LogPrinter printPlainConsole(LogFilter filter) {
        return LogImpl.printPlainConsole(filter);
    }

    // ==================== NODE CREATION ====================

    /**
     * Creates an LNode from key-value pairs
     * @param data Alternating key-value pairs
     * @return Created LNode
     */
    public static LNode of(Object... data) {
        return LogImpl.of(data);
    }

    /**
     * Returns the LNode as-is (identity operation)
     * @param data LNode to return
     * @return Same LNode
     */
    public static LNode of(LNode data) {
        return LogImpl.of(data);
    }

    /**
     * Creates an LNode from separate key and value arrays
     * @param keys Array of keys
     * @param vals Array of values
     * @return Created LNode
     */
    public static LNode of2(Object[] keys, Object[] vals) {
        return LogImpl.of2(keys, vals);
    }

    /**
     * Creates an LNode from separate key and value lists
     * @param keys List of keys
     * @param vals List of values
     * @return Created LNode
     */
    public static LNode of2(java.util.List<? extends CharSequence> keys, java.util.List<?> vals) {
        return LogImpl.of2(keys, vals);
    }

    /**
     * Creates an LNode from a Map
     * @param vals Map with key-value pairs
     * @return Created LNode
     */
    public static LNode ofMap(java.util.Map<String, ?> vals) {
        return LogImpl.ofMap(vals);
    }

    /**
     * Creates a composite LNode from multiple LNodes
     * @param data Array of LNodes to combine
     * @return Composite LNode
     */
    public static LNode of(LNode... data) {
        return LogImpl.of(data);
    }

    // ==================== UTILITY WRAPPERS ====================

    /**
     * Wraps an Executor to preserve logging context in executed tasks
     * @param executor Original executor to wrap
     * @return Wrapped executor that preserves logging context
     */
    public static Executor wrapExecutor(Executor executor) {
        return LogImpl.wrapExecutor(executor);
    }

    /**
     * Wraps a Runnable with logging context
     * @param t Runnable to wrap
     * @param data Context data for the wrapper
     * @return Wrapped Runnable
     */
    public static ARunnable wrapRunnable(ARunnable t, Object... data) {
        return LogImpl.wrapRunnable(t, data);
    }

    /**
     * Wraps a Supplier with current logging context
     * @param t Supplier to wrap
     * @return Wrapped Supplier
     */
    public static <T> ASupplier<T> wrapSupplier(ASupplier<T> t) {
        return LogImpl.wrapSupplier(t);
    }

    /**
     * Wraps a Supplier with current logging context
     * @param t Supplier to wrap
     * @return Wrapped Supplier
     */
    public static <T> ASupplier<T> wrap(ASupplier<T> t) {
        return LogImpl.wrap(t);
    }

    /**
     * Wraps a Runnable with current logging context
     * @param t Runnable to wrap
     * @return Wrapped Runnable
     */
    public static ARunnable wrap(ARunnable t) {
        return LogImpl.wrap(t);
    }

    /**
     * Wraps a Consumer with current logging context
     * @param t Consumer to wrap
     * @return Wrapped Consumer
     */
    public static <T> AConsumer<T> wrap(AConsumer<T> t) {
        return LogImpl.wrap(t);
    }

    /**
     * Wraps a BiConsumer with current logging context
     * @param t BiConsumer to wrap
     * @return Wrapped BiConsumer
     */
    public static <T, T2> ABiConsumer<T, T2> wrap(ABiConsumer<T, T2> t) {
        return LogImpl.wrap(t);
    }

    /**
     * Wraps a Function with current logging context
     * @param t Function to wrap
     * @return Wrapped Function
     */
    public static <T, R> AFunction<T, R> wrap(AFunction<T, R> t) {
        return LogImpl.wrap(t);
    }

    // ==================== GLOBAL CONTROLS ====================

    /**
     * Disables all logging globally
     */
    public static void loggerOff() {
        LogImpl.loggerOff();
    }

    /**
     * Enables all logging globally
     */
    public static void loggerOn() {
        LogImpl.loggerOn();
    }

    /**
     * Adds a listener for log events with filtering
     * @param filter Filter for which events to receive
     * @param consumer Consumer to handle log events
     * @return AutoCloseable to remove the listener
     */
    public static AutoCloseable addListener(LogFilter filter, AConsumer<LNode> consumer) {
        return LogImpl.addListener(filter, consumer);
    }

    /**
     * Log levels
     */
    public enum Level {
        TRACE,
        DEBUG,
        INFO,
        WARN,
        ERROR,
    }

    /**
     * Annotation for tagging loggers
     */
    @java.lang.annotation.Retention(java.lang.annotation.RetentionPolicy.RUNTIME)
    @java.lang.annotation.Inherited
    public @interface Tag {
        String value();
    }

    /**
     * AutoCloseable interface for logging contexts
     */
    public interface LogAutoClose extends AutoCloseable {
        LogAutoClose EMPTY = () -> {};

        @Override
        void close();
    }
}