package io.aether.logger;

import io.aether.utils.AString;
import io.aether.utils.CTypeI;
import io.aether.utils.RU;
import io.aether.utils.ToString;
import io.aether.utils.interfaces.*;
import io.aether.utils.slots.EventConsumer;
import org.jetbrains.annotations.NotNull;

import java.lang.ref.WeakReference;
import java.util.*;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Implementation class for Aether Logger containing all internal logic.
 * This class should not be used directly - use Log class instead.
 */
class LogImpl {
    static final LogFilter filter = new LogFilter();
    static final EventConsumer<LNode> onEvent = new EventConsumer<>();
    static final AtomicLong ID_COUNTER = new AtomicLong();
    static final AtomicReference<LogPrinter[]> consolePrinter = new AtomicReference<>();
    static final Iterator<String> EMPTY_ITERATOR = new Iterator<>() {
        @Override
        public boolean hasNext() {
            return false;
        }

        @Override
        public String next() {
            return null;
        }
    };
    private static final Set<String> SET_STUB = new SetStub();
    private static final ThreadLocal<Map<String, WeakReference<LNode>>> namedNodes = ThreadLocal.withInitial(HashMap::new);
    public static boolean ENABLED = true;
    public static ThreadLocal<Deque<LNode>> stack = ThreadLocal.withInitial(ArrayDeque::new);

    public static void storeNode(String name, LNode n) {
        if (!ENABLED) return;
        namedNodes.get().put(name, new WeakReference<>(n));
    }

    public static void setStored(String nameNode, String key, Object value) {
        if (!ENABLED) return;
        var nr = namedNodes.get().get(nameNode);
        if (nr == null) return;
        var n = nr.get();
        if (n == null) return;
        n.set(key, value);
    }

    public static Deque<LNode> getStack() {
        return stack.get();
    }

    public static AutoCloseable addListener(LogFilter filter, AConsumer<LNode> consumer) {
        AConsumer<LNode> c = n -> {
            if (filter.predicate.test(n)) {
                consumer.accept(filter.dropFields(n));
            }
        };
        onEvent.add(c);
        return () -> onEvent.remove(c);
    }

    public static LNode createContext(Object... data) {
        if (!ENABLED) return LNode.EMPTY;
        return LNode.ofMulti(LNode.of(data), get());
    }

    public static LNode createContext(LNode... data) {
        if (!ENABLED) return LNode.EMPTY;
        return LNode.ofMulti(LNode.ofMulti(data), get());
    }

    public static LNode createContext() {
        if (!ENABLED) return LNode.EMPTY;
        return get();
    }

    public static Log.LogAutoClose context(Object... data) {
        if (!ENABLED) return Log.LogAutoClose.EMPTY;
        var n = push0(data);
        return () -> pop(n);
    }

    public static Log.LogAutoClose context(LNode parent, Object... addData) {
        if (!ENABLED) return Log.LogAutoClose.EMPTY;
        return context(LogImpl.of(addData), parent);
    }

    public static Log.LogAutoClose context(LNode... nn) {
        if (!ENABLED) return Log.LogAutoClose.EMPTY;
        if (nn == null || nn.length == 0 || nn.length == 1 && nn[0] == null) {
            return () -> {
            };
        }
        var n = LNode.ofMulti(nn);
        push(n);
        return () -> pop(n);
    }

    public static Log.LogAutoClose context(LNode n) {
        if (!ENABLED) return Log.LogAutoClose.EMPTY;
        if (n == null || n == LNode.EMPTY) {
            return () -> {
            };
        }
        push(n);
        return () -> pop(n);
    }

    private static LNode push0(Object... data) {
        LNode n = LNode.of(data);
        push(n);
        return n;
    }

    public static void push(LNode n) {
        if (!ENABLED) return;
        if (n == null) {
            throw new IllegalStateException();
        }
        var q = stack.get();
        if (q == null) {
            q = new LinkedList<>();
            stack.set(q);
        }
        q.addFirst(n);
    }

    public static LNode get() {
        if (!ENABLED) return LNode.EMPTY;
        var q = stack.get();
        if (q == null || q.isEmpty()) return LNode.EMPTY;
        if (q.size() == 1) return q.getFirst();
        return LNode.ofMulti(q.toArray(new LNode[0]));
    }

    public static Executor wrapExecutor(Executor executor) {
        if (!ENABLED) return executor;
        return new LogExecutor(executor);
    }

    public static ARunnable wrapRunnable(ARunnable t, Object... data) {
        if (!ENABLED) return t;
        var l = createContext(data);
        if (l == null || l.isEmpty()) {
            return t;
        }
        return new ARunnableWrapper(l, t);
    }

    public static void pop(LNode n) {
        if (!ENABLED) return;
        var q = stack.get();
        if (q.removeFirst() != n) {
            throw new IllegalStateException();
        }
    }

    public static <T> ASupplier<T> wrapSupplier(ASupplier<T> t) {
        if (!ENABLED) return t;
        var l = get();
        return new ASupplierWrapper<>(l, t);
    }

    public static <T> ASupplier<T> wrap(ASupplier<T> t) {
        if (!ENABLED) return t;
        return wrapSupplier(t);
    }

    public static ARunnable wrap(ARunnable t) {
        if (!ENABLED) return t;
        var l = get();
        return new ARunnableWrapper(l, t);
    }

    public static <T> AConsumer<T> wrap(AConsumer<T> t) {
        if (!ENABLED) return t;
        var l = get();
        return new AConsumerWrapper<>(l, t);
    }

    public static <T, T2> ABiConsumer<T, T2> wrap(ABiConsumer<T, T2> t) {
        if (!ENABLED) return t;
        var l = get();
        return new ABiConsumerWrapper<>(l, t);
    }

    public static <T, R> AFunction<T, R> wrap(AFunction<T, R> t) {
        if (!ENABLED) return t;
        var l = get();
        return new AFunctionWrapper<>(l, t);
    }

    public static void log(LNode node) {
        if (!ENABLED) return;
        assert node != null;
        if (filter.predicate.test(node)) {
            onEvent.fire(LNode.ofMulti(node, LNode.of(Log.TIME, new Date())));
        }
    }

    private static void log0(LNode node) {
        if (!ENABLED) return;
        assert node != null;
        if (filter.predicate.test(node)) {
            onEvent.fire(node);
        }
    }

    private static LNode log(Log.Level level, String msg, LNode... data) {
        return log(level, msg, of(data));
    }

    private static LNode log(Log.Level level, String msg, LNode data) {
        if (!ENABLED) return LNode.EMPTY;
        LNode[] nn;
        var q = stack.get();
        if (q == null) {
            nn = new LNode[]{data, LNode.of(Log.TIME, new Date(), Log.LEVEL, level, Log.MSG, msg)};
        } else {
            nn = new LNode[2 + q.size()];
            nn[0] = data;
            nn[1] = LNode.of(Log.TIME, new Date(), Log.LEVEL, level, Log.MSG, msg);
            var it = q.iterator();
            int i = 2;
            while (it.hasNext()) {
                nn[i++] = it.next();
            }
        }
        var n = LNode.ofMulti(nn);
        log0(n);
        return n;
    }

    public static LNode trace(ToString msg, Object... data) {
        return trace(msg.toString2(), data);
    }

    public static LNode trace(String msg, LNode... data) {
        return trace(msg, of(data));
    }

    public static LNode trace(String msg, LNode data) {
        if (!ENABLED) return LNode.EMPTY;
        if (!Log.TRACE) return LNode.EMPTY;
        return log(Log.Level.TRACE, msg, data);
    }

    public static LNode trace(String msg, Object... data) {
        if (!ENABLED) return LNode.EMPTY;
        if (!Log.TRACE) return LNode.EMPTY;
        return log(Log.Level.TRACE, msg, of(data));
    }

    public static LNode debug(ToString msg, Object... data) {
        return debug(msg.toString2(), data);
    }

    public static LNode debug(String msg, Object... data) {
        return debug(msg, of(data));
    }

    public static LNode debug(String msg, LNode... data) {
        return debug(msg, of(data));
    }

    public static LNode debug(String msg, LNode data) {
        if (!ENABLED) return LNode.EMPTY;
        return log(Log.Level.DEBUG, msg, data);
    }

    public static LNode info(ToString msg, Object... data) {
        return info(msg.toString2(), of(data));
    }

    public static LNode info(String msg, Object... data) {
        return info(msg, of(data));
    }

    public static LNode info(String msg, LNode... data) {
        return info(msg, of(data));
    }

    public static LNode info(String msg, LNode data) {
        if (!ENABLED) return LNode.EMPTY;
        return log(Log.Level.INFO, msg, data);
    }

    public static LNode warn(ToString msg, Object... data) {
        return warn(msg.toString2(), data);
    }

    public static LNode warn(String msg, Object... data) {
        return warn(msg, of(data));
    }

    public static LNode warn(String msg, LNode... data) {
        return warn(msg, of(data));
    }

    public static LNode warn(String msg, LNode data) {
        if (!ENABLED) return LNode.EMPTY;
        return log(Log.Level.WARN, msg, data);
    }

    public static LNode warn(String msg, Throwable throwable, Object... data) {
        if (!ENABLED) return LNode.EMPTY;
        var n = LNode.ofMulti(LNode.of(Log.TIME, new Date(), Log.LEVEL, Log.Level.WARN, "error", throwable, Log.MSG, msg), LNode.of(data), get());
        log0(n);
        return n;
    }

    public static LNode error(String msg, Throwable throwable, Object... data) {
        var n = LNode.ofMulti(LNode.of(Log.TIME, new Date(), Log.LEVEL, Log.Level.ERROR, Log.EXCEPTION_STR, throwable, Log.MSG, msg), LNode.of(data), get());
        log0(n);
        return n;
    }

    public static LNode error(String msg, Object... data) {
        var n = LNode.ofMulti(LNode.of(Log.TIME, new Date(), Log.LEVEL, Log.Level.ERROR, Log.MSG, msg), LNode.of(data), get());
        log0(n);
        return n;
    }

    public static LNode error(Throwable throwable, Object... data) {
        return error(throwable.getMessage(), throwable, data);
    }

    public static void addFilterNot(APredicate<LNode> p) {
        filter.not(p);
    }

    public static void addFilter(APredicate<LNode> p) {
        filter.filter(p);
    }

    public static LogPrinter printConsoleColored() {
        return printConsoleColored(new LogFilter());
    }

    public static LogPrinter printConsoleColored(LogFilter filter) {
        var ar = consolePrinter.get();
        if (ar != null) {
            return ar[0];
        } else {
            ar = new LogPrinter[1];
        }
        if (consolePrinter.compareAndSet(null, ar)) {
            var printer = new LogPrinter(new LogPrinter.Column[]{
                    LogPrinter.col(Log.TIME, (Date v) -> Log.DATE_FORMAT.format(v)),
                    LogPrinter.splitter(" "),
                    LogPrinter.col(Log.LEVEL).min(5),
                    LogPrinter.splitter("│"),
                    LogPrinter.col(Log.SYSTEM_COMPONENT).minAuto().min(10),
                    LogPrinter.splitter("│"),
                    LogPrinter.col(Log.MSG, (o, n, kk) -> {
                        n.printMessage(o, kk, null, null);
                    }).min(100),
                    LogPrinter.splitter("  "),
                    LogPrinter.colAll(),
            }, filter) {
                @Override
                public AString printNode(AString s, LNode n) {
                    s = s.limitByteArrays(50);
                    switch (n.getLevel()) {
                        case TRACE:
                            s.styleForeground(null, 150, 150, 150);
                            break;
                        case DEBUG:
                            s.styleForeground(null, 100, 255, 100);
                            break;
                        case INFO:
                            s.styleForeground(null, 100, 100, 255);
                            break;
                        case WARN:
                            s.styleForeground(null, 255, 50, 50);
                            break;
                        case ERROR:
                            s.styleForeground(null, 255, 0, 0);
                            break;
                    }
                    var res = super.printNode(s, n);
                    s.styleClear();
                    return res;
                }
            };
            ar[0] = printer;
            return printer;
        }
        return ar[0];
    }

    public static LogPrinter printPlainConsole(LogFilter filter) {
        var ar = new LogPrinter[1];
        if (consolePrinter.compareAndSet(null, ar)) {
            var printer = new LogPrinter(new LogPrinter.Column[]{
                    LogPrinter.col(Log.TIME, (Date v) -> Log.DATE_FORMAT.format(v)),
                    LogPrinter.splitter(" "),
                    LogPrinter.col(Log.LEVEL).min(5),
                    LogPrinter.splitter("│"),
                    LogPrinter.col(Log.SYSTEM_COMPONENT).minAuto().min(10),
                    LogPrinter.splitter("│"),
                    LogPrinter.col(Log.MSG, (o, n, kk) -> {
                        n.printMessage(o, kk, null, null);
                    }).min(100),
                    LogPrinter.colAll(),
            }, filter);
            ar[0] = printer;
            return printer;
        }
        return ar[0];
    }

    public static LNode of(Object... data) {
        if (!ENABLED) return LNode.EMPTY;
        return LNode.of(data);
    }

    public static LNode of(LNode data) {
        return data;
    }

    public static LNode of2(Object[] keys, Object[] vals) {
        return LNode.of2(keys, vals);
    }

    public static LNode of2(List<? extends CharSequence> keys, List<?> vals) {
        return LNode.of2(keys, vals);
    }

    public static LNode ofMap(Map<String, ?> vals) {
        return LNode.ofMap(vals);
    }

    public static LNode of(LNode... data) {
        if (!ENABLED) return LNode.EMPTY;
        return LNode.ofMulti(data);
    }

    public static void loggerOff() {
        ENABLED = false;
    }

    public static void loggerOn() {
        ENABLED = true;
    }

    private static class SetStub extends AbstractSet<String> {
        @Override
        public @NotNull Iterator<String> iterator() {
            return EMPTY_ITERATOR;
        }

        @Override
        public boolean add(String s) {
            return true;
        }

        @Override
        public int size() {
            return 0;
        }
    }

    private static class LogExecutor implements Executor, ObjectFind {
        private final Executor executor;

        public LogExecutor(Executor executor) {
            this.executor = executor;
        }

        @Override
        public <T> T find(Class<T> type) {
            if (type.isInstance(this)) return RU.cast(this);
            if (type.isInstance(executor)) return type.cast(executor);
            if (executor instanceof ObjectFind) {
                return ((ObjectFind) executor).find(type);
            }
            return null;
        }

        @Override
        public <T> T find(CTypeI<T> type) {
            if (type.isInstance(this)) return RU.cast(this);
            if (type.isInstance(executor)) return type.cast(executor);
            if (executor instanceof ObjectFind) {
                return ((ObjectFind) executor).find(type);
            }
            return null;
        }

        @Override
        public void execute(@NotNull Runnable command) {
            var q = stack.get();
            if (q == null || q.isEmpty()) {
                executor.execute(() -> {
                    try {
                        command.run();
                    } catch (Throwable e) {
                        Log.error(e);
                        RU.error(e);
                    }
                });
            } else {
                var nn = get();
                executor.execute(() -> {
                    LogImpl.push(nn);
                    try {
                        command.run();
                    } catch (Throwable e) {
                        Log.error(e);
                    } finally {
                        LogImpl.pop(nn);
                    }
                });
            }
        }
    }

    private static class AConsumerWrapper<T> implements AConsumer<T>, LogWrapper {
        private final LNode l;
        private final AConsumer<T> t;

        public AConsumerWrapper(LNode l, AConsumer<T> t) {
            this.l = l;
            this.t = t;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) return true;
            if (obj instanceof AConsumerWrapper) {
                return t.equals(((AConsumerWrapper<?>) obj).t);
            } else {
                return obj.equals(t);
            }
        }

        @Override
        public int hashCode() {
            return t.hashCode();
        }

        @Override
        public void accept2(T v) throws Throwable {
            LogImpl.push(l);
            try {
                t.accept(v);
            } finally {
                LogImpl.pop(l);
            }
        }
    }

    private static class ABiConsumerWrapper<T, T2> implements ABiConsumer<T, T2>, LogWrapper {
        private final LNode l;
        private final ABiConsumer<T, T2> t;

        public ABiConsumerWrapper(LNode l, ABiConsumer<T, T2> t) {
            this.l = l;
            this.t = t;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) return true;
            if (obj instanceof ABiConsumerWrapper) {
                return t.equals(((ABiConsumerWrapper<?, ?>) obj).t);
            } else {
                return obj.equals(t);
            }
        }

        @Override
        public int hashCode() {
            return t.hashCode();
        }

        @Override
        public void accept2(T v, T2 v2) throws Throwable {
            LogImpl.push(l);
            try {
                t.accept(v, v2);
            } finally {
                LogImpl.pop(l);
            }
        }
    }

    private static class AFunctionWrapper<T, R> implements AFunction<T, R>, LogWrapper {
        private final LNode l;
        private final AFunction<T, R> t;

        public AFunctionWrapper(LNode l, AFunction<T, R> t) {
            this.l = l;
            this.t = t;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) return true;
            if (obj instanceof AFunctionWrapper) {
                return t.equals(((AFunctionWrapper<?, ?>) obj).t);
            } else {
                return obj.equals(t);
            }
        }

        @Override
        public int hashCode() {
            return t.hashCode();
        }

        @Override
        public R apply2(T v) throws Throwable {
            LogImpl.push(l);
            try {
                return t.apply(v);
            } finally {
                LogImpl.pop(l);
            }
        }
    }

    private static class ARunnableWrapper implements ARunnable, LogWrapper {
        private final LNode l;
        private final ARunnable t;

        public ARunnableWrapper(LNode l, ARunnable t) {
            this.l = l;
            this.t = t;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) return true;
            if (obj instanceof ARunnableWrapper) {
                return t.equals(((ARunnableWrapper) obj).t);
            } else {
                return obj.equals(t);
            }
        }

        @Override
        public int hashCode() {
            return t.hashCode();
        }

        @Override
        public void run2() throws Throwable {
            LogImpl.push(l);
            try {
                t.run();
            } finally {
                LogImpl.pop(l);
            }
        }
    }

    private static class ASupplierWrapper<T> implements ASupplier<T>, LogWrapper {
        private final LNode l;
        private final ASupplier<T> t;

        public ASupplierWrapper(LNode l, ASupplier<T> t) {
            this.l = l;
            this.t = t;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) return true;
            if (obj instanceof ASupplierWrapper) {
                return t.equals(((ASupplierWrapper<?>) obj).t);
            } else {
                return obj.equals(t);
            }
        }

        @Override
        public int hashCode() {
            return t.hashCode();
        }

        @Override
        public T get2() throws Exception {
            LogImpl.push(l);
            try {
                return t.get();
            } finally {
                LogImpl.pop(l);
            }
        }
    }
}