package io.aether.utils;

import io.aether.utils.interfaces.ABiConsumer;
import io.aether.utils.interfaces.AFunction;
import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.lang.ref.SoftReference;
import java.nio.charset.StandardCharsets;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

public interface AString extends CharSequence {

    ABiConsumer<?, AString> DEFAULT_RENDERER = (v, s) -> s.add(String.valueOf(v));

    default AString limit(int count) {
        var self = this;
        return new AString() {
            int limit = count - 3;
            boolean end = false;

            private void end() {
                if (end) return;
                self.add("...");
                end = true;
            }

            @Override
            public @NotNull String toString() {
                return self.toString();
            }

            @Override
            public AString add(CharSequence val) {
                if (val.length() == 0) return this;
                if (val.length() == limit) {
                    limit = 0;
                    self.add(val);
                } else if (val.length() <= limit) {
                    limit -= val.length();
                    self.add(val);
                } else {
                    self.add(val.subSequence(0, limit));
                    limit = 0;
                    end();
                }
                return this;
            }

            @Override
            public AString add(char val) {
                if (limit == 0) {
                    end();
                    return this;
                }
                self.add(val);
                return this;
            }

            @Override
            public int length() {
                return self.length();
            }

            @Override
            public char charAt(int index) {
                return self.charAt(index);
            }

            @Override
            public @NotNull CharSequence subSequence(int start, int end) {
                return self.subSequence(start, end);
            }
        };
    }

    default AString add(ToString val) {
        val.toString(this);
        return this;
    }

    AString add(CharSequence val);

    default AString addSpace(int count) {
        if (count <= 0) return this;
        for (int i = 0; i < count; i++) add(' ');
        return this;
    }

    default AString repeat(int count, Object val) {
        for (int i = 0; i < count; i++) {
            add(val);
        }
        return this;
    }

    default AString repeat(int count, char val) {
        for (int i = 0; i < count; i++) {
            add(val);
        }
        return this;
    }

    default AString repeat(int count, String val) {
        for (int i = 0; i < count; i++) {
            add(val);
        }
        return this;
    }

    default int calcVisibleSymbols(int begin) {
        return calcVisibleSymbols(begin, length());
    }

    default int calcVisibleSymbols(int begin, int end) {
        if (end > length()) {
            throw new IllegalStateException();
        }
        int i = 0;
        while (begin < end) {
            if (charAt(begin) == '\u001B') {
                while (begin < end && charAt(begin) != 'm') begin++;
            } else {
                i++;
            }
            begin++;
        }
        return i;
    }

    default AString add(String val) {
        return add((CharSequence) val);
    }

    default AString styleClear() {
        return add("\u001B[0m");
    }

    default AString styleForeground(Style mode, int red, int green, int blue) {
        add("\u001B[");
//        add("\u001B[");
        if (mode != null) {
            add(mode.ordinal()).add(";");
        }
        add("38;2;").add(red).add(";").add(green).add(";").add(blue).add("m");
        return this;
    }

    default AString styleBackground(Style mode, int red, int green, int blue) {
        add("\u001B[");
        if (mode != null) {
            add(mode.ordinal()).add(";");
        }
        add("48;2;").add(red).add(";").add(green).add(";").add(blue).add("m");
        return this;
    }

    default AString color(Color color) {
        add("\u001B[").add(color.code).add("m");
        return this;
    }

    default AString style(Style mode, Color color) {
        if (color == null) return style(mode);
        add("\u001B[");
        if (mode != null) {
            add(mode.ordinal()).add(";");
        }
        add(color.code);
        add("m");
        return this;
    }

    default AString style(Style mode) {
        add("\u001B[");
        add(mode.ordinal());
        add("m");
        return this;
    }

    default AString add(char[] val) {
        if (val == null || val.length == 0) return this;
        add(val, 0, val.length);
        return this;
    }

    default AString add(char[] val, int offset, int len) {
        return add(new CharSequenceByArray(val, offset, offset + len));
    }

    default AString add(CharSequence val, int offset, int len) {
        return add(val.subSequence(offset, offset + len));
    }

    AString add(char val);

    default AString add(boolean val) {
        add(Boolean.toString(val));
        return this;
    }

    default AString add(int val) {
        add(Integer.toString(val));
        return this;
    }

    default AString add(byte val) {
        add(Byte.toString(val));
        return this;
    }

    default AString add(short val) {
        add(Short.toString(val));
        return this;
    }

    default AString add(long val) {
        add(Long.toString(val));
        return this;
    }

    default AString add(float val) {
        add(Float.toString(val));
        return this;
    }

    default AString add(double val) {
        add(Double.toString(val));
        return this;
    }

    default AString add(byte[] val) {
        HexUtils.toHexString(val, 0, val.length, this);
        return this;
    }

    default AString add(Object[] val) {
        if (val == null) {
            add("null");
        } else {
            boolean first = true;
            for (Object j : val) {
                if (first) {
                    first = false;
                } else {
                    add(", ");
                }
                add(j);
            }
        }
        return this;
    }

    default AString add(int[] val) {
        if (val == null) {
            add("null");
        } else {
            boolean first = true;
            for (int j : val) {
                if (first) {
                    first = false;
                } else {
                    add(',');
                }
                add(j);
            }
        }
        return this;
    }

    default AString add(short[] val) {
        if (val == null) {
            add("null");
        } else {
            boolean first = true;
            for (var j : val) {
                if (first) {
                    first = false;
                } else {
                    add(',');
                }
                add(j);
            }
        }
        return this;
    }

    default AString add(long[] val) {
        if (val == null) {
            add("null");
        } else {
            boolean first = true;
            for (long j : val) {
                if (first) {
                    first = false;
                } else {
                    add(',');
                }
                add(j);
            }
        }
        return this;
    }

    default AString add(float[] val) {
        if (val == null) {
            add("null");
        } else {
            boolean first = true;
            for (float j : val) {
                if (first) {
                    first = false;
                } else {
                    add(',');
                }
                add(j);
            }
        }
        return this;
    }

    default AString add(double[] val) {
        if (val == null) {
            add("null");
        } else {
            boolean first = true;
            for (double j : val) {
                if (first) {
                    first = false;
                } else {
                    add(',');
                }
                add(j);
            }
        }
        return this;
    }

    default <T> ABiConsumer<T, AString> getLocalRenderer(Class<T> cl) {
        return AString.getRenderer(cl);
    }

    default AString addNull() {
        add("null");
        return this;
    }

    default AString addStackTrace(Throwable e) {
        e.printStackTrace(toPrintWriter());
        return this;
    }

    default AString add(Iterable<?> val) {
        if (val == null) {
            addNull();
            return this;
        }
        boolean first = true;
        for (var e : val) {
            if (first) {
                first = false;
            } else {
                add(",");
            }
            add(e);
        }
        return this;
    }

    default AString add(Object val) {
        if (val == null) {
            addNull();
            return this;
        }
        if (val instanceof Iterable) {
            return add((Iterable<?>) val);
        }
        var cl = val.getClass();
        getLocalRenderer(cl).accept(RU.cast(val), this);
        return this;
    }

    default int indexOf(CharSequence c) {
        return indexOf(c, 0);
    }

    default int indexOf(CharSequence c, int offset) {
        if (c == null) return -1;
        var len = length() - c.length();
        lo:
        for (int i = offset; i <= len; i++) {
            int ii = i;
            for (int i2 = 0; i2 < c.length(); i2++) {
                if (charAt(ii++) != c.charAt(i2)) {
                    continue lo;
                }
            }
            return i;
        }
        return -1;
    }

    default PrintWriter toPrintWriter() {
        return new PrintWriter(new Writer() {
            @Override
            public void write(char @NotNull [] cbuf, int off, int len) throws IOException {
                AString.this.add(cbuf, off, len);
            }

            @Override
            public void flush() throws IOException {

            }

            @Override
            public void close() throws IOException {

            }
        });
    }    Map<Class<?>, ABiConsumer<?, AString>> renderers = new ConcurrentHashMap<Class<?>, ABiConsumer<?, AString>>() {
        {
            putRenderer(ToString.class, (v, s) -> s.add(v));
            putRenderer(byte[].class, (v, s) -> s.add(v));
            putRenderer(short[].class, (v, s) -> s.add(v));
            putRenderer(int[].class, (v, s) -> s.add(v));
            putRenderer(long[].class, (v, s) -> s.add(v));
            putRenderer(float[].class, (v, s) -> s.add(v));
            putRenderer(double[].class, (v, s) -> s.add(v));
            putRenderer(char[].class, (v, s) -> s.add(v));
            putRenderer(String.class, (v, s) -> s.add(v));
            putRenderer(Byte.class, (v, s) -> s.add((byte) v));
            putRenderer(Short.class, (v, s) -> s.add((short) v));
            putRenderer(Integer.class, (v, s) -> s.add((int) v));
            putRenderer(Long.class, (v, s) -> s.add((long) v));
            putRenderer(Float.class, (v, s) -> s.add((float) v));
            putRenderer(Double.class, (v, s) -> s.add((double) v));
            putRenderer(CharSequence.class, (v, s) -> s.add(v));
            putRenderer(Character.class, (v, s) -> s.add((char) v));
            DateFormat DATE_FORMAT = new SimpleDateFormat("MM:dd:HH:mm:ss.SSSS");
            putRenderer(Date.class, (v, s) -> s.add(DATE_FORMAT.format(v)));
            putRenderer(Boolean.class, (v, s) -> s.add((boolean) v));
            putRenderer(SoftReference.class, (v, s) -> {
                var vv = v.get();
                s.add(vv);
            });
            putRenderer(Throwable.class, (v, s) -> {
                s.add(v.getClass().getSimpleName()).add("(");
                if (v.getMessage() != null) {
                    s.add(v.getMessage());
                }
                s.add(")");
            });
            putRenderer(UUID.class, (v, sb) -> {
                long vv = v.getMostSignificantBits();
                HexUtils.toHexString(vv >> (7 * 8), sb);
                HexUtils.toHexString(vv >> (6 * 8), sb);
                HexUtils.toHexString(vv >> (5 * 8), sb);
                HexUtils.toHexString(vv >> (4 * 8), sb);
                sb.add('-');
                HexUtils.toHexString(vv >> (3 * 8), sb);
                HexUtils.toHexString(vv >> (2 * 8), sb);
                sb.add('-');
                HexUtils.toHexString(vv >> (8), sb);
                HexUtils.toHexString(vv >> (0), sb);
                sb.add('-');
                vv = v.getLeastSignificantBits();
                HexUtils.toHexString(vv >> (7 * 8), sb);
                HexUtils.toHexString(vv >> (6 * 8), sb);
                sb.add('-');
                HexUtils.toHexString(vv >> (5 * 8), sb);
                HexUtils.toHexString(vv >> (4 * 8), sb);
                HexUtils.toHexString(vv >> (3 * 8), sb);
                HexUtils.toHexString(vv >> (2 * 8), sb);
                HexUtils.toHexString(vv >> (8), sb);
                HexUtils.toHexString(vv >> (0), sb);
            });
        }

        <T> void putRenderer(Class<T> cl, ABiConsumer<T, AString> f) {
            put(cl, f);
        }
    };

    default AString addVars(CharSequence msg, Object... vars) {
        return addVars(msg, v -> {
            for (int i = 0; i < vars.length; i += 2) {
                if (Objects.equals(vars[i], v)) {
                    return vars[i + 1];
                }
            }
            return null;
        });
    }

    default AString addVars(CharSequence msg, AFunction<CharSequence, Object> f) {
        int vi = -1;
        for (int i = 0; i < msg.length(); i++) {
            if (msg.charAt(i) == '$') {
                vi = i;
                break;
            }
        }
        if (vi == -1) {
            add(msg);
            return this;
        }
        int start = 0;
        lo:
        while (true) {
            int vEnd = vi + 1;
            add(msg, start, vi - start);
            while (vEnd < msg.length() && Character.isJavaIdentifierPart(msg.charAt(vEnd))) vEnd++;
            var key = msg.subSequence(vi + 1, vEnd);
            var v = f.apply(key);
            if (v != null) {
                try {
                    add(v);
                } catch (Exception e) {
                    try {
                        add("[[error]]: ").add(v.toString());
                    } catch (Exception e2) {
                        add("[[error]]");
                    }
                }
            }
            start = vEnd;
            for (int i = vEnd; i < msg.length(); i++) {
                if (msg.charAt(i) == '$') {
                    vi = i;
                    continue lo;
                }
            }
            break;
        }
        var l = msg.length() - start;
        if (l > 0) add(msg, start, l);
        return this;
    }

    default AString replaceAll(CharSequence src, char sample, String s) {
        for (int i = 0; i < src.length(); i++) {
            var c = src.charAt(i);
            if (c == sample) {
                add(s);
            } else {
                add(c);
            }
        }
        return this;
    }

    default AString limitByteArrays(int max) {
        var self = this;
        return new LimitByteArrays(self, max);
    }

    default byte[] getBytes() {
        return toString().getBytes(StandardCharsets.UTF_8);
    }

    default AString addWithAlign(int width, int firstIndent, int indent, String txt) {
        var text = txt.split(" ");
        int i = 0;
        int line = 0;
        addSpace(indent - firstIndent);
        while (i < text.length) {
            if (line + text[i].length() <= width || (line == 0 && text[i].length() > width)) {
                if (line > 0) {
                    add(" ");
                }
                add(text[i]);
                line += text[i].length();
            } else {
                line = 0;
                add("\n");
                addSpace(indent);
                continue;
            }
            i++;
        }
        add("\n");
        return this;
    }

    static <T> void putRenderer(Class<T> cl, ABiConsumer<T, AString> f) {
        renderers.put(cl, f);
    }

    static AString of() {
        return of(new StringBuilder());
    }

    static AString of(StringBuilder stringBuilder) {
        return new Simple(stringBuilder);
    }

    static <T> void addRenderer(Class<T> type, ABiConsumer<T, AString> renderer) {
        renderers.put(type, renderer);
    }

    static <T> ABiConsumer<T, AString> getRenderer(Class<T> type) {
        if (type == null) {
            return RU.cast(DEFAULT_RENDERER);
        }
        var renderer = renderers.get(type);
        if (renderer != null) {
            return RU.cast(renderer);
        } else {
            if (type.isArray()) {
                renderer = (v, s) -> s.add((Object[]) v);
            } else {
                for (var i : type.getInterfaces()) {
                    renderer = getRenderer(i);
                    if (renderer != DEFAULT_RENDERER) {
                        break;
                    }
                }
                if (renderer == DEFAULT_RENDERER) {
                    renderer = getRenderer(type.getSuperclass());
                }
            }
            if (renderer == null) renderer = DEFAULT_RENDERER;
            renderers.put(type, renderer);
            return RU.cast(renderer);
        }
    }

    enum Style {
        CLEAR, BRIGHT, DIM, ITALIC, UNDERSCORE, BLINK, UNKNOWN1, REVERSE, HIDDEN, UNKNOWN2, CROSSED,
    }

    enum Color {
        BLACK("30"), RED("31"), GREEN("32"), ORANGE("33"), BLUE("34"), PURPLE("35"), CYAN("36"), WHITE("37"),
        ;
        public final String code;

        Color(String code) {
            this.code = code;
        }
    }

    enum BackgroundColor {
        BLACK("40"), RED("41"), GREEN("42"), ORANGE("43"), BLUE("44"), PURPLE("45"), CYAN("46"), WHITE("47"),
        ;
        public final String code;

        BackgroundColor(String code) {
            this.code = code;
        }
    }

    enum Style2 {
        UNDERSCORE_DOUBLE("21"), BORDER("51"), BORDER2("52"), UNDERSCORE2("53"),
        ;
        public final String code;

        Style2(String code) {
            this.code = code;
        }
    }

    class Simple implements AString {
        private final StringBuilder stringBuilder;

        public Simple(StringBuilder stringBuilder) {
            this.stringBuilder = stringBuilder;
        }

        @Override
        public @NotNull String toString() {
            return stringBuilder.toString();
        }

        @Override
        public AString add(char val) {
            stringBuilder.append(val);
            return this;
        }

        @Override
        public AString add(CharSequence val) {
            stringBuilder.append(val);
            return this;
        }

        @Override
        public AString add(char[] val, int offset, int len) {
            stringBuilder.append(val, offset, len);
            return this;
        }

        @Override
        public AString add(String val) {
            stringBuilder.append(val);
            return this;
        }

        @Override
        public AString add(char[] val) {
            stringBuilder.append(val);
            return this;
        }

        @Override
        public int length() {
            return stringBuilder.length();
        }

        @Override
        public char charAt(int index) {
            return stringBuilder.charAt(index);
        }

        @Override
        public @NotNull CharSequence subSequence(int start, int end) {
            return stringBuilder.subSequence(start, end);
        }
    }

    class LimitByteArrays implements AString {

        private final AString self;
        private final int max;
        int cycleCounter;

        public LimitByteArrays(AString self, int max) {
            this.self = self;
            this.max = max;
        }

        @Override
        public AString add(CharSequence val) {
            self.add(val);
            return this;
        }

        @Override
        public @NotNull String toString() {
            return self.toString();
        }

        @Override
        public AString add(byte[] val) {
            if (val == null) return this;
            var l = Math.min(val.length, max);
            HexUtils.toHexString(val, 0, l, this);
            if (l != val.length) {
                add("...");
            }
            return this;
        }

        @Override
        public AString add(Object val) {
            cycleCounter++;
            if (cycleCounter > 10) {
                throw new IllegalStateException();
            }
            try {
                if (val instanceof byte[]) {
                    return add((byte[]) val);
                }
                AString.super.add(val);
                return this;
            } finally {
                cycleCounter--;
            }
        }

        @Override
        public AString add(char val) {
            self.add(val);
            return this;
        }

        @Override
        public int length() {
            return self.length();
        }

        @Override
        public char charAt(int index) {
            return self.charAt(index);
        }

        @Override
        public @NotNull CharSequence subSequence(int start, int end) {
            return self.subSequence(start, end);
        }
    }




}
