package io.aether.utils;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;

public final class AutoRun {
    public static final Map<Executor, WeakCollection<Task>> ALL = new ConcurrentHashMap<>();

    static {
        AtomicBoolean flag = new AtomicBoolean();
        RU.scheduleAtFixedRate(10L, () -> {
            if (!flag.compareAndSet(false, true)) return;
            for (var e : ALL.entrySet()) {
                var tt = e.getValue();
                e.getKey().execute(() -> {
                    var currentTime = System.currentTimeMillis();
                    for (var t : tt) {
                        t.runAutoRunTask(currentTime);
                    }
                });
            }
            flag.set(false);
        });
    }

    public static void add(Executor executor, Task task) {
        ALL.computeIfAbsent(executor, k -> new WeakCollection<>()).add(task);
    }

    public interface Task {
        void runAutoRunTask(long currentTime);
    }

    public static abstract class TaskImpl implements Task {
        public long lastWork = Long.MAX_VALUE;
        public long timeout = 5;

        public TaskImpl(Executor executor) {
            AutoRun.add(executor, this);
        }

        public void setTimeout(long timeout) {
            this.timeout = timeout;
        }

        public abstract void work();

        public void free() {
            lastWork = Long.MAX_VALUE;
        }

        public void needWork() {
            lastWork = System.currentTimeMillis();
        }

        @Override
        public void runAutoRunTask(long currentTime) {
            if (currentTime - lastWork > timeout) {
                var old = lastWork;
                if (System.currentTimeMillis() - old > timeout) {
                    work();
                    if (lastWork == old) {
                        lastWork = 0;
                    }
                }
            }
        }
    }

    public static final class Multi implements Task {
        private final CopyOnWriteArrayList<Task> subTasks = new CopyOnWriteArrayList<>();
        private long totalLastWork = Long.MAX_VALUE;
        private long minTimeout;

        public Multi(Executor executor) {
            var mint = Long.MAX_VALUE;
            for (var s : subTasks) {
                mint = Math.min(mint, s.timeout);
            }
            minTimeout = mint;
            AutoRun.add(executor, this);
        }

        public void addTask(Task task) {
            subTasks.add(task);
        }

        private void updateMinTimeout(long t) {
            minTimeout = Math.min(minTimeout, t);
        }

        public void free() {
            if (totalLastWork == Long.MAX_VALUE) return;
            for (var s : subTasks) {
                s.free();
            }
            totalLastWork = Long.MAX_VALUE;
        }

        @Override
        public final void runAutoRunTask(long currentTime) {
            if (currentTime - totalLastWork > minTimeout) {
                totalLastWork = Long.MAX_VALUE;
                for (var s2 : subTasks) {
                    s2.runAutoRunTask(currentTime);
                    totalLastWork = Math.min(s2.lastWork, totalLastWork);
                }
            }
        }

        public static abstract class Task {
            private final Multi multi;
            private long lastWork = Long.MAX_VALUE;
            private long timeout = 5;

            public Task(Multi multi) {
                this.multi = multi;
                multi.addTask(this);
            }

            public abstract void work();

            public void free() {
                lastWork = Long.MAX_VALUE;
            }

            public void needWork() {
                if (RU.time() - lastWork > timeout) {
                    lastWork = Long.MAX_VALUE;
                    work();
                } else if (lastWork == Long.MAX_VALUE) {
                    refreshNeedWork();
                }
            }

            public void refreshNeedWork() {
                lastWork = System.currentTimeMillis();
                if (multi.totalLastWork == Long.MAX_VALUE) {
                    multi.totalLastWork = lastWork;
                }
            }

            public void setTimeout(long timeout) {
                this.timeout = timeout;
                multi.updateMinTimeout(timeout);
            }

            private void runAutoRunTask(long currentTime) {
                var old = lastWork;
                if (currentTime - old > timeout) {
                    work();
                    if (lastWork == old) {
                        lastWork = Long.MAX_VALUE;
                    }
                }
            }
        }
    }
}
