package io.aether.utils;

import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.ASupplier;
import io.leangen.geantyref.AnnotationFormatException;
import io.leangen.geantyref.GenericTypeReflector;
import io.leangen.geantyref.TypeFactory;
import jdk.jfr.Unsigned;
import org.jetbrains.annotations.NotNull;

import java.lang.annotation.Annotation;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.VarHandle;
import java.lang.reflect.*;
import java.util.*;

/**
 * Interface representing a type with additional metadata and utility methods.
 * Provides methods for type inspection, manipulation, and comparison.
 *
 * @param <T> the type represented by this instance
 */
public interface CTypeI<T> extends ToString {
    // Constants for common types with annotations
    Unsigned UNSIGNED = CTypeI.annotation(Unsigned.class, Map.of());
    CTypeI<Byte> UBYTE = CTypeI.of(UNSIGNED, byte.class);
    CTypeI<Short> USHORT = CTypeI.of(UNSIGNED, short.class);
    CTypeI<Integer> UINT = CTypeI.of(UNSIGNED, int.class);
    CTypeI<Long> ULONG = CTypeI.of(UNSIGNED, long.class);
    String CLASS_NAME = CTypeI.class.getCanonicalName();
    CTypeI<String> STRING = CTypeI.of(String.class);
    CTypeI<Character> CHAR = CTypeI.of(char.class);
    CTypeI<Character> CHAR_BOX = CTypeI.of(Character.class);
    CTypeI<Boolean> BOOLEAN = CTypeI.of(boolean.class);
    CTypeI<Boolean> BOOLEAN_BOX = CTypeI.of(Boolean.class);
    CTypeI<Byte> BYTE = CTypeI.of(byte.class);
    CTypeI<Byte> BYTE_BOX = CTypeI.of(Byte.class);
    CTypeI<Short> SHORT = CTypeI.of(short.class);
    CTypeI<Short> SHORT_BOX = CTypeI.of(Short.class);
    CTypeI<Integer> INT = CTypeI.of(int.class);
    CTypeI<Integer> INT_BOX = CTypeI.of(Integer.class);
    CTypeI<Long> LONG = CTypeI.of(long.class);
    CTypeI<Long> LONG_BOX = CTypeI.of(Long.class);
    CTypeI<Float> FLOAT = CTypeI.of(float.class);
    CTypeI<Float> FLOAT_BOX = CTypeI.of(Float.class);
    CTypeI<Double> DOUBLE = CTypeI.of(double.class);
    CTypeI<Double> DOUBLE_BOX = CTypeI.of(Double.class);
    CTypeI<byte[]> BYTE_ARR = CTypeI.of(byte[].class);
    CTypeI<boolean[]> BOOLEAN_ARR = CTypeI.of(boolean[].class);
    CTypeI<short[]> SHORT_ARR = CTypeI.of(short[].class);
    CTypeI<int[]> INT_ARR = CTypeI.of(int[].class);
    CTypeI<long[]> LONG_ARR = CTypeI.of(long[].class);
    CTypeI<float[]> FLOAT_ARR = CTypeI.of(float[].class);
    CTypeI<double[]> DOUBLE_ARR = CTypeI.of(double[].class);

    /**
     * Checks if the type is mutable (can be modified after creation).
     *
     * @return true if the type is mutable, false otherwise
     */
    boolean mutable();

    /**
     * Gets a field accessor for the specified field name.
     *
     * @param <FT> the field type
     * @param field the name of the field
     * @return a field accessor for the specified field
     */
    <FT> FieldAccessor<FT> getFieldAccessor(String field);

    /**
     * Gets a field accessor for the specified field.
     *
     * @param <FT> the field type
     * @param field the field object
     * @return a field accessor for the specified field
     */
    <FT> FieldAccessor<FT> getFieldAccessor(Field field);

    /**
     * Adds annotations to this type and returns a new type instance with the annotations.
     *
     * @param aa the annotations to add
     * @return a new type instance with the added annotations
     */
    CTypeI<T> addAnnotations(Annotation... aa);

    /**
     * Gets the simple name of the raw type with type parameters.
     *
     * @return the simple name with type parameters
     */
    String getRawSimpleNameWithParameters();

    /**
     * Appends the simple name of the raw type with type parameters to the specified string builder.
     *
     * @param sb2 the string builder to append to
     */
    void getRawSimpleNameWithParameters(AString sb2);

    /**
     * Gets the canonical name of the type with type parameters.
     *
     * @return the canonical name with type parameters
     */
    CharSequence getCanonicalNameWithParameters();

    /**
     * Appends the canonical name of the type with type parameters to the specified string builder.
     *
     * @param sb the string builder to append to
     */
    void getCanonicalNameWithParameters(AString sb);

    /**
     * Gets the simple name of the raw type.
     *
     * @return the simple name of the raw type
     */
    String getRawSimpleName();

    /**
     * Gets the declaration code for this type, including annotations and type parameters.
     *
     * @return the declaration code
     */
    String declareCode();

    /**
     * Gets a unique identifier for this type.
     *
     * @return the unique identifier
     */
    String toId();

    /**
     * Gets the raw class representing this type.
     *
     * @return the raw class
     */
    Class<T> getRaw();

    /**
     * Gets the raw class representing this type (alternative to getRaw).
     *
     * @return the raw class
     */
    Class<?> getRaw2();

    /**
     * Checks if this type is an instance of the specified type.
     *
     * @param t the type to check against
     * @return true if this type is an instance of the specified type, false otherwise
     */
    boolean instanceOf(Type t);

    /**
     * Checks if this type is an instance of the specified type.
     *
     * @param t the type to check against
     * @return true if this type is an instance of the specified type, false otherwise
     */
    boolean instanceOf(CTypeI<?> t);

    /**
     * Checks if this type is an instance of the specified annotated type.
     *
     * @param t the annotated type to check against
     * @return true if this type is an instance of the specified type, false otherwise
     */
    boolean instanceOf(AnnotatedType t);

    /**
     * Converts this type to a standard Type object.
     *
     * @return the Type representation
     */
    @NotNull Type toType();

    /**
     * Converts this type to an AnnotatedType object.
     *
     * @return the AnnotatedType representation
     */
    @NotNull AnnotatedType toAnnotatedType();

    /**
     * Gets the component type for array types.
     *
     * @return the component type, or null if not an array
     */
    CTypeI<Object> getComponent();

    /**
     * Gets the component type by name for parameterized types.
     *
     * @param name the name of the type parameter
     * @return the component type, or null if not found
     */
    CTypeI<Object> getComponentByName(String name);

    /**
     * Gets the component type at the specified index.
     *
     * @param i the index of the component type
     * @return the component type, or null if not found
     */
    CTypeI<Object> getComponent(int i);

    /**
     * Gets the component type for the specified base class and index.
     *
     * @param base the base class
     * @param i the index of the type parameter
     * @return the resolved component type
     */
    CTypeI<Object> getComponent(Class<?> base, int i);

    /**
     * Checks if this type is annotated with the specified annotation.
     *
     * @param aClass the annotation class to check for
     * @return true if the annotation is present, false otherwise
     */
    boolean isAnnotated(Class<? extends Annotation> aClass);

    /**
     * Gets the specified annotation if present on this type.
     *
     * @param <A> the annotation type
     * @param aClass the annotation class
     * @return the annotation, or null if not present
     */
    <A extends Annotation> A getAnnotation(Class<A> aClass);

    /**
     * Checks if this type represents an array.
     *
     * @return true if this is an array type, false otherwise
     */
    boolean isArray();

    /**
     * Checks if this type represents a primitive type.
     *
     * @return true if this is a primitive type, false otherwise
     */
    boolean isPrimitive();

    /**
     * Checks if this type represents a number (primitive or boxed).
     *
     * @return true if this is a number type, false otherwise
     */
    boolean isNumber();

    /**
     * Checks if this type represents an enum.
     *
     * @return true if this is an enum type, false otherwise
     */
    boolean isEnum();

    /**
     * Checks if this type represents an interface.
     *
     * @return true if this is an interface, false otherwise
     */
    boolean isInterface();

    /**
     * Gets the type variables for generic types.
     *
     * @return the type variables
     */
    TypeVariable<?>[] getGenericVariables();

    /**
     * Gets all declared methods of this type.
     *
     * @return a flow of declared methods
     */
    Flow<Method> getMethods();

    /**
     * Gets all fields of this type, including inherited ones.
     *
     * @return a flow of all fields
     */
    Flow<Field> getAllFields();

    /**
     * Gets all methods of this type, including inherited ones.
     *
     * @return a flow of all methods
     */
    Flow<Method> getAllMethods();

    /**
     * Gets all declared fields of this type.
     *
     * @return a flow of declared fields
     */
    Flow<Field> getFields();

    /**
     * Gets all annotations on this type.
     *
     * @return an array of annotations
     */
    Annotation[] getAnnotations();

    /**
     * Resolves the return type of the specified method in the context of this type.
     *
     * @param m the method
     * @return the resolved return type
     */
    CTypeI<?> resolveReturnType(Method m);

    /**
     * Resolves the type arguments of the specified method in the context of this type.
     *
     * @param m the method
     * @return an array of resolved type arguments
     */
    CTypeI<?>[] resolveTypeArgs(Method m);

    /**
     * Resolves the extended type for the specified interface in the context of this type.
     *
     * @param inf the interface type
     * @return the resolved extended type
     */
    CTypeI<?> resolveExtend(CTypeI<Object> inf);

    /**
     * Resolves the type of the specified field in the context of this type.
     *
     * @param f the field
     * @return the resolved field type
     */
    CTypeI<?> resolveTypeField(Field f);

    /**
     * Checks if this type represents a boxed primitive.
     *
     * @return true if this is a boxed primitive, false otherwise
     */
    boolean isBoxedPrimitive();

    /**
     * Checks if this type represents a boxed primitive or raw primitive.
     *
     * @return true if this is a boxed or raw primitive, false otherwise
     */
    boolean isBoxedOrRawPrimitive();

    /**
     * Converts this type to its primitive equivalent if it's a boxed primitive.
     *
     * @return the primitive type
     */
    CTypeI<T> toPrimitive();

    /**
     * Unboxes this type if it's a boxed primitive.
     *
     * @return the unboxed type
     */
    CTypeI<T> unbox();

    /**
     * Gets the type parameters for this type.
     *
     * @return an array of type parameters
     */
    CTypeI<Object>[] getParameters();

    /**
     * Gets the parent type of this type.
     *
     * @return the parent type, or null if none exists
     */
    CTypeI<? super T> getParent();

    /**
     * Gets the permitted subclasses for this sealed type.
     *
     * @return an array of permitted subclasses
     */
    CTypeI<? extends T>[] getChildrenClasses();

    /**
     * Collects all dependencies of this type.
     *
     * @param res the set to store dependencies in
     * @param withTransient whether to include transient dependencies
     */
    void getAllDependencies(Set<CTypeI<?>> res, boolean withTransient);

    /**
     * Gets all children classes of this type, including nested subclasses.
     *
     * @return a flow of all children classes
     */
    Flow<CTypeI<? extends T>> getAllChildrenClasses();

    /**
     * Checks if this type is abstract.
     *
     * @return true if this type is abstract, false otherwise
     */
    boolean isAbstract();

    /**
     * Attempts to box this type if it's a primitive.
     *
     * @return the boxed type, or this type if not primitive
     */
    CTypeI<T> tryToBox();

    /**
     * Gets all interfaces implemented by this type.
     *
     * @return a flow of interface types
     */
    Flow<CTypeI<?>> getInterfaces();

    /**
     * Creates a new instance of this type with the specified arguments.
     *
     * @param args the constructor arguments
     * @return a new instance
     */
    @SuppressWarnings("unchecked")
    T makeAuto(Object... args);

    /**
     * Gets a VarHandle for the specified field.
     *
     * @param field the field
     * @return a VarHandle for the field
     */
    VarHandle getFieldVarHandle(Field field);

    /**
     * Gets a VarHandle for the field with the specified name.
     *
     * @param name the field name
     * @return a VarHandle for the field
     */
    VarHandle getFieldVarHandle(String name);

    /**
     * Gets a supplier that can create instances of this type.
     *
     * @return a supplier for creating instances
     */
    ASupplier<?> maker();

    /**
     * Gets the constructor with the specified parameter types.
     *
     * @param args the parameter types
     * @return the matching constructor, or null if not found
     */
    Constructor<?> getConstructor(Class<?>... args);

    /**
     * Gets a MethodHandle for the constructor with the specified parameter types.
     *
     * @param args the parameter types
     * @return a MethodHandle for the constructor
     */
    MethodHandle maker(Class<?>... args);

    /**
     * Gets the enum values if this type represents an enum.
     *
     * @return an array of enum values, or null if not an enum
     */
    Enum<?>[] getEnumValues();

    /**
     * Checks if this type is part of the Aether framework.
     *
     * @return true if this is an Aether class, false otherwise
     */
    boolean isAetherClass();

    /**
     * Gets the canonical name of this type.
     *
     * @return the canonical name
     */
    String getCanonicalName();

    /**
     * Checks if this type is a parameterized class.
     *
     * @return true if this is a parameterized class, false otherwise
     */
    boolean isParametrizedClass();

    /**
     * Checks if this type is a type variable.
     *
     * @return true if this is a type variable, false otherwise
     */
    boolean isVariable();

    /**
     * Checks if this type is a parameterized type.
     *
     * @return true if this is a parameterized type, false otherwise
     */
    boolean isParameterizedType();

    /**
     * Gets the package name for this type.
     *
     * @return the package name
     */
    String getPackage();

    /**
     * Checks if the specified object is an instance of this type.
     *
     * @param o the object to check
     * @return true if the object is an instance of this type, false otherwise
     */
    boolean isInstance(Object o);

    /**
     * Compares two values of this type, potentially using custom comparison logic.
     *
     * @param val1 the first value
     * @param val2 the second value
     * @param consumer the comparison event consumer
     * @return true if the values are equal, false otherwise
     */
    boolean compareValues(T val1, T val2, CompareEvent consumer);

    /**
     * Converts this type to an array type.
     *
     * @return the array type
     */
    CTypeI<T[]> toArray();

    /**
     * Casts this type to the specified type.
     *
     * @param <C> the target type
     * @return the cast type
     */
    <C> CTypeI<C> cast();

    /**
     * Gets the class loader associated with this type.
     *
     * @return the class loader
     */
    ClassLoader getClassLoader();

    /**
     * Casts the specified object to this type.
     *
     * @param o the object to cast
     * @return the cast object
     */
    T cast(Object o);

    /**
     * Creates a type instance for the specified class.
     *
     * @param <T> the type
     * @param type the class
     * @return the type instance
     */
    static <T> CTypeI<T> of(Class<T> type) {
        return of((Type) type);
    }

    /**
     * Creates a type instance for the specified type.
     *
     * @param <T> the type
     * @param type the type
     * @return the type instance
     */
    static <T> CTypeI<T> of(Type type) {
        assert type != null;
        var res = CType.CACHE1.get(type);
        if (type instanceof TypeVariable<?>) {
            res = new CType.WType<Object>((TypeVariable<?>) type);
        } else {
            res = new CType.PType<Object>(type);
        }
        CType.CACHE1.put(type, res);
        return RU.cast(res);
    }

    /**
     * Creates a type instance for the specified annotated type.
     *
     * @param <T> the type
     * @param type the annotated type
     * @return the type instance
     */
    static <T> CTypeI<T> of(AnnotatedType type) {
        return new CType.AType<T>(type);
    }

    /**
     * Creates a type instance with the specified annotation, class, and type arguments.
     *
     * @param <T> the type
     * @param aa the annotation
     * @param type the class
     * @param args the type arguments
     * @return the type instance
     */
    static <T> CTypeI<T> of(Annotation aa, Class<T> type, AnnotatedType... args) {
        return of(TypeFactory.parameterizedAnnotatedClass(type, new Annotation[]{aa}, args));
    }

    /**
     * Creates a type instance with the specified annotations, class, and type arguments.
     *
     * @param aa the annotations
     * @param type the class
     * @param args the type arguments
     * @return the type instance
     */
    static CTypeI<Object> of(Annotation[] aa, Class<?> type, AnnotatedType... args) {
        return of(TypeFactory.parameterizedAnnotatedClass(type, aa, args));
    }

    /**
     * Creates a type instance with the specified annotations, class, and type arguments.
     *
     * @param <T> the type
     * @param aa the annotations
     * @param type the class
     * @param args the type arguments
     * @return the type instance
     */
    static <T> CTypeI<T> of(Annotation[] aa, Class<T> type, Type... args) {
        if (aa.length == 0) {
            return of(type, args);
        }
        return of(TypeFactory.parameterizedAnnotatedClass(type, aa, Flow.flow(args).map(GenericTypeReflector::annotate).toArray(AnnotatedType.class)));
    }

    /**
     * Creates a type instance with the specified annotations and type.
     *
     * @param <T> the type
     * @param aa the annotations
     * @param type the type
     * @return the type instance
     */
    static <T> CTypeI<T> of(Annotation[] aa, Type type) {
        return of(GenericTypeReflector.annotate(type, aa));
    }

    /**
     * Creates a type instance with the specified annotations and existing type.
     *
     * @param <T> the type
     * @param aa the annotations to add
     * @param type the existing type
     * @return the new type instance with added annotations
     */
    static <T> CTypeI<T> of(Annotation[] aa, CTypeI<T> type) {
        Annotation[] aaa = type.getAnnotations();
        if (aaa == null || aaa.length == 0) {
            aaa = aa;
        } else {
            Set<Annotation> aaaSet = new HashSet<>();
            Collections.addAll(aaaSet, aaa);
            Collections.addAll(aaaSet, aa);
            aaa = aaaSet.toArray(new Annotation[0]);
        }
        return of(GenericTypeReflector.updateAnnotations(type.toAnnotatedType(), aaa));
    }

    /**
     * Creates a type instance for the specified raw class and type arguments.
     *
     * @param <T> the type
     * @param raw the raw class
     * @param args the type arguments
     * @return the type instance
     */
    static <T> CTypeI<T> of(Class<? super T> raw, Type... args) {
        if (args.length == 0) {
            return CTypeI.<T>of(raw);
        }
        return of(TypeFactory.parameterizedClass(raw, args));
    }

    /**
     * Creates a type instance for the specified raw class and type arguments.
     *
     * @param <T> the type
     * @param raw the raw class
     * @param args the type arguments
     * @return the type instance
     */
    static <T> CTypeI<T> of(Class<? super T> raw, CTypeI<?>... args) {
        return of(TypeFactory.parameterizedAnnotatedClass(raw, new Annotation[]{}, Flow.flow(args).map(CTypeI::tryToBox).map(CTypeI::toAnnotatedType).toArray(AnnotatedType.class)));
    }

    /**
     * Creates a type instance for the specified raw annotated type and type arguments.
     *
     * @param <T> the type
     * @param raw the raw annotated type
     * @param args the type arguments
     * @return the type instance
     */
    static <T> CTypeI<T> of(AnnotatedType raw, AnnotatedType[] args) {
        return of(TypeFactory.parameterizedAnnotatedClass(of(raw).getRaw(), raw.getAnnotations(), args));
    }

    /**
     * Creates a type instance for the specified raw annotated type and type arguments.
     *
     * @param <T> the type
     * @param raw the raw annotated type
     * @param args the type arguments
     * @return the type instance
     */
    static <T> CTypeI<T> of(AnnotatedType raw, CTypeI<Object>[] args) {
        return of(TypeFactory.parameterizedAnnotatedClass(of(raw).getRaw(), raw.getAnnotations(), Flow.flow(args).map(CTypeI::toAnnotatedType).toArray(AnnotatedType.class)));
    }

    /**
     * Creates an array type for the specified component type.
     *
     * @param <T> the component type
     * @param component the component type
     * @return the array type
     */
    static <T> CTypeI<T[]> buildArray(Type component) {
        return of(TypeFactory.arrayOf(component));
    }

    /**
     * Creates an array type for the specified component type with annotations.
     *
     * @param <T> the component type
     * @param component the component type
     * @param annotations the annotations
     * @return the array type
     */
    static <T> CTypeI<T[]> buildArray(AnnotatedType component, Annotation[] annotations) {
        return of(TypeFactory.arrayOf(component, annotations));
    }

    /**
     * Creates an array type for the specified component type.
     *
     * @param <T> the component type
     * @param component the component type
     * @return the array type
     */
    static <T> CTypeI<T[]> buildArray(CTypeI<T> component) {
        return of(TypeFactory.arrayOf(component.toAnnotatedType(), new Annotation[0]));
    }

    /**
     * Creates an array type for the specified component type with annotations.
     *
     * @param component the component type
     * @param annotations the annotations
     * @return the array type
     */
    static CTypeI<Object> buildArray(CTypeI<Object> component, Annotation[] annotations) {
        return of(TypeFactory.arrayOf(component.toAnnotatedType(), annotations));
    }

    /**
     * Creates a type instance for the specified type and parameters.
     *
     * @param <T> the type
     * @param res the type
     * @param pp the parameters
     * @return the type instance
     */
    static <T> CTypeI<T> of(Type res, List<Type> pp) {
        return RU.cast(of((Class<?>) res, pp.toArray(new Type[0])));
    }

    /**
     * Converts an annotation to its code representation.
     *
     * @param a the annotation
     * @return the code representation
     */
    static String annotationToCode(Annotation a) {
        StringBuilder sb = new StringBuilder();
        annotationToCode(sb, a);
        return sb.toString();
    }

    /**
     * Converts an annotation to its build code representation.
     *
     * @param a the annotation
     * @return the build code representation
     */
    static String annotationToBuildCode(Annotation a) {
        StringBuilder sb = new StringBuilder();
        annotationToCode(sb, a);
        return sb.toString();
    }

    /**
     * Appends the build code representation of an annotation to the specified string builder.
     *
     * @param sb the string builder
     * @param a the annotation
     */
    static void annotationToBuildCode(StringBuilder sb, Annotation a) {
        sb.append(CLASS_NAME).append(".annotation(").append(a.annotationType().getCanonicalName()).append(".class,").append(Map.class.getCanonicalName()).append(".of(");
        var mm = a.annotationType().getDeclaredMethods();
        boolean first = true;
        try {
            for (var m : mm) {
                if (first) {
                    first = false;
                } else {
                    sb.append(',');
                }
                m.setAccessible(true);
                var rt = m.getReturnType();
                final var v = m.invoke(a);
                if (v == null) {
                    continue;
                }
                sb.append("\"").append(m.getName()).append("\"");
                sb.append(',');
                if (rt.isArray()) {
                    sb.append("new ").append(CTypeI.of(rt).declareCode()).append("[]");
                    sb.append("{");
                    boolean first1 = true;
                    System.exit(1);
                    for (int i = 0; i < Array.getLength(v); i++) {
                        if (first1) {
                            first1 = false;
                        } else {
                            sb.append(",");
                        }
                    }
                    sb.append("}");
                } else if (Annotation.class.isAssignableFrom(rt)) {
                    annotationToCode(sb, (Annotation) v);
                } else if (rt == String.class) {
                    sb.append('"');
                    sb.append(String.valueOf(v).replaceAll("([\"\\\\])", "\\$1"));
                    sb.append('"');
                } else {
                    sb.append(v);
                }
            }
        } catch (Exception e) {
            throw new RuntimeException();
        }
        sb.append("))");
    }

    /**
     * Appends the code representation of an annotation to the specified string builder.
     *
     * @param sb the string builder
     * @param a the annotation
     */
    static void annotationToCode(StringBuilder sb, Annotation a) {
        sb.append('@');
        var t = a.annotationType();
        sb.append(t.getCanonicalName());
        try {
            var mm = t.getDeclaredMethods();
            if (mm.length > 0) {
                sb.append('(');
                boolean first = true;
                for (var m : mm) {
                    if (first) {
                        first = false;
                    } else {
                        sb.append(',');
                    }
                    m.setAccessible(true);
                    var rt = m.getReturnType();
                    final var v = m.invoke(a);
                    if (v == null) {
                        continue;
                    }
                    sb.append(m.getName());
                    sb.append('=');
                    if (rt.isArray()) {
                        sb.append('{');
                        rt = rt.getComponentType();
                        boolean first2 = true;
                        if (Annotation.class.isAssignableFrom(rt)) {
                            for (int i = 0; i < Array.getLength(v); i++) {
                                if (first2) {
                                    first2 = false;
                                } else {
                                    sb.append(',');
                                }
                                annotationToCode(sb, (Annotation) v);
                            }
                        } else if (rt == String.class) {
                            for (int i = 0; i < Array.getLength(v); i++) {
                                if (first2) {
                                    first2 = false;
                                } else {
                                    sb.append(',');
                                }
                                sb.append('"');
                                sb.append(v);
                                sb.append('"');
                            }
                        } else {
                            for (int i = 0; i < Array.getLength(v); i++) {
                                if (first2) {
                                    first2 = false;
                                } else {
                                    sb.append(',');
                                }
                                sb.append(v);
                            }
                        }
                        sb.append('}');
                    } else if (Annotation.class.isAssignableFrom(rt)) {
                        annotationToCode(sb, (Annotation) v);
                    } else if (rt == String.class) {
                        sb.append('"');
                        sb.append(v);
                        sb.append('"');
                    } else {
                        sb.append(v);
                    }
                }
                sb.append(')');
            }
        } catch (IllegalAccessException | InvocationTargetException e) {
            RU.error(e);
        }
    }

    /**
     * Creates an annotation instance of the specified type with the given values.
     *
     * @param <T> the annotation type
     * @param type the annotation class
     * @param args the annotation values
     * @return the annotation instance
     */
    static <T extends Annotation> T annotation(Class<T> type, Map<String, Object> args) {
        try {
            return TypeFactory.annotation(type, args);
        } catch (AnnotationFormatException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Gets the default (no-arg) constructor for the specified class.
     *
     * @param c the class
     * @return the default constructor, or null if not found
     */
    static Constructor<?> getDefaultConstructor(Class<?> c) {
        for (var cc : c.getDeclaredConstructors()) {
            if (cc.getParameterCount() == 0) return cc;
        }
        return null;
    }

    /**
     * Collects all fields (including inherited) for the specified class.
     *
     * @param raw the class
     * @param res the list to store fields in
     */
    static void getAllFields(Class<?> raw, List<Field> res) {
        int p = 0;
        for (var f : raw.getDeclaredFields()) {
            if (Modifier.isStatic(f.getModifiers())) continue;
            res.add(p++, f);
        }
        if (!raw.isInterface() && raw.getSuperclass() != null) {
            getAllFields(raw.getSuperclass(), res);
        }
    }

    /**
     * Collects all methods (including inherited) for the specified class.
     *
     * @param raw the class
     * @param res the set to store methods in
     */
    static void getAllMethods(Class<?> raw, Set<Method> res) {
        for (var m : raw.getDeclaredMethods()) {
            if (Modifier.isStatic(m.getModifiers())) continue;
            res.add(m);
        }
        if (!raw.isInterface() && raw.getSuperclass() != null) {
            getAllMethods(raw.getSuperclass(), res);
        }
        for (var i : raw.getInterfaces()) {
            getAllMethods(i, res);
        }
    }

    /**
     * Gets a flow of all parent classes and interfaces for the specified type.
     *
     * @param type the starting type
     * @return a flow of parent types
     */
    static Flow<Class<?>> getAllParentClasses(Class<?> type) {
        Queue<Class<?>> q = new ArrayDeque<>();
        q.add(type);
        return new Flow<>() {
            @Override
            public boolean hasNext() {
                return !q.isEmpty();
            }

            @Override
            public Class<?> next() {
                var t = q.poll();
                assert t != null;
                var sc = t.getSuperclass();
                if (sc != null) {
                    q.add(sc);
                }
                Collections.addAll(q, t.getInterfaces());
                return t;
            }
        };
    }

    /**
     * Gets a flow of all annotations on the specified type and its parents.
     *
     * @param type the type
     * @return a flow of annotations
     */
    static Flow<Annotation> getAllAnnotations(Class<?> type) {
        return getAllParentClasses(type).flatMap2(Class::getAnnotations);
    }

    /**
     * Interface for field access operations.
     *
     * @param <T2> the field type
     */
    interface FieldAccessor<T2> extends ToString {
        /**
         * Gets the field value from the specified owner object.
         *
         * @param owner the owner object
         * @return the field value
         */
        T2 get(Object owner);

        /**
         * Sets the field value on the specified owner object.
         *
         * @param owner the owner object
         * @param value the new field value
         */
        void set(Object owner, T2 value);
    }

    /**
     * Interface for comparison events.
     */
    interface CompareEvent {
        /**
         * Called when a field comparison occurs.
         *
         * @param field the field being compared
         * @param val1 the first object
         * @param val2 the second object
         * @param fVal1 the first field value
         * @param fVal2 the second field value
         * @throws Exception if an error occurs during comparison
         */
        void compareEvent(Field field, Object val1, Object val2, Object fVal1, Object fVal2) throws Exception;
    }
}