package io.aether.utils;

import java.util.Arrays;

public class HexUtils {
    private static final char[] HEX_ARRAY = "0123456789ABCDEF".toCharArray();

    /**
     * Converts a hexadecimal string to a byte array.
     *
     * @param s The hexadecimal string
     * @return A byte array representation of the hexadecimal string
     */

    public static byte[] hexToBytes(CharSequence s) {
        byte[] byteArray = new byte[s.length() / 2];
        for (int i = 0; i < s.length(); i = i + 2) {
            var c1 = Arrays.binarySearch(HEX_ARRAY, s.charAt(i));
            var c2 = Arrays.binarySearch(HEX_ARRAY, s.charAt(i + 1));
            byteArray[i / 2] = (byte) ((c1 << 4) | c2);
        }
        return byteArray;
    }

    /**
     * Converts a byte array to a hexadecimal string.
     *
     * @param bytes The byte array
     * @return A hexadecimal string representation of the byte array
     */
    public static String toHexString(byte[] bytes) {
        return new String(toHexString2(bytes));
    }

    public static char[] toHexString2(byte[] bytes) {
        return toHexString2(bytes, 0, bytes.length);
    }

    /**
     * Converts a byte array to a hexadecimal string with a specified offset and end index.
     *
     * @param bytes      The byte array
     * @param offset     The starting index
     * @param endIndex   The ending index (exclusive)
     * @return A hexadecimal string representation of the specified portion of the byte array
     */
    public static String toHexString(byte[] bytes, int offset, int endIndex) {
        return new String(toHexString2(bytes, offset, endIndex));
    }

    /**
     * Converts a byte array to a hexadecimal string with a specified offset and end index.
     *
     * @param bytes      The byte array
     * @param offset     The starting index
     * @param endIndex   The ending index (exclusive)
     * @return A hexadecimal string representation of the specified portion of the byte array
     */
    public static char[] toHexString2(byte[] bytes, int offset, int endIndex) {
        if (bytes == null || bytes.length == 0) return new char[0];
        int len = endIndex - offset;
        char[] res = new char[len * 2];
        toHexString(bytes, offset, endIndex, res, 0);
        return res;
    }

    /**
     * Converts a byte array to a hexadecimal string with a specified offset and end index.
     *
     * @param bytes      The byte array
     * @param offset     The starting index
     * @param endIndex   The ending index (exclusive)
     * @param hexChars   The character array to store the result
     * @param destOffset The starting position in the destination array
     */
    public static void toHexString(byte[] bytes, int offset, int endIndex, char[] hexChars, int destOffset) {
        if (bytes == null || bytes.length == 0) return;
        int len = endIndex - offset;
        assert hexChars.length - destOffset >= len * 2 : hexChars.length + "-" + destOffset + "<" + len + "*2==" + (len * 2);
        for (int j = 0; j < len; j++) {
            int v = bytes[j + offset] & 0xFF;
            hexChars[destOffset + j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[destOffset + j * 2 + 1] = HEX_ARRAY[v & 0x0F];
        }
    }

    /**
     * Converts an integer to a hexadecimal string.
     *
     * @param src The integer
     * @param hexChars The character array to store the result
     */
    public static void toHexString(int src, AString hexChars) {
        src = src & 0xff;
        hexChars.add(HEX_ARRAY[src >>> 4]);
        hexChars.add(HEX_ARRAY[src & 0x0F]);
    }

    /**
     * Converts a long integer to a hexadecimal string.
     *
     * @param src The long integer
     * @param hexChars The character array to store the result
     */
    public static void toHexString(long src, AString hexChars) {
        toHexString((int) src, hexChars);
    }

    /**
     * Converts a byte array to a hexadecimal string with a specified offset and end index.
     *
     * @param bytes      The byte array
     * @param offset     The starting index
     * @param endIndex   The ending index (exclusive)
     * @param hexChars   The character array to store the result
     */
    public static void toHexString(byte[] bytes, int offset, int endIndex, AString hexChars) {
        if (bytes == null || bytes.length == 0) return;
        int len = endIndex - offset;
        for (int j = 0; j < len; j++) {
            int v = bytes[j + offset] & 0xFF;
            toHexString(v, hexChars);
        }
    }

    /**
     * Converts an integer to a hexadecimal string with a specified destination array and offset.
     *
     * @param bytes The integer
     * @param hexChars The character array to store the result
     * @param destOffset The starting position in the destination array
     */
    public static void toHexString(int bytes, char[] hexChars, int destOffset) {
        final int len = 4;
        assert hexChars.length - destOffset >= len * 2;
        for (int j = 0; j < len; j++) {
            int v = (bytes >> 8 * j) & 0xFF;
            hexChars[destOffset + j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[destOffset + j * 2 + 1] = HEX_ARRAY[v & 0x0F];
        }
    }

    /**
     * Converts a long integer to a hexadecimal string with a specified destination array and offset.
     *
     * @param bytes The long integer
     * @param hexChars The character array to store the result
     * @param destOffset The starting position in the destination array
     */
    public static void toHexString(long bytes, char[] hexChars, int destOffset) {
        final int len = 8;
        assert hexChars.length - destOffset >= len * 2;
        for (int j = 0; j < len; j++) {
            int v = (int) ((bytes >> 8 * j) & 0xFF);
            hexChars[destOffset + j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[destOffset + j * 2 + 1] = HEX_ARRAY[v & 0x0F];
        }
    }
}
