package io.aether.utils;

import io.aether.utils.RU;
import io.aether.utils.interfaces.ARunnable;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledFuture;

/**
 * A thread-safe, concurrent Map implementation with weak keys.
 *
 * <p>This implementation is designed for high-concurrency environments: it relies on
 * lock-free data structures (ConcurrentHashMap) and a single global scheduled task for
 * cleanup, avoiding synchronized blocks and explicit locks on all critical paths.</p>
 *
 * @param <K> the type of keys maintained by this map
 * @param <V> the type of mapped values
 */
public class WeakConcurrentHashMap<K, V> implements ConcurrentMap<K, V> {

    // --- Static Global Cleanup Infrastructure (Lock-Free) ---

    /**
     * A list of weak references (trackers) to all active WeakConcurrentHashMap instances.
     * CopyOnWriteArrayList is used for concurrent iteration/modification (registration/collection).
     */
    private static final List<MapTracker> ALL_TRACKERS = new CopyOnWriteArrayList<>();

    /** The ReferenceQueue used to track when the MapTracker itself should be removed (i.e., when the map instance is collected). */
    private static final ReferenceQueue<WeakConcurrentHashMap<?, ?>> TRACKER_QUEUE = new ReferenceQueue<>();

    /** Global scheduled task for periodic queue cleaning. */
    private static final ScheduledFuture<?> CLEANER_TASK;

    /**
     * Internal class to track a WeakConcurrentHashMap instance.
     * When the referent (map) is collected, this tracker is moved to the TRACKER_QUEUE.
     */
    private static final class MapTracker extends WeakReference<WeakConcurrentHashMap<?, ?>> {
        /** The ReferenceQueue of the specific map instance, needed for cleanup. */
        final ReferenceQueue<?> mapRefQueue;

        MapTracker(WeakConcurrentHashMap<?, ?> map, ReferenceQueue<?> q) {
            super(map, TRACKER_QUEUE); // Register with the global TRACKER_QUEUE
            this.mapRefQueue = map.referenceQueue;
        }
    }

    static {
        // Initialize the global scheduled task to run every 1000 milliseconds (1 second).
        // The task itself uses non-blocking operations.
        CLEANER_TASK = RU.scheduleAtFixedRate(1000, () -> {
            // 1. Flush entries (WeakKeys) for all currently active map instances.
            for (MapTracker tracker : ALL_TRACKERS) {
                WeakConcurrentHashMap<?, ?> map = tracker.get();
                if (map != null) {
                    map.flushMem(); // Clean the map's internal key queue (uses non-blocking poll/remove)
                }
            }

            // 2. Flush map trackers (WeakReferences to map instances) from the global list.
            // This uses non-blocking poll and CopyOnWriteArrayList's thread-safe remove.
            MapTracker tracker;
            while ((tracker = (MapTracker) TRACKER_QUEUE.poll()) != null) {
                ALL_TRACKERS.remove(tracker);
            }
        });
    }

    // --- Instance Fields ---

    /** The backing ConcurrentMap. Keys are wrapped in WeakKey; values are strong references. */
    private final ConcurrentMap<WeakKey<K>, V> body;

    /** The queue to which collected WeakKeys (entries) are enqueued by the Garbage Collector. */
    private final ReferenceQueue<K> referenceQueue = new ReferenceQueue<>();

    /**
     * Creates a new WeakConcurrentHashMap and registers it for global cleanup.
     */
    public WeakConcurrentHashMap() {
        this.body = new ConcurrentHashMap<>();
        // Register this instance using MapTracker for global cleanup
        ALL_TRACKERS.add(new MapTracker(this, referenceQueue));
    }

    /**
     * Processes all references currently waiting in the map's internal ReferenceQueue.
     * This method is called periodically by the global static CLEANER_TASK.
     * Uses only non-blocking operations (poll and ConcurrentHashMap.remove).
     */
    private void flushMem() {
        WeakKey<K> weakKey;
        while ((weakKey = RU.cast( referenceQueue.poll())) != null) {
            body.remove(weakKey);
        }
    }

    /**
     * A WeakReference subclass that stores the key's hashCode and a strong reference
     * to the key (or a lookup key) for correct {@code equals()} and {@code hashCode()} behavior.
     *
     * @param <K> the type of the key
     */
    private static class WeakKey<K> extends WeakReference<K> {
        final int hash;
        /** A strong reference to the object for identity when the reference is alive,
         * or the key object itself if this is a temporary sample key. */
        final Object identity;

        /**
         * Creates a WeakKey for storage in the map.
         * @param referent the key object to be weakly referenced
         * @param q the ReferenceQueue for cleanup notifications
         */
        public WeakKey(K referent, ReferenceQueue<K> q) {
            super(referent, q);
            this.hash = referent.hashCode();
            this.identity = referent;
        }

        /**
         * Creates a temporary WeakKey for lookup (get, remove, containsKey).
         * @param referent the object to use for comparison
         */
        @SuppressWarnings("unchecked")
        public WeakKey(Object referent) {
            super(null, null);
            this.hash = referent.hashCode();
            this.identity = (K) referent;
        }

        @Override
        public int hashCode() {
            return hash;
        }

        /**
         * Compares this WeakKey with another object for equality.
         * @param obj the object to compare against
         * @return true if the underlying key objects are equal, false otherwise
         */
        @Override
        public boolean equals(Object obj) {
            if (obj == this) return true;
            if (!(obj instanceof WeakKey)) return false;

            WeakKey<?> other = (WeakKey<?>) obj;

            if (this.hash != other.hash) return false;

            Object t = this.get();
            Object o = other.get();
            if (o == null && other.identity != null) {
                o = other.identity;
            }

            if (t != null && o != null) {
                return t.equals(o);
            }

            return false;
        }
    }

    /**
     * Helper method to create a temporary WeakKey object for map lookups.
     * @param key the key object for the lookup
     * @return a temporary WeakKey instance
     */
    @SuppressWarnings("unchecked")
    private WeakKey<K> makeSampleKey(Object key) {
        return new WeakKey(key);
    }

    // --- ConcurrentMap Methods (standard implementation using ConcurrentHashMap) ---

    /** {@inheritDoc} */
    @Override
    public V put(@NotNull K key, V value) {
        return body.put(new WeakKey<>(key, referenceQueue), value);
    }

    /** {@inheritDoc} */
    @Override
    public V get(Object key) {
        return body.get(makeSampleKey(key));
    }

    /** {@inheritDoc} */
    @Override
    public V remove(Object key) {
        return body.remove(makeSampleKey(key));
    }

    /** {@inheritDoc} */
    @Override
    public int size() {
        return body.size();
    }

    /** {@inheritDoc} */
    @Override
    public boolean isEmpty() {
        return body.isEmpty();
    }

    /** {@inheritDoc} */
    @Override
    public boolean containsKey(Object key) {
        return body.containsKey(makeSampleKey(key));
    }

    /** {@inheritDoc} */
    @Override
    public boolean containsValue(Object value) {
        return body.containsValue(value);
    }

    /** {@inheritDoc} */
    @Override
    public V putIfAbsent(@NotNull K key, V value) {
        return body.putIfAbsent(new WeakKey<>(key, referenceQueue), value);
    }

    /** {@inheritDoc} */
    @Override
    public boolean remove(@NotNull Object key, Object value) {
        return body.remove(makeSampleKey(key), value);
    }

    /** {@inheritDoc} */
    @Override
    public boolean replace(@NotNull K key, @NotNull V oldValue, @NotNull V newValue) {
        return body.replace(new WeakKey<>(key, referenceQueue), oldValue, newValue);
    }

    /** {@inheritDoc} */
    @Override
    public V replace(@NotNull K key, @NotNull V value) {
        return body.replace(new WeakKey<>(key, referenceQueue), value);
    }

    // --- Collection Views (standard implementation with filtering) ---

    /** {@inheritDoc} */
    @Override
    public void putAll(@NotNull Map<? extends K, ? extends V> m) {
        m.forEach(this::put);
    }

    /** {@inheritDoc} */
    @Override
    public void clear() {
        body.clear();
    }

    /**
     * {@inheritDoc}
     *
     * <p>The set is backed by the map. The iterator filters out keys that have been garbage collected ("dead" entries).</p>
     */
    @Override
    public @NotNull Set<K> keySet() {
        return new AbstractSet<>() {
            @Override
            public int size() { return body.size(); }
            @Override
            public boolean contains(Object o) { return WeakConcurrentHashMap.this.containsKey(o); }

            @Override
            public @NotNull Iterator<K> iterator() {
                Iterator<WeakKey<K>> it = body.keySet().iterator();
                return new Iterator<K>() {
                    K nextKey = null;

                    @Override
                    public boolean hasNext() {
                        if (nextKey != null) return true;

                        while (it.hasNext()) {
                            WeakKey<K> weakKey = it.next();
                            K k = weakKey.get();
                            if (k != null) {
                                nextKey = k;
                                return true;
                            }
                        }
                        return false;
                    }

                    @Override
                    public K next() {
                        if (!hasNext()) throw new NoSuchElementException();
                        K k = nextKey;
                        nextKey = null;
                        return k;
                    }

                    @Override
                    public void remove() { it.remove(); }
                };
            }
        };
    }

    /**
     * {@inheritDoc}
     * The returned collection contains strong references to the values.
     */
    @Override
    public @NotNull Collection<V> values() {
        return body.values();
    }

    /**
     * {@inheritDoc}
     *
     * <p>The set is backed by the map. The iterator filters out entries whose keys have been garbage collected.</p>
     */
    @Override
    public @NotNull Set<Entry<K, V>> entrySet() {
        return new AbstractSet<Entry<K, V>>() {
            @Override
            public int size() { return body.size(); }

            @Override
            public boolean contains(Object o) {
                if (!(o instanceof Map.Entry)) return false;
                Map.Entry<?, ?> e = (Map.Entry<?, ?>) o;
                V value = WeakConcurrentHashMap.this.get(e.getKey());
                return Objects.equals(value, e.getValue());
            }

            @Override
            public @NotNull Iterator<Entry<K, V>> iterator() {
                Iterator<Entry<WeakKey<K>, V>> it = body.entrySet().iterator();
                return new Iterator<Entry<K, V>>() {
                    Entry<K, V> nextEntry = null;

                    @Override
                    public boolean hasNext() {
                        if (nextEntry != null) return true;

                        while (it.hasNext()) {
                            Entry<WeakKey<K>, V> entry = it.next();
                            K k = entry.getKey().get();
                            if (k != null) {
                                nextEntry = new SimpleEntry<>(k, entry.getValue());
                                return true;
                            }
                        }
                        return false;
                    }

                    @Override
                    public Entry<K, V> next() {
                        if (!hasNext()) throw new NoSuchElementException();
                        Entry<K, V> e = nextEntry;
                        nextEntry = null;
                        return e;
                    }

                    @Override
                    public void remove() { it.remove(); }
                };
            }
        };
    }

    /**
     * A utility class that implements the {@link Map.Entry} interface.
     * @param <K> the type of the key
     * @param <V> the type of the value
     */
    private static class SimpleEntry<K, V> implements Entry<K, V> {
        private final K key;
        private V value;

        public SimpleEntry(K key, V value) {
            this.key = key;
            this.value = value;
        }

        @Override
        public K getKey() {
            return key;
        }

        @Override
        public V getValue() {
            return value;
        }

        @Override
        public V setValue(V value) {
            V oldValue = this.value;
            this.value = value;
            return oldValue;
        }
    }

    /**
     * Shuts down the global scheduled cleaner task.
     * This method should be called only once when the application exits.
     */
    public static void shutdownGlobalCleaner() {
        if (CLEANER_TASK != null && !CLEANER_TASK.isDone()) {
            CLEANER_TASK.cancel(true);
        }
    }
}