package io.aether.utils.dataio;

public interface DataIn {
    /**
     * Returns the size of the data that can be read from this input stream, in bytes.
     */
    int getSizeForRead();

    /**
     * Reads a byte array from this input stream and returns the number of bytes actually read.
     * @param b The byte array to read into.
     * @return The number of bytes actually read.
     */
    default int read(byte[] b) {
        return read(b, 0, b.length);
    }

    /**
     * Reads an integer array from this input stream and returns the number of integers actually read.
     * @param b The integer array to read into.
     * @return The number of integers actually read.
     */
    default int read(int[] b) {
        return read(b, 0, b.length);
    }

    /**
     * Reads a byte array from this input stream and returns the number of bytes actually read.
     * @param b The byte array to read into.
     * @param offset The starting position in the array to write to.
     * @param len The maximum number of bytes to read.
     * @return The number of bytes actually read.
     */
    default int read(byte[] b, int offset, int len) {
        // Check if there is enough space in the buffer to read all the requested data
        int endIndex = Math.min(offset + len, b.length);
        for (int i = offset; i < endIndex; i++) {
            // Read a byte from the input stream and write it to the array
            b[i] = readByte();
        }
        return endIndex - offset;
    }

    /**
     * Reads an integer array from this input stream and returns the number of integers actually read.
     * @param b The integer array to read into.
     * @param offset The starting position in the array to write to.
     * @param len The maximum number of integers to read.
     * @return The number of integers actually read.
     */
    default int read(int[] b, int offset, int len) {
        // Check if there is enough space in the buffer to read all the requested data
        int endIndex = Math.min(offset + len, b.length);
        for (int i = offset; i < endIndex; i++) {
            // Read an integer from the input stream and write it to the array
            b[i] = readInt();
        }
        return endIndex - offset;
    }

    /**
     * Returns true if this input stream is empty, false otherwise.
     */
    default boolean isEmpty() {
        return getSizeForRead() == 0;
    }

    /**
     * Skips the specified number of bytes in this input stream.
     * @param n The number of bytes to skip.
     */
    default void skipBytes(int n) {
        // Read each byte from the input stream and discard it
        for (int i = 0; i < n; i++) {
            readUByte();
        }
    }

    /**
     * Returns true if this input stream is readable, false otherwise.
     */
    default boolean isReadable() {
        return getSizeForRead() != 0;
    }

    /**
     * Reads a byte from this input stream and returns it as an integer.
     */
    default boolean readBoolean() {
        // Return true if the first bit of the byte is set, false otherwise
        return readUByte() != 0;
    }

    /**
     * Reads a byte from this input stream and returns it as a byte value.
     */
    default byte readByte() {
        // Return the byte cast to a byte value
        return (byte) readUByte();
    }

    /**
     * Returns the next byte of data from this input stream, casting it to an unsigned byte value in the range 0-255.
     */
    int readUByte();

    /**
     * Reads a subarray of bytes from this input stream and returns the number of bytes actually read.
     * @param length The maximum number of bytes to read.
     */
    default DataIO readSubData(int length) {
        // Create a new data output object with the specified size
        var res = new DataInOutStatic(length);
        // Read all the requested data into the buffer
        read(res.data);
        return res;
    }

    /**
     * Returns the next short value from this input stream, casting it to a short value in the range -32768 to 32767.
     */
    default short readShort() {
        // Return the short cast to a short value
        return (short) readUShort();
    }

    /**
     * Returns the next unsigned short value from this input stream, casting it to an int value in the range 0-65535.
     */
    default int readUShort() {
        // Initialize a variable to hold the result of reading two bytes from the input stream
        int res = 0;
        // Read each byte from the input stream and shift it into the result
        for (int i = 0; i < 2; i++) {
            res = res | ((readUByte() & 255) << (8 * i));
        }
        return res;
    }

    /**
     * Returns the next character value from this input stream, casting it to a char value in the range '\u0000' to '\uffff'.
     */
    default char readChar() {
        // Return the byte cast to a char value
        return (char) readUByte();
    }

    /**
     * Returns the next int value from this input stream, casting it to an int value in the range -2147483648 to 2147483647.
     */
    default int readInt() {
        // Initialize a variable to hold the result of reading four bytes from the input stream
        int res = 0;
        // Read each byte from the input stream and shift it into the result
        for (int i = 0; i < 4; i++) {
            res = res | (readUByte() << (8 * i));
        }
        return res;
    }

    /**
     * Returns the next long value from this input stream, casting it to a long value in the range -9223372036854775808 to 9223372036854775807.
     */
    default long readUInt() {
        // Initialize a variable to hold the result of reading four bytes from the input stream
        long res = 0;
        // Read each byte from the input stream and shift it into the result
        for (int i = 0; i < 4; i++) {
            res = res | (((long) readUByte()) << (8 * i));
        }
        return res;
    }

    /**
     * Returns the next long value from this input stream, casting it to a long value in the range -9223372036854775808 to 9223372036854775807.
     */
    default long readLong() {
        // Initialize a variable to hold the result of reading eight bytes from the input stream
        long res = 0;
        // Read each byte from the input stream and shift it into the result
        for (int i = 0; i < 8; ++i) {
            res = res | (((long) readUByte()) << (8 * i));
        }
        return res;
    }

    /**
     * Returns the next float value from this input stream, casting it to a float value in the range -3.4028235E+38F to 3.4028235E+38F.
     */
    default float readFloat() {
        // Return the int cast to a float value
        return Float.intBitsToFloat(readInt());
    }

    /**
     * Returns the next double value from this input stream, casting it to a double value in the range -1.7976931348623157E+308D to 1.7976931348623157E+308D.
     */
    default double readDouble() {
        // Return the long cast to a double value
        return Double.longBitsToDouble(readLong());
    }

    /**
     * Reads a string from this input stream and returns it.
     */
    default String readString1() {
        // Read the length of the string from the input stream
        var len = readUByte();
        // Create a new byte array to hold the data for the string
        byte[] data = new byte[len];
        // Read all the requested data into the buffer
        read(data);
        // Return the string created from the bytes in the buffer
        return new String(data);
    }

    /**
     * Returns a copy of this input stream as an array of bytes.
     */
    default byte[] toArray() {
        // Create a new byte array with the size of the data that can be read from this input stream
        byte[] d = new byte[getSizeForRead()];
        // Read all the requested data into the buffer
        read(d);
        return d;
    }

    /**
     * Skips all bytes in this input stream.
     */
    default void skipAllBytes() {
        // Skip each byte from the input stream and discard it
        for (int i = 0; i < getSizeForRead(); i++) {
            readUByte();
        }
    }

    /**
     * Returns the index within this input stream of the first occurrence of the specified byte, or -1 if no such byte exists.
     */
    int indexOf(int limit, byte val);

    default byte[] readBytes(int len){
        byte[] res=new byte[len];
        read(res);
        return res;
    }
}