package io.aether.utils.dataio;

import io.aether.utils.HexUtils;
import io.aether.utils.RU;
import org.jetbrains.annotations.NotNull;

public class DataInOutStatic implements DataIO {
    public final byte[] data;
    public int writePos;
    public int readPos;

    @Override
    public void skipBytes(int n) {
        readPos += n;
        if (data.length < readPos) {
            throw new IllegalStateException("data.length(" + data.length + ") < readPos(" + readPos + ")");
        }
    }

    public DataInOutStatic(int initSize) {
        this(new byte[initSize], 0, 0);
    }

    public DataInOutStatic(byte[] ar) {
        this(ar, 0, ar.length);
    }

    public DataInOutStatic(byte[] ar, int readPos, int writePos) {
        data = ar;
        this.readPos = readPos;
        this.writePos = writePos;
    }

    @Override
    public String toString() {
        return HexUtils.toHexString(data, readPos, writePos);
    }

    public byte[] getData() {
        return data;
    }

    public int getReadPos() {
        return readPos;
    }

    public void setReadPos(int readPos) {
        this.readPos = readPos;
    }

    public int getWritePos() {
        return writePos;
    }

    public void setWritePos(int writePos) {
        assert writePos <= data.length;
        this.writePos = writePos;
    }

    @Override
    public int getSizeForWrite() {
        return data.length - writePos;
    }

    @Override
    public boolean isWritable() {
        return writePos < data.length;
    }

    public void clear() {
        writePos = 0;
        readPos = 0;
    }

    @Override
    public int getSizeForRead() {
        return writePos - readPos;
    }

    @Override
    public int read(byte[] b, int offset, int len) {
        var a = getSizeForRead();
        int l = Math.min(Math.min(len, b.length), a);
        if (l > 0) {
            System.arraycopy(data, readPos, b, offset, l);
            readPos += l;
        }
        return l;
    }

    @Override
    public void write(byte @NotNull [] b) {
        if (b.length > data.length - writePos) {
            throw new IllegalStateException();
        }
        System.arraycopy(b, 0, data, writePos, b.length);
        writePos += b.length;
    }

    @Override
    public int write(byte @NotNull [] b, int off, int len) {
        var l = Math.min(len, (data.length - writePos));
        System.arraycopy(b, off, data, writePos, l);
        writePos += l;
        return l;
    }

    @Override
    public int readInt() {
        if (readPos + 4 > writePos) {
            throw new RuntimeException();
        }
        int res = 0;
        for (int i = 0; i < 4; i++) {
            res = res | (Byte.toUnsignedInt(data[readPos++]) << (8 * i));
        }
        return res;
    }

    @Override
    public long readLong() {
        if (readPos + 8 > writePos) throw new RuntimeException();
        long res = 0;
        for (int i = 0; i < 8; i++) {
            res = res | (Byte.toUnsignedLong(data[readPos++]) << (8 * i));
        }
        return res;
    }

    @Override
    public int readUByte() {
        if (readPos >= writePos)
            throw new IllegalStateException("Read position [" + readPos + "] > write position [" + writePos + "]");
        return Byte.toUnsignedInt(data[readPos++]);
    }

    @Override
    public byte[] toArray() {
        if (readPos == 0 && writePos == data.length) return data;
        return DataIO.super.toArray();
    }

    @Override
    public int indexOf(int limit, byte val) {
        for (int i = readPos; i < writePos && i < limit; i++) {
            if (data[i] == val) return i;
        }
        return -1;
    }

    @Override
    public void writeByte(int v) {
        data[writePos++] = (byte) v;
    }

    public int total() {
        return data.length;
    }

    public static DataInOutStatic ofBytes(String s) {
        s = s.replaceAll("\\[", "");
        s = s.replaceAll("]", "");
        var ss = s.split(", ");
        byte[] bb = new byte[ss.length];
        for (int i = 0; i < bb.length; i++) {
            bb[i] = Byte.parseByte(ss[i].trim());
        }
        return new DataInOutStatic(bb);
    }
}
