package io.aether.utils.dataio;


import io.aether.utils.HexUtils;

public interface DataOut {
    /**
     * Writes a byte array to this output stream and returns the number of bytes actually written.
     * @param b The byte array to write from.
     */
    default void write(byte[] b) {
        var r = write(b, 0, b.length);
        assert r == b.length;
    }

    /**
     * Writes an integer array to this output stream and returns the number of integers actually written.
     * @param b The integer array to write from.
     */
    default void write(int[] b) {
        var r = write(b, 0, b.length);
        assert r == b.length;
    }

    /**
     * Writes a byte array to this output stream and returns the number of bytes actually written.
     * @param b The byte array to write from.
     * @param off The starting position in the array to read from.
     * @param len The maximum number of bytes to write.
     */
    default int write(byte[] b, int off, int len) {
        for (int i = off; i < off + len; i++) {
            if (!isWritable()) return i - off;
            writeByte(b[i]);
        }
        return len;
    }

    /**
     * Writes an integer array to this output stream and returns the number of integers actually written.
     * @param b The integer array to write from.
     * @param off The starting position in the array to read from.
     * @param len The maximum number of integers to write.
     */
    default int write(int[] b, int off, int len) {
        for (int i = off; i < off + len; i++) {
            if (getSizeForWrite() < 4) return i - off;
            writeInt(b[i]);
        }
        return len;
    }

    /**
     * Clears this output stream.
     */
    default void clear() {
    }

    /**
     * Writes a boolean value to this output stream.
     * @param v The boolean value to write.
     */
    default void writeBoolean(boolean v) {
        writeByte(v ? 1 : 0);
    }

    /**
     * Writes a byte value to this output stream.
     * @param v The byte value to write.
     */
    void writeByte(int v);

    default void writeShort(int v) {
        writeShort((short)v);
    }
    /**
     * Writes a short value to this output stream.
     * @param v The short value to write.
     */
    default void writeShort(short v) {
        writeByte((byte) v);
        writeByte((byte) (v >>> 8));
    }

    /**
     * Writes a character value to this output stream.
     * @param v The character value to write.
     */
    default void writeChar(char v) {
        writeByte((byte) v);
    }

    /**
     * Writes an integer value to this output stream.
     * @param v The integer value to write.
     */
    default void writeInt(int v) {
        writeShort((short) v);
        writeShort((short) (v >>> 16));
    }

    /**
     * Writes a long value to this output stream.
     * @param v The long value to write.
     */
    default void writeLong(long v) {
        writeInt((int) v);
        writeInt((int) (v >>> 32));
    }

    /**
     * Writes a float value to this output stream.
     * @param v The float value to write.
     */
    default void writeFloat(float v) {
        writeInt(Float.floatToIntBits(v));
    }

    /**
     * Writes a double value to this output stream.
     * @param v The double value to write.
     */
    default void writeDouble(double v) {
        writeLong(Double.doubleToLongBits(v));
    }

    /**
     * Returns whether this output stream is writable or not.
     */
    default boolean isWritable() {
        return true;
    }

    /**
     * Returns the size of this output stream for writing, in bytes.
     */
    default int getSizeForWrite() {
        return Integer.MAX_VALUE;
    }

    /**
     * Writes a string to this output stream as hexadecimal bytes.
     * @param hex The string to write as hexadecimal bytes.
     */
    default void writeHexBytes(String hex) {
        write(HexUtils.hexToBytes(hex));
    }

    /**
     * Writes a data input stream to this output stream.
     * @param data The data input stream to write.
     */
    default void write(DataInOut data) {
        var r = write(data.data, data.readPos, data.writePos);
        assert r == data.getSizeForRead();
        data.readPos = 0;
        data.writePos = 0;
    }

    /**
     * Writes a static data input stream to this output stream.
     * @param data The static data input stream to write.
     */
    default void write(DataInOutStatic data) {
        var r = write(data.data, data.readPos, data.writePos);
        assert r == data.getSizeForRead();
        data.readPos = 0;
        data.writePos = 0;
    }

    /**
     * Writes a data input stream to this output stream.
     * @param data The data input stream to write.
     */
    default void write(DataIn data) {
        write(data.toArray());
    }
}