package io.aether.utils.flow;

import io.aether.utils.RU;
import io.aether.utils.interfaces.*;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.longs.*;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.NoSuchElementException;

public interface FlowLong extends LongIterator, LongIterable {
	static final long[] EMPTY_AR = new long[0];
	FlowLong EMPTY = new FlowCompletedLong() {
		@Override
		public boolean hasNext() {
			return false;
		}
		@Override
		public long nextLong() {
			return 0;
		}
		@Override
		public int count() {
			return 0;
		}
		@Override
		public long @NotNull [] toArray() {
			return EMPTY_AR;
		}
		@Override
		public @NotNull LongList toList() {
			return new LongArrayList();
		}
		@Override
		public @NotNull <K, V> Map<K, V> toMap(AFunctionL2O<K> keyFactory, AFunctionL2O<V> valFactory) {
			return new Object2ObjectOpenHashMap<>();
		}
	};

	default Flow<Long> box() {
		var self = this;
		return new Flow<>() {
			@Override
			public boolean hasNext() {
				return self.hasNext();
			}

			@Override
			public Long next() {
				return self.nextLong();
			}
		};
	}

    @NotNull
	static FlowLong of(@NotNull FlowLong stream) {
		return stream;
	}
	@NotNull
	static FlowLong of(long... array) {
		if (array.length == 0) return of();
		return new FlowCompletedLong() {
			int pos;
			@Override
			public boolean hasNext() {
				return pos < array.length;
			}
			@Override
			public long nextLong() {
				return array[pos++];
			}
			@Override
			public long @NotNull [] toArray() {
				return array;
			}
			@Override
			public void foreach(AConsumer<Long> c) {
				for (var v : array) {
					c.accept(v);
				}
			}
			@Override
			public int count() {
				return array.length;
			}
			@Override
			public @NotNull LongList toList() {
				return new LongArrayList(array);
			}
			@Override
			public @NotNull ASupplierLong toSupplier() {
				return new ASupplierLong() {
					int pos;
					@Override
					public long get2() {
						if (pos == array.length) throw new NoSuchElementException();
						return array[pos++];
					}
				};
			}
		};
	}
	@NotNull
	static FlowLong of(@NotNull LongIterable iterable) {
		if (iterable instanceof LongSet) return of((LongSet)iterable);
		if (iterable instanceof LongList ) return of((LongList)iterable);
		return of(iterable.iterator());
	}
	@NotNull
	static FlowLong of(@NotNull LongSet collection) {
		if (collection.isEmpty()) return of();
		var it = collection.iterator();
		return new FlowCompletedLong() {
			@Override
			public int count() {
				return collection.size();
			}
			@Override
			public boolean hasNext() {
				return it.hasNext();
			}
			@Override
			public long nextLong() {
				return it.nextLong();
			}
			@Override
			public @NotNull LongSet toSet() {
				return collection;
			}
		};
	}
	@NotNull
	static FlowLong of(@NotNull LongList collection) {
		if (collection.isEmpty()) return of();
		var it = collection.iterator();
		return new FlowCompletedLong() {
			@Override
			public int count() {
				return collection.size();
			}
			@Override
			public boolean hasNext() {
				return it.hasNext();
			}
			@Override
			public long nextLong() {
				return it.nextLong();
			}
			@Override
			public @NotNull LongList toList() {
				return collection;
			}
		};
	}
	@NotNull
	static FlowLong of(@NotNull LongIterator iterator) {
		if (!iterator.hasNext()) return of();
		return new FlowLong() {
			@Override
			public boolean hasNext() {
				return iterator.hasNext();
			}
			@Override
			public long nextLong() {
				return iterator.nextLong();
			}
		};
	}
	static FlowLong of() {
		return EMPTY;
	}
	static void quickSort(long[] source, int leftBorder, int rightBorder, AComparatorLong comparator) {
		int leftMarker = leftBorder;
		int rightMarker = rightBorder;
		long pivot = source[(leftMarker + rightMarker) / 2];
		do {
			while (comparator.compare(source[leftMarker], pivot) < 0) {
				leftMarker++;
			}
			while (comparator.compare(source[rightMarker], pivot) > 0) {
				rightMarker--;
			}
			if (leftMarker <= rightMarker) {
				if (leftMarker < rightMarker) {
					long tmp = source[leftMarker];
					source[leftMarker] = source[rightMarker];
					source[rightMarker] = tmp;
				}
				leftMarker++;
				rightMarker--;
			}
		} while (leftMarker <= rightMarker);
		if (leftMarker < rightBorder) {
			quickSort(source, leftMarker, rightBorder, comparator);
		}
		if (leftBorder < rightMarker) {
			quickSort(source, leftBorder, rightMarker, comparator);
		}
	}
	@NotNull
	@Override
	default LongIterator iterator() {
		return this;
	}
	default boolean anyMatch(APredicateLong p) {
		while (hasNext()) {
			if (p.test(nextLong())) return true;
		}
		return false;
	}
	default boolean noneMatch(APredicateLong p) {
		while (hasNext()) {
			if (p.test(nextLong())) return false;
		}
		return true;
	}
	default FlowLong add(long value) {
		final var oit = this;
		return new FlowLong() {
			private boolean index;
			@Override
			public boolean hasNext() {
				return !index || oit.hasNext();
			}
			@Override
			public long nextLong() {
				if (!index) {
					index = true;
					return value;
				}
				return oit.nextLong();
			}
		};
	}
	default boolean noneMatchValue(long p) {
		while (hasNext()) {
			if (p == nextLong()) return false;
		}
		return true;
	}
	default boolean anyMatchValue(long p) {
		while (hasNext()) {
			if (p == nextLong()) return true;
		}
		return false;
	}
	default boolean allMatch(APredicateLong p) {
		while (hasNext()) {
			if (!p.test(nextLong())) return false;
		}
		return true;
	}
	@NotNull
	default <E> Flow<E> mapToObj(@NotNull AFunctionL2O<E> f) {
		final var self = this;
		return new Flow<>() {
			@Override
			public boolean hasNext() {
				return self.hasNext();
			}
			@Override
			public E next() {
				return f.apply(self.nextLong());
			}
		};
	}
	@NotNull
	default FlowLong map(@NotNull AFunctionL2L f) {
		final var self = this;
		return new FlowLong() {
			@Override
			public boolean hasNext() {
				return self.hasNext();
			}
			@Override
			public long nextLong() {
				return f.apply(self.nextLong());
			}
		};
	}
	@NotNull
	default FlowLong apply(AConsumerLong c) {
		final var self = this;
		return new FlowLong() {
			@Override
			public boolean hasNext() {
				return self.hasNext();
			}
			@Override
			public long nextLong() {
				var v = self.nextLong();
				c.accept(v);
				return v;
			}
		};
	}
	default FlowLong addAllEls(long... values) {
		final var oit = this;
		return new FlowLong() {
			private int index;
			@Override
			public boolean hasNext() {
				return index < values.length || oit.hasNext();
			}
			@Override
			public long nextLong() {
				if (index < values.length) {
					return values[index++];
				}
				return oit.nextLong();
			}
		};
	}
	default FlowLong addAll(FlowLong values) {
		return addAll((LongIterator) values);
	}
	default FlowLong addAll(LongIterator values) {
		if (!values.hasNext()) return this;
		final var oit = this;
		return new FlowLong() {
			@Override
			public boolean hasNext() {
				return values.hasNext() || oit.hasNext();
			}
			@Override
			public long nextLong() {
				if (values.hasNext()) {
					return values.nextLong();
				}
				return oit.nextLong();
			}
		};
	}
	default FlowLong addAll(LongIterable values) {
		return addAll(values.iterator());
	}
	@NotNull
	default FlowLong flatMap(@NotNull AFunctionL2IterableLong f) {
		return flatMap((e, c) -> {
			var ii = f.apply(e);
			if (ii != null) {
				for (var v : ii) {
					c.accept(v);
				}
			}
		});
	}
	@NotNull
	default FlowLong flatMap(@NotNull AFunctionL2Array f) {
		return flatMap((e, c) -> {
			var ii = f.apply(e);
			if (ii != null) {
				for (var v : ii) {
					c.accept(v);
				}
			}
		});
	}
	default void foreach(AConsumer<Long> c) {
		while (hasNext()) {
			c.accept(nextLong());
		}
	}
	@NotNull
	default FlowLong flatMap(@NotNull AFunctionL2StreamLong f) {
		var self = this;
		return new FlowLong() {
			FlowLong cur;
			@Override
			public boolean hasNext() {
				while (cur == null || !cur.hasNext()) {
					if (cur != null) {
						cur = null;
					}
					if (!self.hasNext()) return false;
					cur = f.apply(self.nextLong());
				}
				return true;
			}
			@Override
			public long nextLong() {
				return cur.nextLong();
			}
		};
	}
	@NotNull
	default FlowLong flatMap(@NotNull ABiConsumerL2O<AConsumerLong> f) {
		final var oit = this;
		LongArrayList list = new LongArrayList();
		final AConsumerLong cc = list::add;
		return new FlowLong() {
			int pos;
			@Override
			public boolean hasNext() {
				while (list.isEmpty()) {
					if (!oit.hasNext()) return false;
					f.accept(oit.nextLong(), cc);
				}
				return true;
			}
			@Override
			public long nextLong() {
				var res = list.getLong(pos++);
				if (pos == list.size()) {
					pos = 0;
					list.clear();
				}
				return res;
			}
		};
	}
	default long @NotNull [] toArray() {
		long[] res = new long[10];
		int i = 0;
		while (hasNext()) {
			if (i == res.length) {
				res = Arrays.copyOf(res, (int) (res.length * 1.5));
			}
			res[i++] = nextLong();
		}
		return Arrays.copyOf(res, i);
	}
	default <E extends Collection<Long>> E toCollection(@NotNull E collection) {
		while (hasNext()) {
			collection.add(nextLong());
		}
		return collection;
	}
	default <E extends LongCollection> E toCollection(@NotNull E collection) {
		while (hasNext()) {
			collection.add(nextLong());
		}
		return collection;
	}
	@NotNull
	default FlowLong filter(@Nullable APredicateLong predicate) {
		if (predicate == null) return this;
		var oit = this;
		return new FlowLong() {
			long last;
			boolean hasNext;
			@Override
			public boolean hasNext() {
				if (hasNext) return true;
				while (oit.hasNext()) {
					last = oit.nextLong();
					if (predicate.test(last)) {
						hasNext = true;
						return true;
					}
				}
				return false;
			}
			@Override
			public long nextLong() {
				assert hasNext;
				hasNext = false;
				return last;
			}
		};
	}
	@NotNull
	default FlowLong filterNot(@Nullable APredicateLong predicate) {
		if (predicate == null) return this;
		var oit = this;
		return new FlowLong() {
			long last;
			boolean hasNext;
			@Override
			public boolean hasNext() {
				if (hasNext) return true;
				while (oit.hasNext()) {
					last = oit.nextLong();
					if (!predicate.test(last)) {
						hasNext = true;
						return true;
					}
				}
				return false;
			}
			@Override
			public long nextLong() {
				assert hasNext;
				hasNext = false;
				return last;
			}
		};
	}
	@NotNull
	default FlowLong ifEmpty(ARunnable task) {
		if (task == null) return this;
		var oit = this;
		return new FlowLong() {
			boolean first = true;
			@Override
			public boolean hasNext() {
				var res = oit.hasNext();
				if (!res && first) {
					task.run();
				}
				first = false;
				return res;
			}
			@Override
			public long nextLong() {
				return oit.nextLong();
			}
		};
	}
	@NotNull
	default FlowLong ifEmpty(Exception error) {
		return ifEmpty(() -> {
			throw error;
		});
	}
	@NotNull
	default FlowLong ignoreError(Class<? extends Exception> ee) {
		if (ee == null) return this;
		var oit = this;
		return new FlowLong() {
			long last;
			boolean hasNext;
			@Override
			public boolean hasNext() {
				if (hasNext) return true;
				while (oit.hasNext()) {
					try {
						last = oit.nextLong();
						hasNext = true;
						return true;
					} catch (Exception ex) {
						if (!ee.isInstance(ex)) {
							RU.error(ex);
						}
					}
				}
				return false;
			}
			@Override
			public long nextLong() {
				assert hasNext;
				hasNext = false;
				return last;
			}
		};
	}
	@NotNull
	default FlowLong sort(@NotNull AComparatorLong comparator) {
		long[] arr = toArray();
		quickSort(arr, 0, arr.length, comparator);
		return of(arr);
	}
	@NotNull
	default LongIterable toIterable() {
		return () -> this;
	}
	@NotNull
	default LongList toList() {
		return new LongArrayList(toArray());
	}
	@NotNull
	default LongSet toSet() {
		return new LongOpenHashSet(toArray());
	}
	default void to(AConsumerLong consumer) {
		while (hasNext()) {
			consumer.accept(nextLong());
		}
	}
	default <E> E streamTo(AFunction<FlowLong, E> consumer) {
		return consumer.apply(this);
	}
	@NotNull
	default ASupplierLong toSupplier() {
		return () -> {
			if (hasNext()) {
				return nextLong();
			}
			throw new NoSuchElementException();
		};
	}
	@NotNull
	default <K, V> Map<K, V> toMap(AFunctionL2O<K> keyFactory, AFunctionL2O<V> valFactory) {
		Object2ObjectOpenHashMap<K, V> res = new Object2ObjectOpenHashMap<>();
		while (hasNext()) {
			var val = nextLong();
			res.put(keyFactory.apply(val), valFactory.apply(val));
		}
		return res;
	}
	@NotNull
	default <V> Int2ObjectMap<V> toMapI2O(AFunctionL2I keyFactory, AFunctionL2O<V> valFactory) {
		Int2ObjectOpenHashMap<V> res = new Int2ObjectOpenHashMap<>();
		while (hasNext()) {
			var val = nextLong();
			res.put(keyFactory.apply(val), valFactory.apply(val));
		}
		return res;
	}
	@NotNull
	default <K> Object2IntMap<K> toMapO2I(AFunctionL2O<K> keyFactory, AFunctionL2I valFactory) {
		Object2IntOpenHashMap<K> res = new Object2IntOpenHashMap<>();
		while (hasNext()) {
			var val = nextLong();
			res.put(keyFactory.apply(val), valFactory.apply(val));
		}
		return res;
	}
	@NotNull
	default <V> Long2ObjectMap<V> toMapL2O(AFunctionL2L keyFactory, AFunctionL2O<V> valFactory) {
		Long2ObjectOpenHashMap<V> res = new Long2ObjectOpenHashMap<>();
		while (hasNext()) {
			var val = nextLong();
			res.put(keyFactory.apply(val), valFactory.apply(val));
		}
		return res;
	}
	default String join(String delimer) {
		StringBuilder sb = new StringBuilder();
		boolean first = true;
		while (hasNext()) {
			if (first){
				first=false;
			}else{
				sb.append(delimer);
			}
			sb.append(nextLong());
		}
		return sb.toString();
	}
	default FlowLong distinct() {
		return filter(new APredicateLong() {
			private final LongSet old = new LongOpenHashSet();
			public boolean test2(long value) {
				return old.add(value);
			}
		});
	}
	default long random() {
		var ar = toArray();
		return ar[Flow.RANDOM.nextInt(ar.length)];
	}
	default String join(AFunctionL2O<Object> preparer, String delimer) {
		StringBuilder sb = new StringBuilder();
		boolean first =true;
		while (hasNext()) {
			if (first){
				first=false;
			}else{
				sb.append(delimer);
			}
			sb.append(preparer.apply(nextLong()));
		}
		return sb.toString();
	}
	default long getFirstOr(long def) {
		if (hasNext()) return nextLong();
		return def;
	}
	default long getFirst() {
		if (hasNext()) return nextLong();
		throw new NoSuchElementException();
	}
	default long getFirst(APredicateLong p) {
		return filter(p).getFirst();
	}
	default String join() {
		StringBuilder sb = new StringBuilder();
		while (hasNext()) {
			sb.append(nextLong());
		}
		return sb.toString();
	}
	default String join(String delim, String prefix, String postfix) {
		StringBuilder sb = new StringBuilder(prefix);
		boolean first = true;
		while (hasNext()) {
			if (first){
				first=false;
			}else{
				sb.append(delim);
			}
			sb.append(nextLong());
		}
		sb.append(postfix);
		return sb.toString();
	}
	default long min(AComparatorLong comparator) {
		long min = Long.MAX_VALUE;
		while (hasNext()) {
			var v = nextLong();
			if (comparator.compare(min, v) > 0) {
				min = v;
			}
		}
		return min;
	}
	default long min() {
		long min = Long.MAX_VALUE;
		while (hasNext()) {
			var v = nextLong();
			if (min > v) {
				min = v;
			}
		}
		return min;
	}
	default Long max(AComparatorLong comparator) {
		long max = Long.MIN_VALUE;
		while (hasNext()) {
			var v = nextLong();
			if (comparator.compare(max, v) < 0) {
				max = v;
			}
		}
		return max;
	}
	default Long max() {
		long max = Long.MIN_VALUE;
		while (hasNext()) {
			var v = nextLong();
			if (max < v) {
				max = v;
			}
		}
		return max;
	}
}
