package io.aether.utils.futures;

import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.interfaces.ARunnable;
import io.aether.utils.interfaces.ASupplier;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Iterator;
import java.util.concurrent.*;

/**
 * An asynchronous operation that completes with no return value (void).
 * This interface extends the base asynchronous functionality and provides
 * static factory methods (implemented via proxy to AFutureImpl).
 */
public interface AFuture extends AFutureBase<AFuture> {

    /**
     * Marks the future as successfully completed.
     */
    void done();

    /**
     * Attempts to mark the future as successfully completed.
     * @return true if the status was updated, false otherwise.
     */
    boolean tryDone();
    default boolean waitDoneSeconds(int seconds){
        try {
            toCompletableFuture().get(seconds, TimeUnit.SECONDS);
            return true;
        } catch (Exception ignore) {
        }
        return false;
    }
    /**
     * Chains this future with another, creating a new future that completes
     * when both this and the given future complete successfully.
     * @param f The other future to chain with.
     * @return A new AFuture that completes when both are done.
     */
    AFuture and(AFuture f);

    /**
     * Creates a new AFuture that runs the task upon successful completion of this future.
     * @param t The runnable task to execute.
     * @return A new AFuture that completes after the task is run.
     */
    AFuture apply(ARunnable t);

    /**
     * Maps the successful completion of this void future to a result-carrying future.
     * @param t The supplier to be executed upon completion, providing the result value.
     * @param <T> The type of the result.
     * @return An ARFuture with the result of the supplier.
     */
    <T> ARFuture<T> mapRFuture(ASupplier<T> t);

    // ==================== STATIC FACTORY METHODS (Proxy to AFutureImpl) ====================

    /**
     * Creates a new AFuture instance that is already successfully completed.
     * @return An immediately completed AFuture.
     */
    static AFuture of() {
        return AFutureImpl.DONED;
    }
    static AFuture make() {
        return new AFutureImpl();
    }
    AFuture to(@NotNull AFuture f) ;

    /**
     * Creates a cancelled AFuture instance.
     * @return An immediately cancelled AFuture.
     */
    static AFuture canceled() {
        return AFutureImpl.CANCELED;
    }

    /**
     * Creates an AFuture instance that is already completed with the given Throwable error.
     * @param e The exception to complete the future with.
     * @return An AFuture completed with an error.
     */
    static AFuture of(Throwable e) {
        return new AFutureImpl(e);
    }

    /**
     * Executes a task using the provided executor and completes the returned future when the task is done.
     * @param executor The executor to run the task on.
     * @param task The task to execute, which can call done()/error() on the future explicitly.
     * @return A new AFuture that completes based on the task's execution.
     */
    static AFuture run(Executor executor, AConsumer<AFuture> task) {
        return AFutureImpl.run(executor, task);
    }

    /**
     * Executes a simple runnable task using the provided executor.
     * @param executor The executor to run the task on.
     * @param task The runnable task to execute.
     * @return A new AFuture that completes when the task finishes.
     */
    static AFuture run(Executor executor, ARunnable task) {
        return AFutureImpl.run(executor, task);
    }

    /**
     * Returns an AFuture that completes successfully when any future in the collection completes.
     * Cancels all other futures upon completion.
     * @param ff The collection of futures.
     * @return A future that completes when the fastest one is done (and cancels others).
     */
    static AFuture anyAndCancel(Collection<AFuture> ff) {
        return AFutureImpl.any(ff, AFuture::cancel);
    }

    /**
     * Returns an AFuture that completes successfully when any future in the collection completes.
     * @param ff The collection of futures.
     * @return A future that completes when the fastest one is done.
     */
    static AFuture any(Collection<AFuture> ff) {
        return AFutureImpl.any(ff, (a) -> {});
    }

    /**
     * Returns an AFuture that completes successfully when all futures in the array complete successfully.
     * Fails or cancels immediately if any individual future fails or cancels.
     * @param ff The array of futures.
     * @return A future that completes when all array futures are done.
     */
    static AFuture all(AFuture... ff) {
        return AFutureImpl.all(ff);
    }

    /**
     * Returns an AFuture that completes successfully when all futures in the iterator complete successfully.
     * Fails or cancels immediately if any individual future fails or cancels.
     * @param ff The iterator of futures.
     * @return A future that completes when all iterator futures are done.
     */
    static AFuture all(Iterator<AFuture> ff) {
        return AFutureImpl.all(ff);
    }

    /**
     * Returns an AFuture that completes successfully when all futures in the collection complete successfully.
     * Fails or cancels immediately if any individual future fails or cancels.
     * @param ff The collection of futures.
     * @return A future that completes when all collection futures are done.
     */
    static AFuture all(Collection<AFuture> ff) {
        return AFutureImpl.all(ff);
    }

    static AFuture completed() {
        return AFutureImpl.DONED;
    }

    static AFuture doThrow(Throwable e) {
        return new AFutureImpl(e);
    }
    default CompletableFuture<?> toCompletableFuture(){
        CompletableFuture<Object> res = new CompletableFuture<>();

        to(()->{
            Object rawResult = getNowRaw();
            res.complete(rawResult == AFutureBaseImpl.NULL ? null : rawResult);
        });

        onError(res::completeExceptionally);

        onCancel(f -> {
            res.completeExceptionally(new java.util.concurrent.CancellationException("AFuture was canceled"));
        });

        return res;
    }
}