package io.aether.utils.futures;

import io.aether.logger.LNode;
import io.aether.logger.Log;
import io.aether.utils.CTypeI;
import io.aether.utils.RU;
import io.aether.utils.TimeoutChecker;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.interfaces.ARunnable;
import org.jetbrains.annotations.NotNull;

import java.lang.invoke.VarHandle;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeoutException;

/**
 * Base implementation class for all futures (AFuture and ARFuture).
 * Contains core logic for state management, listeners, waiting, and timeouts.
 * This class is package-private and maintains the original Self generic type
 * for fluent API chaining inherited from AFutureAbstract.
 *
 * @param <Self> The concrete future type (AFuture or ARFuture).
 */
abstract class AFutureBaseImpl<Self extends AFutureBase<Self>> implements AFutureBase<Self> {
    static final Object NULL = new Object();
    static final Object CANCEL_VALUE = new Object();
    static final VarHandle OFFSET_RESULT = CTypeI.of(AFutureBaseImpl.class).getFieldVarHandle("result");
    static final VarHandle TASKS = CTypeI.of(AFutureBaseImpl.class).getFieldVarHandle("tasks");
    static final Task<?> NOP_TASK = new Task<>(d -> {
    }, null, null);
    private static final boolean DISABLE_TIMEOUT = false;

    volatile Object result;
    volatile Task<Self> tasks;

    @Override
    public Self onCancel(ARunnable l) {
        return onCancel(f -> l.run());
    }

    @Override
    public Self onCancel(AConsumer<Self> l) {
        addListener(f -> {
            if (f.isCanceled()) {
                l.accept(f);
            }
        });
        return RU.cast(this);
    }

    @Override
    @SuppressWarnings("unchecked")
    public Self onError(AConsumer<Throwable> l) {
        addListener(f -> {
            if (f.isError()) {
                l.accept(f.getError());
            }
        });
        return (Self) this;
    }

    @Override
    public boolean isFinalStatus() {
        return result != null;
    }

    public void waitSuccessful() {
        if (isFinalStatus()) return;
        var self = this;
        if (!addListener(c -> {
            synchronized (self) {
                self.notifyAll();
            }
        })) return;
        try {
            synchronized (self) {
                while (!isFinalStatus()) {
                    self.wait();
                }
            }
        } catch (InterruptedException e) {
            RU.error(e);
        }
    }

    public boolean waitSuccessful(long timeout) {
        if (isFinalStatus()) return true;
        var self = this;
        if (!addListener(c -> {
            synchronized (self) {
                self.notifyAll();
            }
        })) return true;
        try {
            long begin = RU.time();
            synchronized (self) {
                while (!isFinalStatus()) {
                    var t2 = timeout - (RU.time() - begin);
                    if (t2 <= 0) return false;
                    self.wait(t2);
                }
                return true;
            }
        } catch (InterruptedException e) {
            return RU.error(e);
        }
    }

    @Override
    public boolean isError() {
        return result instanceof Throwable;
    }

    @Override
    @SuppressWarnings("unchecked")
    public Self error(Throwable e) {
        updateStatus(e);
        return (Self) this;
    }

    @Override
    public Throwable getError() {
        return (Throwable) result;
    }

    @Override
    public void setError(Throwable e) {
        updateStatus(e);
    }

    /**
     * Attempts to set the final result status using CompareAndSet.
     *
     * @param result The result object (or Throwable for error, CANCEL_VALUE for cancel).
     * @return true if the status was successfully updated.
     */
    public boolean updateStatus(Object result) {
        if (result == null) result = NULL;
        if (OFFSET_RESULT.compareAndSet(this, null, result)) {
            pingListeners();
            return true;
        }
        return false;
    }

    @Override
    public boolean isNotDone() {
        return !isDone();
    }

    @Override
    public boolean isDone() {
        return isFinalStatus() && !isError() && !isCanceled();
    }

    @Override
    public boolean isCanceled() {
        return result == CANCEL_VALUE;
    }

    @Override
    public void cancel() {
        updateStatus(CANCEL_VALUE);
    }

    @Override
    @SuppressWarnings("unchecked")
    public Self timeoutError(int seconds, String text) {
        if (DISABLE_TIMEOUT) return (Self) this;
        if (isFinalStatus()) return (Self) this;
        var tc = TimeoutChecker.error(seconds, () -> {
            var er = new TimeoutException(text);
            setError(er);
            return text;
        });
        addListener(c -> tc.done());
        return (Self) this;
    }

    @Override
    @SuppressWarnings("unchecked")
    public Self timeout(int seconds, ARunnable task) {
        if (DISABLE_TIMEOUT) return (Self) this;
        if (isFinalStatus()) return (Self) this;
        var tc = TimeoutChecker.error(seconds, () -> {
            task.run();
            return "";
        });
        addListener(c -> tc.done());
        return (Self) this;
    }

    @Override
    public Self timeoutMs(long ms, ARunnable task) {
        if (DISABLE_TIMEOUT) return RU.cast(this);
        if (isFinalStatus()) return RU.cast(this);
        var tc = TimeoutChecker.errorMs(ms, () -> {
            task.run();
            return "";
        });
        addListener(c -> tc.done());
        return RU.cast(this);
    }

    @Override
    public Self timeoutMs(long ms, AConsumer<Self> task) {
        if (DISABLE_TIMEOUT) return RU.cast(this);
        if (isFinalStatus()) return RU.cast(this);
        var tc = TimeoutChecker.errorMs(ms, () -> {
            task.accept(RU.cast(this));
            return "";
        });
        addListener(c -> tc.done());
        return RU.cast(this);
    }

    @Override
    @SuppressWarnings("unchecked")
    public boolean addListener(AConsumer<Self> l) {
        assert l != null;
        Task<Self> old;
        LNode logContext = null;
        do {
            old = tasks;
            if (old == NOP_TASK) {
                l.accept((Self) this);
                return false;
            }
            if (logContext == null) {
                logContext = Log.createContext();
            }
        } while (!TASKS.compareAndSet(this, old, new Task<>(l, old, logContext)));
        return true;
    }

    @SuppressWarnings("unchecked")
    private void pingListeners() {
        var self = (Self) this;
        while (true) {
            var t = tasks;
            if (t == NOP_TASK) return;
            if (TASKS.compareAndSet(this, t, NOP_TASK)) {
                while (t != null) {
                    Log.push(t.logContext);
                    try {
                        t.task.accept(self);
                    } catch (Exception e) {
                        Log.error(e);
                    } finally {
                        Log.pop(t.logContext);
                    }
                    t = t.next;
                }
                return;
            }
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    public Self to(Executor executor, ARunnable t) {
        if (isDone()) {
            executor.execute(t);
            return (Self) this;
        }
        if (isFinalStatus()) return (Self) this;
        addListener(s -> {
            if (s.isDone()) {
                executor.execute(t);
            }
        });
        return (Self) this;
    }

    @Override
    public Self to(ARunnable t) {
        if (isDone()) {
            t.run();
        } else {
            addListener(s -> {
                if (s.isDone()) {
                    t.run();
                }
            });
        }
        return RU.cast(this);
    }

    @Override
    public Object getNowRaw() {
        return result;
    }

    @Override
    public boolean tryError(@NotNull Throwable error) {
        Objects.requireNonNull(error);
        return updateStatus(error);
    }

    @Override
    public Self tryCancel() {
        updateStatus(CANCEL_VALUE);
        return RU.cast(this);
    }

    protected final static class Task<Self> {
        public final AConsumer<Self> task;
        public final LNode logContext;
        public final Task<Self> next;

        public Task(AConsumer<Self> task, Task<Self> next, LNode logContext) {
            this.task = task;
            this.next = next;
            this.logContext = logContext;

        }
    }
}