package io.aether.utils.futures;

import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.*;
import io.aether.utils.tuples.Tuple2;
import io.aether.utils.tuples.Tuple3;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.Executor;

/**
 * An asynchronous operation that completes with a result value of type T.
 * This interface extends the base asynchronous functionality and provides
 * static factory methods (implemented via proxy to ARFutureImpl).
 *
 * @param <T> The type of the result value.
 */
public interface ARFuture<T> extends AFutureBase<ARFuture<T>> {

    ARFuture<Boolean> TRUE = ARFuture.of(true);
    ARFuture<Boolean> FALSE = ARFuture.of(false);

    /**
     * Casts this ARFuture to its flag-carrying subclass.
     *
     * @return The ARFutureWithFlag instance.
     */
    ARFutureWithFlag<T> toWithFlag();

    /**
     * Completes the future with the given value.
     *
     * @param value The result value.
     * @throws IllegalStateException if the future is already completed.
     */
    void done(T value);

    /**
     * Attempts to complete the future with the given value.
     *
     * @param value The result value.
     * @return true if the status was updated, false otherwise.
     */
    boolean tryDone(T value);

    /**
     * Gets the result of the future, waiting if necessary, and throws an exception on error/cancel.
     *
     * @return The result value (T).
     */
    T get();

    /**
     * Gets the result of the future with a timeout, waiting if necessary.
     *
     * @param timout The timeout in milliseconds.
     * @return The result value (T).
     */
    T get(long timout);

    /**
     * Gets the result value if the future is already done, otherwise returns null.
     *
     * @return The result T or null.
     */
    T getNow();

    /**
     * Gets the result value if the future is already done, otherwise returns the provided default value.
     *
     * @param elseValue The default value to return if not done.
     * @return The result T or elseValue.
     */
    T getNowElse(T elseValue);

    /**
     * Registers a callback to be executed upon successful completion.
     *
     * @param onDone The consumer that accepts the result value.
     * @return This future instance.
     */
    ARFuture<T> to(AConsumer<T> onDone);

    /**
     * Registers a callback for success and another for error.
     *
     * @param onDone  The consumer that accepts the result value.
     * @param onError The consumer that accepts the throwable error.
     * @return This future instance.
     */
    ARFuture<T> to(AConsumer<T> onDone, AConsumer<Throwable> onError);

    /**
     * Registers a success callback with a timeout action.
     *
     * @param task      The consumer that accepts the result value.
     * @param timeout   The timeout in seconds.
     * @param onTimeout The runnable to execute on timeout.
     * @return This future instance.
     */
    ARFuture<T> to(@NotNull AConsumer<T> task, int timeout, ARunnable onTimeout);

    /**
     * Transfers the final status (done/error/cancel) to another future.
     *
     * @param f The target future.
     * @return This future instance.
     */
    ARFuture<T> to(@NotNull ARFuture<T> f);

    ARFuture<T> to(@NotNull AFuture f);

    /**
     * Converts this ARFuture into a void AFuture.
     *
     * @return The resulting AFuture.
     */
    AFuture toFuture();

    /**
     * Chains this future with another, combining their results into a Tuple2.
     *
     * @param f    The other future.
     * @param <T2> The type of the other future's result.
     * @return A new ARFuture with a Tuple2 result.
     */
    <T2> ARFuture<Tuple2<T, T2>> and(ARFuture<T2> f);

    /**
     * Transforms the successful result into a different type using the provided function.
     *
     * @param f   The function to apply to the result.
     * @param <E> The new result type.
     * @return A new ARFuture with the transformed result.
     */
    @NotNull
    <E> ARFuture<E> map(@NotNull AFunction<T, E> f);

    /**
     * Transforms the successful result into a different type using a function wrapped for Log context.
     *
     * @param f   The function to apply to the result.
     * @param <E> The new result type.
     * @return A new ARFuture with the transformed result.
     */
    @NotNull
    <E> ARFuture<E> mapWL(@NotNull AFunction<T, E> f);

    /**
     * Transforms the successful result into another ARFuture.
     *
     * @param f   The function returning the new ARFuture.
     * @param <E> The new result type.
     * @return A new ARFuture that completes with the result of the future returned by f.
     */
    @NotNull
    <E> ARFuture<E> mapRFuture(@NotNull AFunction<T, ARFuture<E>> f);
    /**
     * Maps the result to a void AFuture and converts the result to an AFuture.
     *
     * @param f The consumer to apply to the result.
     * @return An AFuture representing the completion of the consumer's execution.
     */
    @NotNull
    AFuture mapToFuture(@NotNull AConsumer<T> f);

    /**
     * Transforms the successful result into a different type in a safe context (e.g., dedicated executor).
     *
     * @param f   The function to apply to the result.
     * @param <E> The new result type.
     * @return A new ARFuture with the transformed result.
     */
    <E> ARFuture<E> mapSafe(@NotNull AFunction<T, E> f);

    /**
     * Executes a runnable upon successful completion, preserving the original result.
     *
     * @param c The runnable to execute.
     * @return A new ARFuture with the same original result.
     */
    ARFuture<T> apply(ARunnable c);

    /**
     * Executes a consumer upon successful completion, preserving the original result.
     *
     * @param c The consumer to execute with the result.
     * @return A new ARFuture with the same original result.
     */
    ARFuture<T> apply(AConsumer<T> c);

    // ==================== STATIC FACTORY METHODS (Proxy to ARFutureImpl) ====================

    /**
     * Creates a new ARFuture instance that is already successfully completed with the given value.
     *
     * @param value The result value.
     * @param <T>   The type of the result.
     * @return An immediately completed ARFuture.
     */
    static <T> ARFuture<T> of(T value) {
        return ARFutureImpl.of(value);
    }

    /**
     * Creates a cancelled ARFuture instance.
     *
     * @param <T> The type of the result.
     * @return An immediately cancelled ARFuture.
     */
    static <T> ARFuture<T> canceled() {
        return ARFutureImpl.canceled();
    }

    /**
     * Creates an ARFuture instance that is already completed with the given Throwable error.
     *
     * @param throwable The exception to complete the future with.
     * @param <T>       The type of the result.
     * @return An ARFuture completed with an error.
     */
    static <T> ARFuture<T> ofThrow(Throwable throwable) {
        return ARFutureImpl.ofThrow(throwable);
    }

    /**
     * Executes a task using the provided executor and returns an ARFuture.
     *
     * @param executor The executor to run the task on.
     * @param task     The function returning the result value.
     * @param <T>      The type of the result.
     * @return A new ARFuture that completes with the result of the task.
     */
    static <T> ARFuture<T> run2(Executor executor, ACallable<ARFuture<T>> task) {
        return ARFutureImpl.run2(executor, task);
    }

    /**
     * Executes a callable task using the provided executor and returns an ARFuture.
     *
     * @param executor The executor to run the task on.
     * @param task     The callable task.
     * @param <T>      The type of the result.
     * @return A new ARFuture that completes with the result of the task.
     */
    static <T> ARFuture<T> run(Executor executor, ACallable<T> task) {
        return ARFutureImpl.run(executor, task);
    }

    /**
     * Returns an ARFuture that completes with the result of the fastest future in the collection,
     * and cancels all other futures.
     *
     * @param ff  The flow of futures.
     * @param <T> The type of the result.
     * @return The ARFuture with the fastest result.
     */
    static <T> ARFuture<T> anyAndCancel(Flow<ARFuture<T>> ff) {
        return ARFutureImpl.anyAndCancel(ff);
    }

    /**
     * Returns an ARFuture that completes with the result of the fastest future in the collection.
     *
     * @param ff  The collection of futures.
     * @param <T> The type of the result.
     * @return The ARFuture with the fastest result.
     */
    static <T> ARFuture<T> any(Collection<ARFuture<T>> ff) {
        return ARFutureImpl.any(ff);
    }

    /**
     * Returns an ARFuture that completes with an array of all results once all futures are completed.
     * Fails or cancels immediately if any individual future fails or cancels.
     *
     * @param elType The class type of the array elements.
     * @param ff     The flow of futures.
     * @param <T>    The type of the result.
     * @return The ARFuture with the array of results.
     */
    static <T> ARFuture<T[]> all(Class<T> elType, Flow<ARFuture<T>> ff) {
        return ARFutureImpl.all(elType, ff);
    }

    /**
     * Returns an ARFuture that completes with an array of all results once all futures are completed.
     *
     * @param elType The class type of the array elements.
     * @param ff     The array of futures.
     * @param <T>    The type of the result.
     * @return The ARFuture with the array of results.
     */
    @SafeVarargs
    static <T> ARFuture<T[]> all(Class<T> elType, ARFuture<T>... ff) {
        return ARFutureImpl.all(elType, ff);
    }

    /**
     * Returns an ARFuture that completes with a Tuple2 of results.
     *
     * @param f1   The first future.
     * @param f2   The second future.
     * @param <T1> The type of the first result.
     * @param <T2> The type of the second result.
     * @return The ARFuture with the Tuple2 of results.
     */
    static <T1, T2> ARFuture<Tuple2<T1, T2>> all(ARFuture<T1> f1, ARFuture<T2> f2) {
        return ARFutureImpl.all(f1, f2);
    }

    /**
     * Returns an ARFuture that completes with a Tuple3 of results.
     *
     * @param f1   The first future.
     * @param f2   The second future.
     * @param f3   The third future.
     * @param <T1> The type of the first result.
     * @param <T2> The type of the second result.
     * @param <T3> The type of the third result.
     * @return The ARFuture with the Tuple3 of results.
     */
    static <T1, T2, T3> ARFuture<Tuple3<T1, T2, T3>> all(ARFuture<T1> f1, ARFuture<T2> f2, ARFuture<T3> f3) {
        return ARFutureImpl.all(f1, f2, f3);
    }

    /**
     * Returns an ARFuture that completes with a List of all results once all futures are completed.
     *
     * @param list The list of futures.
     * @param <T>  The type of the result.
     * @return The ARFuture with the List of results.
     */
    static <T> ARFuture<List<T>> all(@NotNull List<ARFuture<T>> list) {
        return ARFutureImpl.all(list);
    }

    /**
     * Performs an action upon successful completion of two futures.
     *
     * @param f1   The first future.
     * @param f2   The second future.
     * @param task The action to perform with the results.
     * @param <T1> The type of the first result.
     * @param <T2> The type of the second result.
     * @return The ARFuture with the Tuple2 of results.
     */
    static <T1, T2> ARFuture<Tuple2<T1, T2>> all(ARFuture<T1> f1, ARFuture<T2> f2, ABiConsumer<T1, T2> task) {
        return ARFutureImpl.all(f1, f2, task);
    }

    /**
     * Performs an action upon successful completion of three futures.
     *
     * @param f1   The first future.
     * @param f2   The second future.
     * @param f3   The third future.
     * @param task The action to perform with the results.
     * @param <T1> The type of the first result.
     * @param <T2> The type of the second result.
     * @param <T3> The type of the third result.
     * @return The ARFuture with the Tuple3 of results.
     */
    static <T1, T2, T3> ARFuture<Tuple3<T1, T2, T3>> all(ARFuture<T1> f1, ARFuture<T2> f2, ARFuture<T3> f3, A3Consumer<T1, T2, T3> task) {
        return ARFutureImpl.all(f1, f2, f3, task);
    }

    /**
     * Maps the results of two futures to a new result type.
     *
     * @param f1   The first future.
     * @param f2   The second future.
     * @param task The function to map the results.
     * @param <T1> The type of the first result.
     * @param <T2> The type of the second result.
     * @param <R>  The new result type.
     * @return The ARFuture with the mapped result.
     */
    static <T1, T2, R> ARFuture<R> map(ARFuture<T1> f1, ARFuture<T2> f2, A2Function<T1, T2, R> task) {
        return ARFutureImpl.map(f1, f2, task);
    }

    /**
     * Maps the results of three futures to a new result type.
     *
     * @param f1   The first future.
     * @param f2   The second future.
     * @param f3   The third future.
     * @param task The function to map the results.
     * @param <T1> The type of the first result.
     * @param <T2> The type of the second result.
     * @param <T3> The type of the third result.
     * @param <R>  The new result type.
     * @return The ARFuture with the mapped result.
     */
    static <T1, T2, T3, R> ARFuture<R> map(
            ARFuture<T1> f1,
            ARFuture<T2> f2,
            ARFuture<T3> f3,
            A3Function<T1, T2, T3, R> task) {
        return ARFutureImpl.map(f1, f2, f3, task);
    }

    static <T> ARFuture<T> make() {
        return new ARFutureImpl<>();
    }

    static <T> ARFuture<T> doThrow(Throwable e) {
        var res = new ARFutureImpl<T>();
        res.setError(e);
        return res;
    }
}