package io.aether.utils.rcollections;

import io.aether.utils.futures.ARFutureWithFlag;
import io.aether.utils.interfaces.Destroyable;
import io.aether.utils.slots.EventConsumer;
import org.jetbrains.annotations.NotNull;

import java.util.Set;

/**
 * Interface for an asynchronous, throttling map designed to manage the lifecycle of external data requests.
 * It extends {@code RFMap<K, V>} (which is {@code RMap<K, ARFutureWithFlag<V>>}) and adds the necessary
 * methods for throttling and batching external requests, specifically supporting multi-sender request tracking.
 *
 * @param <K> The type of the key.
 * @param <V> The type of the value to be retrieved asynchronously.
 */
public interface BMap<K, V> extends RFMap<K, V>, Destroyable {

    /**
     * Retrieves the ARFutureWithFlag associated with the key. If the value is not present or the previous
     * request has timed out, this method must conditionally add the key to the new request pool
     * for later batching by a separate {@code flush} mechanism.
     *
     * @param key The key.
     * @return An ARFutureWithFlag that represents the eventual completion of the value retrieval.
     */
    @NotNull
    ARFutureWithFlag<V> getFuture(@NotNull K key);

    /**
     * Returns a set of keys that currently require fetching from the external service.
     * This method is designed for systems needing exclusive control over batching and clears
     * the internal pool of newly requested keys upon return.
     *
     * @return A Set of keys that were queued for network submission.
     */
    @NotNull
    Set<K> getPendingRequests();

    /**
     * Puts a fully resolved value into the map, immediately fulfilling any waiting future.
     *
     * @param key   The key for the resolved value.
     * @param value The resolved value.
     */
    void putResolved(@NotNull K key, @NotNull V value);

    /**
     * Marks the future for a key as completed with an error.
     *
     * @param key   The key associated with the error.
     * @param error The error to complete the future with.
     */
    void putError(@NotNull K key, @NotNull Throwable error);

    /**
     * Returns an EventConsumer that fires when a value in the map is definitively resolved.
     *
     * @return The EventConsumer for resolved value updates (K, V).
     */
    EventConsumer<RMap.Update<K, V>> forValueUpdate();

    /**
     * Returns an array of keys that are queued for network submission by this specific sender.
     * This array includes keys that are newly available and keys for which this sender's
     * previous request has timed out.
     * <p>
     * The implementing class is responsible for transferring new requests and updating
     * the sender's flush time during this call.
     *
     * @param elementType The Class of the key type for array creation (often K.class).
     * @param sender      The object responsible for sending the network request (e.g., ConnectionWork).
     * @return An array of keys (K[]) that are queued for network submission by this sender.
     */
    K[] getRequestsFor(Class<K> elementType, Object sender);

    /**
     * Checks if there is any work available globally (e.g., items in the new request pool).
     * This method is a fast check intended to avoid unnecessary work.
     *
     * @return True if work might be available globally, false otherwise.
     */
    boolean isRequests();

    /**
     * Checks if there are any pending requests (newly available or timed-out) for the specified sender.
     * This method is a fast check intended to avoid unnecessary calls to {@code getRequestsFor}.
     *
     * @param sender The sender object.
     * @return True if work might be available for this sender, false otherwise.
     */
    boolean isRequestsFor(@NotNull Object sender);
}