package io.aether.utils.rcollections;

import io.aether.utils.AString;
import io.aether.utils.RU;
import io.aether.utils.ToString;
import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.slots.EventConsumer;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Iterator;
import java.util.Objects;

public interface RCollection<T> extends Collection<T>, ToString {
    EventConsumer<T> forAdd();

    EventConsumer<T> forRemove();

    @Override
    default boolean retainAll(@NotNull Collection<?> c) {
        Objects.requireNonNull(c);
        boolean modified = false;
        Iterator<T> it = iterator();
        while (it.hasNext()) {
            if (!c.contains(it.next())) {
                it.remove();
                modified = true;
            }
        }
        return modified;
    }

    @Override
    default boolean containsAll(@NotNull Collection<?> c) {
        for (var e : c) {
            if (!contains(e)) return false;
        }
        return true;
    }

    @Override
    default boolean addAll(@NotNull Collection<? extends T> c) {
        boolean res = false;
        for (var v : c) {
            res |= add(v);
        }
        return res;
    }

    @Override
    default boolean removeAll(@NotNull Collection<?> c) {
        boolean res = false;
        for (var v : c) {
            res |= remove(v);
        }
        return res;
    }

    @Override
    default void clear() {
        var it = iterator();
        while (it.hasNext()) {
            it.remove();
        }
    }

    default void add0(T val) {
        forAdd().fire(val);
    }

    default void remove0(T val) {
        forRemove().fire(val);
    }

    default @NotNull <T2> RCollection<T2> map(AFunction<T, T2> f,AFunction<T2, T> f2) {
        var self=this;
        return new RCollection<>() {
            final EventConsumer<T2> forAdd = new EventConsumer<>();
            final EventConsumer<T2> forRemove = new EventConsumer<>();

            @Override
            public void toString(AString sb) {
                boolean first=true;
                for(var e:this){
                    if(first){
                        first=false;
                    }else{
                        sb.add(", ");
                    }
                    sb.add(e);
                }
            }

            @Override
            public String toString() {
                return toString2();
            }

            @Override
            public EventConsumer<T2> forAdd() {
                return forAdd;
            }

            @Override
            public EventConsumer<T2> forRemove() {
                return forRemove;
            }

            @Override
            public int size() {
                return self.size();
            }

            @Override
            public boolean isEmpty() {
                return self.isEmpty();
            }

            @Override
            public boolean contains(Object o) {
                return self.contains(f2.apply(RU.cast(o)));
            }

            @Override
            public @NotNull Iterator<T2> iterator() {
                var it = self.iterator();
                return new Iterator<T2>() {
                    @Override
                    public boolean hasNext() {
                        return it.hasNext();
                    }

                    @Override
                    public T2 next() {
                        return f.apply(it.next());
                    }
                };
            }

            @Override
            public @NotNull Object @NotNull [] toArray() {
                return Flow.flow(this).toArray();
            }

            @Override
            public @NotNull <T5> T5 @NotNull [] toArray(@NotNull T5 @NotNull [] a) {
                return Flow.flow(this).toList().toArray(a);
            }

            @Override
            public boolean add(T2 t2) {
                return self.add(f2.apply(t2));
            }

            @Override
            public boolean remove(Object o) {
                return self.remove(f2.apply(RU.cast(o)));
            }
        };
    }

    static <E> RCollection<E> of(Collection<E> src) {
        if (src instanceof RCollection) return RU.cast(src);
        return new RCollectionBySrc<>(src);
    }

}
