package io.aether.utils.rcollections;

import io.aether.utils.RU;
import io.aether.utils.futures.ARFuture;
import io.aether.utils.futures.ARFutureWithFlag;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.slots.EventConsumer;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public interface RFMap<K, V> extends RMap<K, ARFutureWithFlag<V>> {

    default <K2> RFMap<K2, V> mapKeyFuture(AFunction<V, K2> vToK2, AFunction<V, K> vToK) {
        var self = this;
        return new RFMap<>() {
            final EventConsumer<Update<K2, ARFutureWithFlag<V>>> forUpdate = new EventConsumer<>();
            final EventConsumer<Entry<K2, ARFutureWithFlag<V>>> forRemove = new EventConsumer<>();
            final Map<K2, K> keyMapping = new ConcurrentHashMap<>();

            {
                // Обработка обновлений из исходной карты
                self.forUpdate().add(u -> {
                    if (u.newValue != null) {
                        u.newValue.to(v -> {
                            K2 newKey = vToK2.apply(v);
                            K oldOriginalKey = keyMapping.put(newKey, u.key);

                            if (oldOriginalKey != null && !oldOriginalKey.equals(u.key)) {
                                // Удаляем старую запись, если ключ изменился
                                self.remove(oldOriginalKey);
                            }

                            put(newKey, ARFuture.of(v).toWithFlag());
                            forUpdate.fire(new Update<>(newKey, ARFuture.of(v).toWithFlag(), null));
                        });
                    } else {
                        // Удаление значения
                        for (var entry : keyMapping.entrySet()) {
                            if (entry.getValue().equals(u.key)) {
                                K2 keyToRemove = entry.getKey();
                                keyMapping.remove(keyToRemove);
                                remove(keyToRemove);
                                forUpdate.fire(new Update<>(keyToRemove, null, u.oldValue));
                                break;
                            }
                        }
                    }
                });

                // Обработка удалений из исходной карты
                self.forRemove().add(e -> {
                    ARFutureWithFlag<V> future = e.getValue();
                    if (future != null) {
                        future.to(v -> {
                            for (var entry : keyMapping.entrySet()) {
                                if (entry.getValue().equals(e.getKey())) {
                                    K2 keyToRemove = entry.getKey();
                                    keyMapping.remove(keyToRemove);
                                    remove(keyToRemove);
                                    forRemove.fire(new Entry<>() {
                                        @Override
                                        public K2 getKey() {
                                            return keyToRemove;
                                        }

                                        @Override
                                        public ARFutureWithFlag<V> getValue() {
                                            return ARFuture.of(v).toWithFlag();
                                        }

                                        @Override
                                        public ARFutureWithFlag<V> setValue(ARFutureWithFlag<V> value) {
                                            throw new UnsupportedOperationException();
                                        }
                                    });
                                    break;
                                }
                            }
                        });
                    }
                });

                // Инициализация существующих значений
                for (var entry : self.entrySet()) {
                    ARFutureWithFlag<V> future = entry.getValue();
                    if (future != null && future.isDone()) {
                        V value = future.get();
                        K2 newKey = vToK2.apply(value);
                        keyMapping.put(newKey, entry.getKey());
                    }
                }
            }

            @Override
            public EventConsumer<Update<K2, ARFutureWithFlag<V>>> forUpdate() {
                return forUpdate;
            }

            @Override
            public EventConsumer<Entry<K2, ARFutureWithFlag<V>>> forRemove() {
                return forRemove;
            }

            @Override
            public @NotNull RSet<K2> keySet() {
                return new RSet<>() {
                    @Override
                    public EventConsumer<K2> forAdd() {
                        return new EventConsumer<>();
                    }

                    @Override
                    public EventConsumer<K2> forRemove() {
                        return new EventConsumer<>();
                    }

                    @Override
                    public int size() {
                        return keyMapping.size();
                    }

                    @Override
                    public boolean isEmpty() {
                        return keyMapping.isEmpty();
                    }

                    @Override
                    public boolean contains(Object o) {
                        return keyMapping.containsKey(o);
                    }

                    @Override
                    public @NotNull Iterator<K2> iterator() {
                        return keyMapping.keySet().iterator();
                    }

                    @Override
                    public @NotNull Object @NotNull [] toArray() {
                        return keyMapping.keySet().toArray();
                    }

                    @Override
                    public @NotNull <T> T @NotNull [] toArray(@NotNull T @NotNull [] a) {
                        return keyMapping.keySet().toArray(a);
                    }

                    @Override
                    public boolean add(K2 k2) {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public boolean remove(Object o) {
                        K2 key = RU.cast(o);
                        K originalKey = keyMapping.get(key);
                        if (originalKey != null) {
                            self.remove(originalKey);
                            return true;
                        }
                        return false;
                    }
                };
            }

            @Override
            public @NotNull RCollection<ARFutureWithFlag<V>> values() {
                return new RCollection<>() {
                    @Override
                    public EventConsumer<ARFutureWithFlag<V>> forAdd() {
                        return new EventConsumer<>();
                    }

                    @Override
                    public EventConsumer<ARFutureWithFlag<V>> forRemove() {
                        return new EventConsumer<>();
                    }

                    @Override
                    public int size() {
                        return keyMapping.size();
                    }

                    @Override
                    public boolean isEmpty() {
                        return keyMapping.isEmpty();
                    }

                    @Override
                    public boolean contains(Object o) {
                        if (o instanceof ARFuture) {
                            ARFuture<?> future = (ARFuture<?>) o;
                            if (future.isDone()) {
                                Object value = future.get();
                                for (K2 key : keyMapping.keySet()) {
                                    ARFutureWithFlag<V> ourFuture = get(key);
                                    if (ourFuture != null && ourFuture.isDone() && ourFuture.get().equals(value)) {
                                        return true;
                                    }
                                }
                            }
                        }
                        return false;
                    }

                    @Override
                    public @NotNull Iterator<ARFutureWithFlag<V>> iterator() {
                        return new Iterator<>() {
                            private final Iterator<K2> keyIterator = keyMapping.keySet().iterator();

                            @Override
                            public boolean hasNext() {
                                return keyIterator.hasNext();
                            }

                            @Override
                            public ARFutureWithFlag<V> next() {
                                return get(keyIterator.next());
                            }
                        };
                    }

                    @Override
                    public @NotNull Object @NotNull [] toArray() {
                        Object[] result = new Object[keyMapping.size()];
                        int i = 0;
                        for (K2 key : keyMapping.keySet()) {
                            result[i++] = get(key);
                        }
                        return result;
                    }

                    @Override
                    public @NotNull <T> T @NotNull [] toArray(@NotNull T @NotNull [] a) {
                        // Упрощенная реализация
                        return RU.cast(toArray());
                    }

                    @Override
                    public boolean add(ARFutureWithFlag<V> vFuture) {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public boolean remove(Object o) {
                        if (o instanceof ARFuture) {
                            ARFuture<?> future = (ARFuture<?>) o;
                            if (future.isDone()) {
                                Object value = future.get();
                                for (var entry : keyMapping.entrySet()) {
                                    ARFutureWithFlag<V> ourFuture = get(entry.getKey());
                                    if (ourFuture != null && ourFuture.isDone() && ourFuture.get().equals(value)) {
                                        self.remove(entry.getValue());
                                        return true;
                                    }
                                }
                            }
                        }
                        return false;
                    }
                };
            }

            @Override
            public @NotNull RSet<Entry<K2, ARFutureWithFlag<V>>> entrySet() {
                return new RSet<>() {
                    @Override
                    public EventConsumer<Entry<K2, ARFutureWithFlag<V>>> forAdd() {
                        return new EventConsumer<>();
                    }

                    @Override
                    public EventConsumer<Entry<K2, ARFutureWithFlag<V>>> forRemove() {
                        return new EventConsumer<>();
                    }

                    @Override
                    public int size() {
                        return keyMapping.size();
                    }

                    @Override
                    public boolean isEmpty() {
                        return keyMapping.isEmpty();
                    }

                    @Override
                    public boolean contains(Object o) {
                        if (o instanceof Map.Entry) {
                            Entry<?, ?> entry = (Entry<?, ?>) o;
                            ARFutureWithFlag<V> future = get(entry.getKey());
                            return future != null && future.equals(entry.getValue());
                        }
                        return false;
                    }

                    @Override
                    public @NotNull Iterator<Entry<K2, ARFutureWithFlag<V>>> iterator() {
                        return new Iterator<>() {
                            private final Iterator<K2> keyIterator = keyMapping.keySet().iterator();

                            @Override
                            public boolean hasNext() {
                                return keyIterator.hasNext();
                            }

                            @Override
                            public Entry<K2, ARFutureWithFlag<V>> next() {
                                K2 key = keyIterator.next();
                                return new Entry<>() {
                                    @Override
                                    public K2 getKey() {
                                        return key;
                                    }

                                    @Override
                                    public ARFutureWithFlag<V> getValue() {
                                        return get(key);
                                    }

                                    @Override
                                    public ARFutureWithFlag<V> setValue(ARFutureWithFlag<V> value) {
                                        ARFutureWithFlag<V> oldValue = get(key);
                                        put(key, value);
                                        return oldValue;
                                    }
                                };
                            }
                        };
                    }

                    @Override
                    public @NotNull Object @NotNull [] toArray() {
                        Object[] result = new Object[keyMapping.size()];
                        int i = 0;
                        for (K2 key : keyMapping.keySet()) {
                            result[i++] = new Entry<K2, ARFutureWithFlag<V>>() {
                                @Override
                                public K2 getKey() {
                                    return key;
                                }

                                @Override
                                public ARFutureWithFlag<V> getValue() {
                                    return get(key);
                                }

                                @Override
                                public ARFutureWithFlag<V> setValue(ARFutureWithFlag<V> value) {
                                    throw new UnsupportedOperationException();
                                }
                            };
                        }
                        return result;
                    }

                    @Override
                    public @NotNull <T> T @NotNull [] toArray(@NotNull T @NotNull [] a) {
                        // Упрощенная реализация
                        return RU.cast(toArray());
                    }

                    @Override
                    public boolean add(Entry<K2, ARFutureWithFlag<V>> k2ARFutureEntry) {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public boolean remove(Object o) {
                        if (o instanceof Map.Entry) {
                            Entry<?, ?> entry = (Entry<?, ?>) o;
                            K2 key = RU.cast(entry.getKey());
                            K originalKey = keyMapping.get(key);
                            if (originalKey != null) {
                                self.remove(originalKey);
                                return true;
                            }
                        }
                        return false;
                    }
                };
            }

            @Override
            public int size() {
                return keyMapping.size();
            }

            @Override
            public boolean isEmpty() {
                return keyMapping.isEmpty();
            }

            @Override
            public boolean containsKey(Object key) {
                return keyMapping.containsKey(key);
            }

            @Override
            public boolean containsValue(Object value) {
                if (value instanceof ARFuture) {
                    ARFuture<?> future = (ARFuture<?>) value;
                    if (future.isDone()) {
                        Object futureValue = future.get();
                        for (K2 mappedKey : keyMapping.keySet()) {
                            ARFutureWithFlag<V> ourFuture = get(mappedKey);
                            if (ourFuture != null && ourFuture.isDone() && ourFuture.get().equals(futureValue)) {
                                return true;
                            }
                        }
                    }
                }
                return false;
            }

            @Override
            public ARFutureWithFlag<V> get(Object key) {
                K originalKey = keyMapping.get(key);
                return originalKey != null ? self.get(originalKey) : null;
            }

            @Override
            public @Nullable ARFutureWithFlag<V> put(K2 key, ARFutureWithFlag<V> value) {
                throw new UnsupportedOperationException("Cannot put directly to mapped RFMap");
            }

            @Override
            public ARFutureWithFlag<V> remove(Object key) {
                K originalKey = keyMapping.get(key);
                if (originalKey != null) {
                    ARFutureWithFlag<V> removed = self.remove(originalKey);
                    keyMapping.remove(key);
                    return removed;
                }
                return null;
            }

            @Override
            public void putAll(@NotNull Map<? extends K2, ? extends ARFutureWithFlag<V>> m) {
                throw new UnsupportedOperationException("Cannot putAll directly to mapped RFMap");
            }

            @Override
            public void clear() {
                for (K originalKey : keyMapping.values()) {
                    self.remove(originalKey);
                }
                keyMapping.clear();
            }
        };
    }

    default <V2> RFMap<K, V2> mapValFuture(AFunction<V, V2> vToV2, AFunction<V2, V> v2ToV) {
        var self = this;
        return new RFMap<>() {
            final EventConsumer<Update<K, ARFutureWithFlag<V2>>> forUpdate = new EventConsumer<>();
            final EventConsumer<Entry<K, ARFutureWithFlag<V2>>> forRemove = new EventConsumer<>();

            {
                // Обработка обновлений из исходной карты
                self.forUpdate().add(u -> {
                    ARFutureWithFlag<V> future = u.newValue;
                    if (future != null) {
                        var mappedFuture = future.map(vToV2).toWithFlag();
                        forUpdate.fire(new Update<>(u.key, mappedFuture,
                                u.oldValue != null ? u.oldValue.map(vToV2).toWithFlag() : null));
                    } else {
                        forUpdate.fire(new Update<>(u.key, null,
                                u.oldValue != null ? u.oldValue.map(vToV2).toWithFlag() : null));
                    }
                });

                // Обработка удалений из исходной карты
                self.forRemove().add(e -> {
                    forRemove.fire(new Entry<>() {
                        @Override
                        public K getKey() {
                            return e.getKey();
                        }

                        @Override
                        public ARFutureWithFlag<V2> getValue() {
                            return e.getValue().map(vToV2).toWithFlag();
                        }

                        @Override
                        public ARFutureWithFlag<V2> setValue(ARFutureWithFlag<V2> value) {
                            throw new UnsupportedOperationException();
                        }
                    });
                });
            }

            @Override
            public EventConsumer<Update<K, ARFutureWithFlag<V2>>> forUpdate() {
                return forUpdate;
            }

            @Override
            public EventConsumer<Entry<K, ARFutureWithFlag<V2>>> forRemove() {
                return forRemove;
            }

            @Override
            public @NotNull RSet<K> keySet() {
                return self.keySet();
            }

            @Override
            public @NotNull RCollection<ARFutureWithFlag<V2>> values() {
                return new RCollection<>() {
                    @Override
                    public EventConsumer<ARFutureWithFlag<V2>> forAdd() {
                        return new EventConsumer<>();
                    }

                    @Override
                    public EventConsumer<ARFutureWithFlag<V2>> forRemove() {
                        return new EventConsumer<>();
                    }

                    @Override
                    public int size() {
                        return self.size();
                    }

                    @Override
                    public boolean isEmpty() {
                        return self.isEmpty();
                    }

                    @Override
                    public boolean contains(Object o) {
                        if (o instanceof ARFuture) {
                            ARFuture<?> future = (ARFuture<?>) o;
                            if (future.isDone()) {
                                Object value = future.get();
                                for (ARFutureWithFlag<V> originalFuture : self.values()) {
                                    if (originalFuture.isDone()) {
                                        V2 mappedValue = vToV2.apply(originalFuture.get());
                                        if (mappedValue.equals(value)) {
                                            return true;
                                        }
                                    }
                                }
                            }
                        }
                        return false;
                    }

                    @Override
                    public @NotNull Iterator<ARFutureWithFlag<V2>> iterator() {
                        return new Iterator<>() {
                            private final Iterator<ARFutureWithFlag<V>> originalIterator = self.values().iterator();

                            @Override
                            public boolean hasNext() {
                                return originalIterator.hasNext();
                            }

                            @Override
                            public ARFutureWithFlag<V2> next() {
                                return originalIterator.next().map(vToV2).toWithFlag();
                            }
                        };
                    }

                    @Override
                    public @NotNull Object @NotNull [] toArray() {
                        Object[] result = new Object[self.size()];
                        int i = 0;
                        for (ARFutureWithFlag<V> future : self.values()) {
                            result[i++] = future.map(vToV2);
                        }
                        return result;
                    }

                    @Override
                    public @NotNull <T> T @NotNull [] toArray(@NotNull T @NotNull [] a) {
                        // Упрощенная реализация
                        return RU.cast( toArray());
                    }

                    @Override
                    public boolean add(ARFutureWithFlag<V2> v2ARFuture) {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public boolean remove(Object o) {
                        if (o instanceof ARFuture) {
                            ARFuture<?> future = (ARFuture<?>) o;
                            if (future.isDone()) {
                                Object value = future.get();
                                for (var entry : self.entrySet()) {
                                    ARFutureWithFlag<V> originalFuture = entry.getValue();
                                    if (originalFuture.isDone()) {
                                        V2 mappedValue = vToV2.apply(originalFuture.get());
                                        if (mappedValue.equals(value)) {
                                            self.remove(entry.getKey());
                                            return true;
                                        }
                                    }
                                }
                            }
                        }
                        return false;
                    }
                };
            }

            @Override
            public @NotNull RSet<Entry<K, ARFutureWithFlag<V2>>> entrySet() {
                return new RSet<>() {
                    @Override
                    public EventConsumer<Entry<K, ARFutureWithFlag<V2>>> forAdd() {
                        return new EventConsumer<>();
                    }

                    @Override
                    public EventConsumer<Entry<K, ARFutureWithFlag<V2>>> forRemove() {
                        return new EventConsumer<>();
                    }

                    @Override
                    public int size() {
                        return self.size();
                    }

                    @Override
                    public boolean isEmpty() {
                        return self.isEmpty();
                    }

                    @Override
                    public boolean contains(Object o) {
                        if (o instanceof Map.Entry) {
                            Entry<?, ?> entry = (Entry<?, ?>) o;
                            ARFutureWithFlag<V2> ourFuture = get(entry.getKey());
                            return ourFuture != null && ourFuture.equals(entry.getValue());
                        }
                        return false;
                    }

                    @Override
                    public @NotNull Iterator<Entry<K, ARFutureWithFlag<V2>>> iterator() {
                        return new Iterator<>() {
                            private final Iterator<Entry<K, ARFutureWithFlag<V>>> originalIterator = self.entrySet().iterator();

                            @Override
                            public boolean hasNext() {
                                return originalIterator.hasNext();
                            }

                            @Override
                            public Entry<K, ARFutureWithFlag<V2>> next() {
                                Entry<K, ARFutureWithFlag<V>> originalEntry = originalIterator.next();
                                return new Entry<>() {
                                    @Override
                                    public K getKey() {
                                        return originalEntry.getKey();
                                    }

                                    @Override
                                    public ARFutureWithFlag<V2> getValue() {
                                        return originalEntry.getValue().map(vToV2).toWithFlag();
                                    }

                                    @Override
                                    public ARFutureWithFlag<V2> setValue(ARFutureWithFlag<V2> value) {
                                        ARFutureWithFlag<V> oldValue = originalEntry.getValue();
                                        originalEntry.setValue(value.map(v2ToV).toWithFlag());
                                        return oldValue.map(vToV2).toWithFlag();
                                    }
                                };
                            }
                        };
                    }

                    @Override
                    public @NotNull Object @NotNull [] toArray() {
                        Object[] result = new Object[self.size()];
                        int i = 0;
                        for (Entry<K, ARFutureWithFlag<V>> entry : self.entrySet()) {
                            result[i++] = new Entry<K, ARFutureWithFlag<V2>>() {
                                @Override
                                public K getKey() {
                                    return entry.getKey();
                                }

                                @Override
                                public ARFutureWithFlag<V2> getValue() {
                                    return entry.getValue().map(vToV2).toWithFlag();
                                }

                                @Override
                                public ARFutureWithFlag<V2> setValue(ARFutureWithFlag<V2> value) {
                                    throw new UnsupportedOperationException();
                                }
                            };
                        }
                        return result;
                    }

                    @Override
                    public @NotNull <T> T @NotNull [] toArray(@NotNull T @NotNull [] a) {
                        // Упрощенная реализация
                        return RU.cast(toArray());
                    }

                    @Override
                    public boolean add(Entry<K, ARFutureWithFlag<V2>> kARFutureEntry) {
                        throw new UnsupportedOperationException();
                    }

                    @Override
                    public boolean remove(Object o) {
                        if (o instanceof Map.Entry) {
                            Entry<?, ?> entry = (Entry<?, ?>) o;
                            self.remove(entry.getKey());
                            return true;
                        }
                        return false;
                    }
                };
            }

            @Override
            public int size() {
                return self.size();
            }

            @Override
            public boolean isEmpty() {
                return self.isEmpty();
            }

            @Override
            public boolean containsKey(Object key) {
                return self.containsKey(key);
            }

            @Override
            public boolean containsValue(Object value) {
                if (value instanceof ARFuture) {
                    ARFuture<?> future = (ARFuture<?>) value;
                    if (future.isDone()) {
                        Object futureValue = future.get();
                        for (ARFutureWithFlag<V> originalFuture : self.values()) {
                            if (originalFuture.isDone()) {
                                V2 mappedValue = vToV2.apply(originalFuture.get());
                                if (mappedValue.equals(futureValue)) {
                                    return true;
                                }
                            }
                        }
                    }
                }
                return false;
            }

            @Override
            public ARFutureWithFlag<V2> get(Object key) {
                ARFutureWithFlag<V> originalFuture = self.get(key);
                return originalFuture != null ? originalFuture.map(vToV2).toWithFlag() : null;
            }

            @Override
            public @Nullable ARFutureWithFlag<V2> put(K key, ARFutureWithFlag<V2> value) {
                ARFutureWithFlag<V> originalFuture = self.get(key);
                ARFutureWithFlag<V> newFuture = value.map(v2ToV).toWithFlag();
                self.put(key, newFuture);
                return originalFuture != null ? originalFuture.map(vToV2).toWithFlag() : null;
            }

            @Override
            public ARFutureWithFlag<V2> remove(Object key) {
                ARFutureWithFlag<V> removed = self.remove(key);
                return removed != null ? removed.map(vToV2).toWithFlag() : null;
            }

            @Override
            public void putAll(@NotNull Map<? extends K, ? extends ARFutureWithFlag<V2>> m) {
                for (var entry : m.entrySet()) {
                    put(entry.getKey(), entry.getValue());
                }
            }

            @Override
            public void clear() {
                self.clear();
            }
        };
    }

    default <K2, V2> RFMap<K2, V2> mapFuture(AFunction<K, K2> kToK2, AFunction<K2, K> k2ToK,
                                             AFunction<V, V2> vToV2, AFunction<V2, V> v2ToV) {
        // Комбинация mapKeyFuture и mapValFuture
        return mapKeyFuture(
                v -> kToK2.apply(null), // Заглушка, т.к. ключи преобразуются из исходных ключей
                v -> null               // Заглушка
        ).mapValFuture(vToV2, v2ToV);
    }
}