package io.aether.utils.rcollections;

import io.aether.utils.AString;
import io.aether.utils.RU;
import io.aether.utils.flow.Flow;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.slots.EventConsumer;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Iterator;
import java.util.Set;

public interface RSet<T> extends RCollection<T>, Set<T> {
    @Override
    default boolean retainAll(@NotNull Collection<?> c) {
        return RCollection.super.retainAll(c);
    }
    default boolean addAndRemove(Set<T> s){
        var res=addAll(s);
        var t=iterator();
        while (t.hasNext()){
            if(!s.contains(t.next())){
                t.remove();
                res=true;
            }
        }
        return res;
    }
    @Override
    default void clear() {
        RCollection.super.clear();
    }

    @Override
    default boolean addAll(@NotNull Collection<? extends T> c) {
        return RCollection.super.addAll(c);
    }

    @Override
    default boolean containsAll(@NotNull Collection<?> c) {
        return RCollection.super.containsAll(c);
    }

    @Override
    default boolean removeAll(@NotNull Collection<?> c) {
        return RCollection.super.removeAll(c);
    }

    default <K> RMap<K, T> toMap(AFunction<T, K> keyGetter) {
        RMap<K, T> res = RCol.map();
        forAdd().add(v -> res.put(keyGetter.apply(v), v));
        forRemove().add(v -> res.remove(keyGetter.apply(v)));
        res.forUpdate().add(e -> {
            if (e.newValue != null) this.add(e.newValue);
            if (e.oldValue != null) this.remove(e.oldValue);
        });
        res.forRemove().add(e -> {
            if (e.getValue() != null) {
                this.add(e.getValue());
            }
        });
        return res;
    }

    default <T2> void link(RSet<T2> other, AFunction<T, T2> f, AFunction<T2, T> back) {
        forAdd().add(v -> other.add(f.apply(v)));
        forRemove().add(v -> other.remove(f.apply(v)));
        other.forAdd().add(v -> add(back.apply(v)));
        other.forRemove().add(v -> remove(back.apply(v)));
        for (var e : other) {
            add(back.apply(e));
        }
        for (var e : this) {
            other.add(f.apply(e));
        }
    }
    default  void link(RSet<T> other) {
        forAdd().add(other::add);
        forRemove().add(other::remove);
        other.forAdd().add(this::add);
        other.forRemove().add(this::remove);
        this.addAll(other);
        other.addAll(this);
    }

    default <T2> @NotNull RSet<T2> map(AFunction<T, T2> f, AFunction<T2, T> back) {
        var self = this;
        return new RSet<>() {
            final EventConsumer<T2> forAdd = new EventConsumer<>();
            final EventConsumer<T2> forRemove = new EventConsumer<>();

            @Override
            public void toString(AString sb) {
                boolean first=true;
                for(var e:this){
                    if(first){
                        first=false;
                    }else{
                        sb.add(", ");
                    }
                    sb.add(e);
                }
            }

            @Override
            public String toString() {
                return toString2();
            }

            @Override
            public EventConsumer<T2> forAdd() {
                return forAdd;
            }

            @Override
            public EventConsumer<T2> forRemove() {
                return forRemove;
            }

            @Override
            public int size() {
                return self.size();
            }

            @Override
            public boolean isEmpty() {
                return self.isEmpty();
            }

            @Override
            public boolean contains(Object o) {
                return self.contains(back.apply(RU.cast(o)));
            }

            @Override
            public @NotNull Iterator<T2> iterator() {
                var it = self.iterator();
                return new Iterator<T2>() {
                    @Override
                    public boolean hasNext() {
                        return it.hasNext();
                    }

                    @Override
                    public T2 next() {
                        return f.apply(it.next());
                    }
                };
            }

            @Override
            public @NotNull Object @NotNull [] toArray() {
                return Flow.flow(self).map(f).toArray();
            }

            @Override
            public @NotNull <T5> T5 @NotNull [] toArray(@NotNull T5 @NotNull [] a) {
                return Flow.flow(self).map(f).toList().toArray(a);
            }

            @Override
            public boolean add(T2 t2) {
                return self.add(back.apply(t2));
            }

            @Override
            public boolean remove(Object o) {
                return self.remove(back.apply(RU.cast(o)));
            }

        };
    }

    default <T2>RSet<T2> map(AFunction<T,T2> f){
        RSet<T2> res= RCol.set();
        forAdd().add(v->{
            res.add(f.apply(v));
        });
        forRemove().add(v->{
            res.remove(f.apply(v));
        });
        return res;
    }

    static <E> RSet<E> of(Set<E> src) {
        if (src instanceof RSet) return RU.cast(src);
        return new RSetBySrc<>(src);
    }
}

