package io.aether.utils.slots;

import io.aether.utils.CTypeI;
import io.aether.utils.RU;
import io.aether.utils.TaskConsumer;
import io.aether.utils.futures.ARFuture;
import io.aether.utils.interfaces.AConsumer;
import io.aether.utils.interfaces.AFunction;
import io.aether.utils.interfaces.ARunnable;

import java.lang.invoke.VarHandle;

public class AMFuture<T> implements AConsumer<T> {
    protected final static VarHandle UPDATER = CTypeI.of(AMFuture.class).getFieldVarHandle("value");
    public transient final EventConsumer<T> eventConsumer = new EventConsumer<>();
    protected volatile T value;

    public AMFuture(T value) {
        this.value = value;
    }

    public AMFuture() {
    }

    public void refresh() {
        var v = value;
        if (v == null) return;
        eventConsumer.fire(v);
    }

    @Override
    public void accept2(T t) {
        set(t);
    }

    public boolean set(T value) {
        var vv = this.value;
        if (vv == null) {
            if (UPDATER.compareAndSet(this, null, value)) {
                eventConsumer.fire(value);
                return true;
            } else {
                return set(value);
            }
        } else if (vv == value) {
            return false;
        }
        this.value = value;
        eventConsumer.fire(value);
        return true;
    }

    public <V2> AMFuture<V2> map(AFunction<T, V2> f) {
        var res = new AMFuture<V2>();
        add(v -> res.set(f.apply(v)));
        var v = value;
        if (v != null) {
            res.set(f.apply(v));
        }
        return res;
    }

    public void once(AConsumer<T> task, int seconds, ARunnable timeoutTask) {
        once(task, seconds * 1000L, timeoutTask);
    }

    public void once(AConsumer<T> task, long ms, ARunnable timeoutTask) {
        var onceTask = new TaskConsumer<T>() {
            @Override
            protected void runTask(T value) {
                task.accept(value);
            }
        };
        RU.schedule(ms, () -> {
            if (onceTask.isExecuted()) return;
            timeoutTask.run();
        });
        once(onceTask);
    }

    public void once(TaskConsumer<T> t2) {
        eventConsumer.once(t2);
        var v = value;
        if (v != null) {
            try {
                t2.accept(v);
            } catch (RuntimeException e) {
                throw e;
            } catch (Throwable e) {
                RU.error(e);
            }
        }
    }

    public void toOnce(AConsumer<T> task) {
        once(new TaskConsumer<T>() {
            @Override
            protected void runTask(T value) {
                task.accept(value);
            }
        });
    }

    public void once(AConsumer<T> task) {
        if (task == this) return;
        var t2 = new TaskConsumer<T>() {
            @Override
            protected void runTask(T value) {
                task.accept(value);
            }
        };
        once(t2);
    }

    public void addWeak(AConsumer<T> task) {
        add(task.weak());
    }

    public void add(AConsumer<T> task) {
        eventConsumer.add(task);
        var v = value;
        if (v != null) {
            task.accept(v);
        }
    }

    public T getNow() {
        return value;
    }

    public ARFuture<T> mapToARFuture() {
        var v = value;
        if (v != null) return ARFuture.of(v);
        ARFuture<T> res = ARFuture.make();
        once(res::tryDone);
        return res;
    }

    public boolean isDone() {
        return value != null;
    }

    public static <T> AMFuture<T> completed(T value) {
        var res = new AMFuture<T>();
        res.set(value);
        return res;
    }
}
